package ru.yandex.qe.dispenser.domain.dao.notifications;

import java.util.Collection;

import javax.mail.MessagingException;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.MailParseException;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.mail.javamail.MimeMessageHelper;

public class EmailSender {
    private static final Logger LOG = LoggerFactory.getLogger(EmailSender.class);

    private JavaMailSenderImpl mailSender;

    private String from;

    @Nullable
    private String stubMailTo;

    public void sendMessage(@NotNull final String subject, @NotNull final String text, @NotNull final Collection<String> recipients) {
        sendMessage(subject, text, recipients, false);
    }

    public void sendMimeMessage(@NotNull final String subject, @NotNull final String text, @NotNull final Collection<String> recipients) {
        sendMessage(subject, text, recipients, true);
    }

    protected void sendMessage(@NotNull final String subject, @NotNull final String text, @NotNull final Collection<String> recipients, final boolean isHtml) {
        LOG.info("Sending from {} msg ('{}') to {} with subj {}", from, text, recipients, subject);
        final MimeMessageHelper msg = new MimeMessageHelper(mailSender.createMimeMessage());
        try {
            msg.setFrom(from);
            final String[] to = recipients.toArray(ArrayUtils.EMPTY_STRING_ARRAY);
            if (StringUtils.isNotEmpty(stubMailTo)) {
                msg.setTo(stubMailTo);
                final String prefix = "Origin-to: " + StringUtils.join(to, ", ") + (isHtml ? "<br />" : "\n");
                msg.setText(prefix + text, isHtml);
            } else {
                msg.setTo(to);
                msg.setText(text, isHtml);
            }
            msg.setSubject(subject);
        } catch (MessagingException ex) {
            throw new MailParseException(ex);
        }
        mailSender.send(msg.getMimeMessage());
    }

    public void setMailSender(final JavaMailSenderImpl mailSender) {
        this.mailSender = mailSender;
    }

    public void setStubMailTo(final String stubMailTo) {
        this.stubMailTo = stubMailTo;
    }

    public void setFrom(final String from) {
        this.from = from;
    }
}
