package ru.yandex.qe.dispenser.domain.dao.notifications;

import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.notifications.NotificationEntry;

public class SqlNotificationDao extends SqlDaoBase implements NotificationsDao {
    private static final Logger LOG = LoggerFactory.getLogger(SqlNotificationDao.class);

    private static final String GET_ALL_QUERY = "SELECT * FROM notification";
    private static final String INSERT_QUERY = "INSERT INTO notification (project_id, quota_spec_id, quota_limit, creation_time) values(:project_id, :spec_id, :quota_limit, now())";
    private static final String REMOVE_QUERY = "DELETE FROM notification WHERE project_id = :project_id and quota_spec_id = :spec_id and quota_limit = :quota_limit";
    private static final String CLEAR_QUERY = "TRUNCATE notification CASCADE";

    @Autowired
    private HierarchySupplier hierarchySupplier;

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public Set<NotificationEntry> getActualNotifications() {
        final Hierarchy hierarchy = hierarchySupplier.get();
        return jdbcTemplate.queryForSet(GET_ALL_QUERY, (rs, i) -> {
            return new NotificationEntry(
                    hierarchy.getProjectReader().read(rs.getLong("project_id")),
                    hierarchy.getQuotaSpecReader().read(rs.getLong("quota_spec_id")),
                    rs.getInt("quota_limit"), null);
        });
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public void addNotification(@NotNull final NotificationEntry notification) {
        jdbcTemplate.update(INSERT_QUERY, toParams(notification));
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public void remove(@NotNull final NotificationEntry notification) {
        jdbcTemplate.update(REMOVE_QUERY, toParams(notification));
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public void clear() {
        jdbcTemplate.update(CLEAR_QUERY);
    }


    @NotNull
    private Map<String, Object> toParams(@NotNull final NotificationEntry notification) {
        return ImmutableMap.<String, Object>builder()
                .put("project_id", notification.getProject().getId())
                .put("spec_id", notification.getSpec().getId())
                .put("quota_limit", notification.getQuotaLimit())
                .build();
    }
}
