package ru.yandex.qe.dispenser.domain.dao.person;

import java.util.Collection;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.dao.ReadOnlyDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Role;

public interface PersonReader extends ReadOnlyDao.Normalized<Person, String> {

    @NotNull
    Set<Project> getLinkedProjects(@NotNull Person member, @NotNull Role role);

    @NotNull
    Optional<Person> tryReadPersonByUid(final long uid);

    @NotNull
    Set<Person> tryReadPersonsByUids(@NotNull final Collection<Long> uids);

    @NotNull
    Optional<Person> tryReadPersonByLogin(@NotNull final String login);

    @NotNull
    Set<Person> tryReadPersonsByLogins(@NotNull final Collection<String> logins);

    @NotNull
    Optional<Person> tryReadPersonById(final long id);

    @NotNull
    Set<Person> tryReadPersonsByIds(@NotNull final Collection<Long> ids);

    @NotNull
    default Person readPersonByLogin(@NotNull final String login) {
        return tryReadPersonByLogin(login).orElseThrow(() -> new EmptyResultDataAccessException("No person with login " + login, 1));
    }

    @NotNull
    default Set<Person> readPersonsByLogins(@NotNull final Collection<String> logins) {
        final Set<Person> persons = tryReadPersonsByLogins(logins);
        final Set<String> foundLogins = persons.stream().map(Person::getLogin).collect(Collectors.toSet());
        final Set<String> missingLogins = Sets.difference(new HashSet<>(logins), foundLogins);
        if (!missingLogins.isEmpty()) {
            throw new EmptyResultDataAccessException("No persons with logins " + missingLogins, logins.size());
        }
        return persons;
    }

    @NotNull
    default Person readPersonByUid(final long uid) {
        return tryReadPersonByUid(uid).orElseThrow(() -> new EmptyResultDataAccessException("No person with uid " + uid, 1));
    }

    @NotNull
    default Set<Person> readPersonsByUids(@NotNull final Collection<Long> uids) {
        final Set<Person> persons = tryReadPersonsByUids(uids);
        final Set<Long> foundUids = persons.stream().map(Person::getUid).collect(Collectors.toSet());
        final Set<Long> missingUids = Sets.difference(new HashSet<>(uids), foundUids);
        if (!missingUids.isEmpty()) {
            throw new EmptyResultDataAccessException("No persons with uids " + missingUids, uids.size());
        }
        return persons;
    }

    @NotNull
    default Person readPersonById(final long id) {
        return tryReadPersonById(id).orElseThrow(() -> new EmptyResultDataAccessException("No person with id " + id, 1));
    }

    @NotNull
    default Set<Person> readPersonsByIds(@NotNull final Collection<Long> ids) {
        final Set<Person> persons = tryReadPersonsByIds(ids);
        final Set<Long> foundIds = persons.stream().map(Person::getId).collect(Collectors.toSet());
        final Set<Long> missingIds = Sets.difference(new HashSet<>(ids), foundIds);
        if (!missingIds.isEmpty()) {
            throw new EmptyResultDataAccessException("No persons with ids " + missingIds, ids.size());
        }
        return persons;
    }

}
