package ru.yandex.qe.dispenser.domain.dao.person;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.api.v1.DiYandexGroupType;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.PersonAffiliation;
import ru.yandex.qe.dispenser.domain.PersonGroupMembership;
import ru.yandex.qe.dispenser.domain.PersonGroupMembershipType;
import ru.yandex.qe.dispenser.domain.YaGroup;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;

public class SqlPersonGroupMembershipDao extends SqlDaoBase implements PersonGroupMembershipDao {

    private static final String GET_ALL_QUERY = "SELECT m.*, p.login, p.uid, p.is_robot, p.is_dismissed, p.is_deleted, p.affiliation, " +
            "g.url_acceptable_key, g.group_type, g.staff_id, g.deleted FROM person_group_membership m join person p on m.person_id = p.id " +
            "join yandex_group g on m.group_id = g.id ORDER BY id ASC";
    private static final String INSERT_QUERY = "INSERT INTO person_group_membership (person_id, group_id, membership_type) " +
            "VALUES (:person_id, :group_id, cast(:membership_type as person_group_membership_type)) RETURNING id";
    private static final String CREATE_IF_ABSENT_QUERY = "INSERT INTO person_group_membership (person_id, group_id, membership_type) " +
            "VALUES (:person_id, :group_id, cast(:membership_type as person_group_membership_type)) ON CONFLICT DO NOTHING";
    private static final String SELECT_BY_ID_QUERY = "SELECT m.*, p.login, p.uid, p.is_robot, p.is_dismissed, p.is_deleted, p.affiliation, " +
            "g.url_acceptable_key, g.group_type, g.staff_id, g.deleted FROM person_group_membership m join person p on m.person_id = p.id " +
            "join yandex_group g on m.group_id = g.id WHERE m.id = :id";
    private static final String SELECT_BY_PERSONS_QUERY = "SELECT m.*, p.login, p.uid, p.is_robot, p.is_dismissed, p.is_deleted, p.affiliation, " +
            "g.url_acceptable_key, g.group_type, g.staff_id, g.deleted FROM person_group_membership m join person p on m.person_id = p.id " +
            "join yandex_group g on m.group_id = g.id WHERE m.person_id IN (:person_ids) ORDER BY id ASC";
    private static final String UPDATE_QUERY = "UPDATE person_group_membership SET person_id = :person_id, group_id = :group_id, " +
            "membership_type = cast(:membership_type as person_group_membership_type) WHERE id = :id";
    private static final String DELETE_QUERY = "DELETE FROM person_group_membership WHERE id = :id";
    private static final String CLEAR_QUERY = "TRUNCATE person_group_membership CASCADE";
    private static final String SELECT_ALL_DEPARTMENT_GROUPS_BY_LOGIN = "SELECT m.group_id, g.url_acceptable_key, g.group_type, g.staff_id, g.deleted " +
            "FROM person_group_membership m join yandex_group g on m.group_id = g.id WHERE m.person_id = :person_id " +
            "AND (m.membership_type = cast(:department_membership_type as person_group_membership_type) " +
            "OR m.membership_type = cast(:ancestors_department_membership_type as person_group_membership_type))";
    private static final String PERSON_IDS_BY_GROUP_ID = "SELECT person_id FROM person_group_membership WHERE group_id = :group_id";
    private static final String PERSON_IDS_BY_GROUP_IDS = "SELECT person_id, group_id FROM person_group_membership WHERE group_id IN (:group_ids)";
    private static final String GROUPS_BY_PERSON_ID = "SELECT m.group_id, g.url_acceptable_key, g.group_type, g.staff_id, g.deleted " +
            "FROM person_group_membership m join yandex_group g on m.group_id = g.id WHERE m.person_id = :person_id";
    private static final String GROUPS_BY_PERSON_IDS = "SELECT m.group_id, g.url_acceptable_key, g.group_type, g.staff_id, g.deleted, m.person_id " +
            "FROM person_group_membership m join yandex_group g on m.group_id = g.id WHERE m.person_id IN (:person_ids)";
    private static final String PERSONS_IDS_BY_GROUP_IDS_FOR_PROJECT_MEMBERSHIP = "SELECT group_id, person_id FROM person_group_membership " +
            "WHERE group_id IN (SELECT yandex_group_id FROM yandex_group_membership)";

    @Override
    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public Set<PersonGroupMembership> getAll() {
        final Map<Long, Person> personById = new HashMap<>();
        final Map<Long, YaGroup> groupById = new HashMap<>();
        return jdbcTemplate.queryForSet(GET_ALL_QUERY, (rs, i) -> toPersonGroupMembership(rs, personById, groupById));
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public PersonGroupMembership create(@NotNull final PersonGroupMembership object) {
        final long newId = jdbcTemplate.queryForOptional(INSERT_QUERY, ImmutableMap.of("person_id", object.getPerson().getId(),
                "group_id", object.getGroup().getId(), "membership_type", object.getMembershipType().name()),
                (rs, i) -> rs.getLong("id")).get();
        object.setId(newId);
        return object;
    }

    @Override
    public void createIfAbsent(@NotNull final Collection<PersonGroupMembership> objects) {
        if (objects.isEmpty()) {
            return;
        }
        jdbcTemplate.batchUpdate(CREATE_IF_ABSENT_QUERY, objects.stream()
                .map(object -> ImmutableMap.of("person_id", object.getPerson().getId(),
                        "group_id", object.getGroup().getId(), "membership_type", object.getMembershipType().name()))
                .collect(Collectors.toList()));
    }

    @NotNull
    @Override
    public List<PersonGroupMembership> findByPersons(@NotNull final Collection<Person> persons) {
        if (persons.isEmpty()) {
            return Collections.emptyList();
        }
        final Map<Long, Person> personById = new HashMap<>();
        final Map<Long, YaGroup> groupById = new HashMap<>();
        final Set<Long> personIds = persons.stream().map(LongIndexBase::getId).collect(Collectors.toSet());
        return jdbcTemplate.query(SELECT_BY_PERSONS_QUERY, ImmutableMap.of("person_ids", personIds),
                (rs, i) -> toPersonGroupMembership(rs, personById, groupById));
    }

    @NotNull
    @Override
    public Set<YaGroup> getAllPersonDepartmentsByPerson(@NotNull final Person person) {
        return jdbcTemplate.queryForSet(SELECT_ALL_DEPARTMENT_GROUPS_BY_LOGIN, ImmutableMap.of("person_id", person.getId(),
                "department_membership_type", PersonGroupMembershipType.DEPARTMENT.name(), "ancestors_department_membership_type",
                PersonGroupMembershipType.DEPARTMENT_ANCESTORS.name()), (rs, i) -> {
            final long groupId = rs.getLong("group_id");
            return toGroup(rs, groupId);
        });
    }

    @NotNull
    @Override
    public Set<Long> findPersonIdsByGroupId(final long groupId) {
        return jdbcTemplate.queryForSet(PERSON_IDS_BY_GROUP_ID, ImmutableMap.of("group_id", groupId), (rs, i) -> rs.getLong("person_id"));
    }

    @NotNull
    @Override
    public Map<Long, Set<Long>> findPersonIdsByGroupIds(@NotNull final Set<Long> groupIds) {
        final Map<Long, Set<Long>> result = new HashMap<>();
        groupIds.forEach(id -> result.put(id, new HashSet<>()));
        jdbcTemplate.query(PERSON_IDS_BY_GROUP_IDS, ImmutableMap.of("group_ids", groupIds), rs -> {
            result.computeIfAbsent(rs.getLong("group_id"), k -> new HashSet<>()).add(rs.getLong("person_id"));
        });
        return result;
    }

    @NotNull
    @Override
    public Set<YaGroup> findGroupsByPersonId(final long personId) {
        return jdbcTemplate.queryForSet(GROUPS_BY_PERSON_ID, ImmutableMap.of("person_id", personId), (rs, i) -> {
            final long groupId = rs.getLong("group_id");
            return toGroup(rs, groupId);
        });
    }

    @NotNull
    @Override
    public Map<Long, Set<YaGroup>> findGroupsByPersonIds(@NotNull final Set<Long> personIds) {
        final Map<Long, YaGroup> groupsCache = new HashMap<>();
        final Map<Long, Set<YaGroup>> result = new HashMap<>();
        personIds.forEach(id -> result.put(id, new HashSet<>()));
        jdbcTemplate.query(GROUPS_BY_PERSON_IDS, ImmutableMap.of("person_ids", personIds), rs -> {
            final long personId = rs.getLong("person_id");
            final Set<YaGroup> groups = result.computeIfAbsent(personId, key -> new HashSet<>());
            final long groupId = rs.getLong("group_id");
            if (groupsCache.containsKey(groupId)) {
                groups.add(groupsCache.get(groupId));
            } else {
                final YaGroup group = toGroup(rs, groupId);
                groupsCache.put(groupId, group);
                groups.add(group);
            }
        });
        return result;
    }

    @NotNull
    @Override
    public Map<Long, Set<Long>> findPersonIdsByGroupIdsForProjectMembership() {
        final Map<Long, Set<Long>> result = new HashMap<>();
        jdbcTemplate.query(PERSONS_IDS_BY_GROUP_IDS_FOR_PROJECT_MEMBERSHIP, rs -> {
            result.computeIfAbsent(rs.getLong("group_id"), k -> new HashSet<>()).add(rs.getLong("person_id"));
        });
        return result;
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public PersonGroupMembership read(@NotNull final Long id) throws EmptyResultDataAccessException {
        return jdbcTemplate.queryForOptional(SELECT_BY_ID_QUERY, ImmutableMap.of("id", id), this::toPersonGroupMembership)
                .orElseThrow(() -> new EmptyResultDataAccessException("No person group membership with id " + id, 1));
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean update(@NotNull final PersonGroupMembership object) {
        return jdbcTemplate.update(UPDATE_QUERY, ImmutableMap.of("person_id", object.getPerson().getId(),
                "group_id", object.getGroup().getId(), "membership_type", object.getMembershipType().name())) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean updateAll(@NotNull final Collection<PersonGroupMembership> objects) {
        if (objects.isEmpty()) {
            return false;
        }
        final int[] updatedArray = jdbcTemplate.batchUpdate(UPDATE_QUERY,
                objects.stream().map(object -> ImmutableMap.of("person_id", object.getPerson().getId(),
                        "group_id", object.getGroup().getId(), "membership_type", object.getMembershipType().name()))
                        .collect(Collectors.toList()));
        boolean result = false;
        for (final int updated : updatedArray) {
            result |= updated > 0;
        }
        return result;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean delete(@NotNull final PersonGroupMembership object) {
        return jdbcTemplate.update(DELETE_QUERY, ImmutableMap.of("id", object.getId())) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean deleteAll(@NotNull final Collection<PersonGroupMembership> objects) {
        if (objects.isEmpty()) {
            return false;
        }
        final int[] deletedArray = jdbcTemplate.batchUpdate(DELETE_QUERY,
                objects.stream().map(object -> ImmutableMap.of("id", object.getId())).collect(Collectors.toList()));
        boolean result = false;
        for (final int deleted : deletedArray) {
            result |= deleted > 0;
        }
        return result;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean clear() {
        return jdbcTemplate.update(CLEAR_QUERY) > 0;
    }

    @NotNull
    private PersonGroupMembership toPersonGroupMembership(@NotNull final ResultSet rs, final int i) throws SQLException {
        final long membershipId = rs.getLong("id");
        final long personId = rs.getLong("person_id");
        final long groupId = rs.getLong("group_id");
        final PersonGroupMembershipType membershipType = PersonGroupMembershipType.valueOf(rs.getString("membership_type"));
        final YaGroup group = toGroup(rs, groupId);
        final Person person = toPerson(rs, personId);
        final PersonGroupMembership membership = new PersonGroupMembership(person, group, membershipType);
        membership.setId(membershipId);
        return membership;
    }

    @NotNull
    private Person toPerson(@NotNull final ResultSet rs, final long personId) throws SQLException {
        final String personLogin = rs.getString("login");
        final long personUid = rs.getLong("uid");
        final boolean personIsRobot = rs.getBoolean("is_robot");
        final boolean personIsDismissed = rs.getBoolean("is_dismissed");
        final boolean personIsDeleted = rs.getBoolean("is_deleted");
        final PersonAffiliation personAffiliation = PersonAffiliation.valueOf(rs.getString("affiliation"));
        return new Person(personId, personLogin, personUid, personIsRobot, personIsDismissed, personIsDeleted, personAffiliation);
    }

    @NotNull
    private YaGroup toGroup(@NotNull final ResultSet rs, final long groupId) throws SQLException {
        final String groupUrl = rs.getString("url_acceptable_key");
        final DiYandexGroupType groupType = DiYandexGroupType.valueOf(rs.getString("group_type"));
        final long groupStaffId = rs.getLong("staff_id");
        final boolean groupDeleted = rs.getBoolean("deleted");
        final YaGroup group = new YaGroup(groupUrl, groupType, groupStaffId, groupDeleted);
        group.setId(groupId);
        return group;
    }

    @NotNull
    private PersonGroupMembership toPersonGroupMembership(@NotNull final ResultSet rs, @NotNull final Map<Long, Person> personById,
                                                          @NotNull final Map<Long, YaGroup> groupById) throws SQLException {
        final long membershipId = rs.getLong("id");
        final long personId = rs.getLong("person_id");
        final long groupId = rs.getLong("group_id");
        final PersonGroupMembershipType membershipType = PersonGroupMembershipType.valueOf(rs.getString("membership_type"));
        final Person person;
        if (personById.containsKey(personId)) {
            person = personById.get(personId);
        } else {
            person = toPerson(rs, personId);
            personById.put(personId, person);
        }
        final YaGroup group;
        if (groupById.containsKey(groupId)) {
            group = groupById.get(groupId);
        } else {
            group = toGroup(rs, groupId);
            groupById.put(groupId, group);
        }
        final PersonGroupMembership membership = new PersonGroupMembership(person, group, membershipType);
        membership.setId(membershipId);
        return membership;
    }

}
