package ru.yandex.qe.dispenser.domain.dao.project;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaDao;

@Component
@ParametersAreNonnullByDefault
public class ProjectManager {

    private final ProjectDao projectDao;
    private final QuotaDao quotaDao;

    public ProjectManager(final ProjectDao projectDao,
                          final QuotaDao quotaDao) {
        this.projectDao = projectDao;
        this.quotaDao = quotaDao;
    }

    public void moveQuotas(final Project origin, final Project newParent) {
        final Project lca = origin.getLeastCommonAncestor(newParent);

        final List<Project> pathFromOriginToAncestor = origin.getPathToAncestor(lca);
        final List<Project> pathFromNewParentToAncestor = newParent.getPathToAncestor(lca);

        final Set<Project> unitedPaths = new HashSet<>();

        unitedPaths.addAll(pathFromOriginToAncestor);
        unitedPaths.addAll(pathFromNewParentToAncestor);

        final Set<Project> lockedProjects = new HashSet<>(projectDao.lockForUpdate(unitedPaths));

        if (unitedPaths.size() == lockedProjects.size()) {
            for (final Project project : unitedPaths) {
                final Project.Key curKey = project.getKey();
                for (final Project lockedProject : lockedProjects) {
                    if (lockedProject.getKey().equals(curKey)) {
                        if (lockedProject.isRemoved()) {
                            throw new EmptyResultDataAccessException("No project with key = " + origin.getPublicKey(), 1);
                        }
                        if (!project.getParent().equals(lockedProject.getParent())) {
                            throw new RuntimeException("Parents of '" + project.getPublicKey() + "' is '" + project.getParent().getPublicKey()
                                    + "' but locked is '" + lockedProject.getParent().getPublicKey() + "'");
                        }
                    }
                }
            }
        } else {
            throw new RuntimeException("Lists of projects have different sizes!");
        }

        final Set<Quota> quotas = quotaDao.getProjectsQuotasForUpdate(unitedPaths);
        final Map<Quota.Key, Quota> quotasMap = quotas.stream().collect(Collectors.toMap(Quota::getKey, quota -> quota));

        final Map<Quota.Key, Long> quotasMax = new HashMap<>();

        quotas.stream()
                .filter(quota -> quota.getProject().equals(origin))
                .forEach(quota -> {
                    final long originMax = quota.getMax();
                    if (originMax > 0) {
                        pathFromOriginToAncestor.stream()
                                .skip(1)
                                .forEach(project -> quotasMax.put(quota.getKey().withProject(project),
                                        quotasMap.get(quota.getKey().withProject(project)).getMax() - originMax));
                        pathFromNewParentToAncestor
                                .forEach(project -> quotasMax.put(quota.getKey().withProject(project),
                                        quotasMap.get(quota.getKey().withProject(project)).getMax() + originMax));
                    }

                });
        quotaDao.applyChanges(quotasMax, Collections.emptyMap(), Collections.emptyMap());
    }
}
