package ru.yandex.qe.dispenser.domain.dao.project;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.util.StreamUtils;

public enum ProjectUtils {
    ;

    @NotNull
    public static Project root(@NotNull final Collection<Project> allProjects) {
        return StreamUtils.requireSingle(allProjects.stream().filter(Project::isRoot));
    }

    @NotNull
    public static Set<Project> subTree(final @NotNull Project project) {
        final Set<Project> subTree = new HashSet<>();
        dfs(project, subTree);
        return subTree;
    }

    @NotNull
    public static Set<Project> subTree(@NotNull final Collection<Project> nodeProjects) {
        return nodeProjects.stream().flatMap(p -> subTree(p).stream()).collect(Collectors.toSet());
    }

    @NotNull
    public static Set<Project> filterLeafs(@NotNull final Collection<Project> projects) {
        return projects.stream().filter(Project::isLeaf).collect(Collectors.toSet());
    }

    @NotNull
    public static Set<Project> filterRealLeafs(@NotNull final Collection<Project> projects) {
        return projects.stream().filter(Project::isRealLeaf).collect(Collectors.toSet());
    }

    @NotNull
    public static Set<Project> subTreeLeafs(@NotNull final Project project) {
        return filterLeafs(subTree(project));
    }

    @NotNull
    public static Set<Project> subTreeLeafs(@NotNull final Project project, @NotNull final Collection<Project> allProjects) {
        return allProjects.stream().filter(Project::isLeaf).filter(p -> p.equalsOrSon(project)).collect(Collectors.toSet());
    }

    @NotNull
    public static Set<Project> subTreeLeafs(@NotNull final Collection<Project> nodeProjects) {
        return nodeProjects.stream().flatMap(p -> subTreeLeafs(p).stream()).collect(Collectors.toSet());
    }

    @NotNull
    public static Set<Project> brothers(@NotNull final Project project) {
        if (project.isRoot()) {
            return Collections.emptySet();
        }
        final Set<Project> brothers = new HashSet<>(project.getParent().getExistingSubProjects());
        brothers.remove(project);
        return brothers;
    }

    @NotNull
    public static List<Project> topologicalOrder(@NotNull final Project root) {
        final List<Project> result = new ArrayList<>();
        dfs(root, result, false);
        return result;
    }

    private static void dfs(@NotNull final Project project, @NotNull final Collection<Project> result) {
        dfs(project, result, true);
    }

    private static void dfs(@NotNull final Project project, @NotNull final Collection<Project> result, final boolean onlyExisting) {
        if (onlyExisting && project.isRemoved()) {
            return;
        }
        result.add(project);

        final Set<Project> projects = onlyExisting ? project.getExistingSubProjects() : project.getSubProjects();

        for (final Project subproject : projects) {
            dfs(subproject, result, onlyExisting);
        }
    }
}
