package ru.yandex.qe.dispenser.domain.dao.property;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;

import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.property.Property;

public class SqlPropertyDao extends SqlDaoBase implements PropertyDao {

    private final static String QUERY_BASE_PROPERTY = "SELECT * FROM property";
    private final static String INSERT_BASE_PROPERTY = "INSERT INTO property (entity_key, property_key, type) VALUES (:entityKey, :propertyKey, cast(:type as property_type))";
    private final static String DELETE_CASCADE = "DELETE FROM property WHERE id = :id";
    private final static String TRUNCATE_CASCADE = "TRUNCATE property CASCADE";

    private final Map<Property.Type<?>, TypeAdapter> adaptersByType;

    public SqlPropertyDao() {
        adaptersByType = Arrays.stream(TypeAdapter.values())
                .collect(Collectors.toMap(TypeAdapter::getType, Function.identity()));
    }

    @Override
    public Set<Property> readAll(final Property.Type<?> type) {
        final TypeAdapter typeAdapter = getTypeAdapter(type);
        return jdbcTemplate.queryForSet(QUERY_BASE_PROPERTY + " " + typeAdapter.getValueQueryPart(), this::toProperty);
    }

    public TypeAdapter getTypeAdapter(final Property.Type<?> type) {
        if (!adaptersByType.containsKey(type)) {
            throw new IllegalArgumentException("Property with type '" + type.getName() + "' is not supported");
        }
        return adaptersByType.get(type);
    }

    private Property toProperty(final ResultSet resultSet, final int i) throws SQLException {
        final Property.Type type = Property.Type.valueOf(resultSet.getString("type"));
        final TypeAdapter typeAdapter = getTypeAdapter(type);
        return new Property(
                resultSet.getLong("id"),
                resultSet.getString("entity_key"),
                resultSet.getString("property_key"),
                typeAdapter.getValue(resultSet, "value")
        );
    }

    private Map<String, ?> toPropertyParams(final Property property) {
        return ImmutableMap.of(
                "entityKey", property.getEntityKey(),
                "propertyKey", property.getPropertyKey(),
                "type", property.getValue().getType().getName()
        );
    }

    @Override
    public Property create(final Property property) {
        final long id = jdbcTemplate.insert(INSERT_BASE_PROPERTY, toPropertyParams(property));
        final TypeAdapter typeAdapter = getTypeAdapter(property.getValue().getType());
        typeAdapter.create(jdbcTemplate, id, property.getValue().get());

        return new Property(id, property.getEntityKey(), property.getPropertyKey(), property.getValue());
    }

    @Override
    public boolean update(final Property property) {
        final TypeAdapter typeAdapter = getTypeAdapter(property.getValue().getType());
        return typeAdapter.update(jdbcTemplate, property.getId(), property.getValue().get());
    }

    @Override
    public boolean delete(final Property property) {
        return jdbcTemplate.update(DELETE_CASCADE, Collections.singletonMap("id", property.getId())) > 0;
    }

    @Override
    public boolean clear() {
        return jdbcTemplate.update(TRUNCATE_CASCADE) > 0;
    }
}
