package ru.yandex.qe.dispenser.domain.dao.quota;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaSpec;
import ru.yandex.qe.dispenser.domain.QuotaView;
import ru.yandex.qe.dispenser.domain.Resource;

@ParametersAreNonnullByDefault
public class QuotaCacheImpl implements QuotaCache {

    private final Set<QuotaView> sortedViews;
    private final Map<Quota.Key, QuotaView> quotaViewByKey;

    public QuotaCacheImpl(final Collection<Quota> quotas) {
        quotaViewByKey = QuotaUtils.aggregateWithActualValuesInternal(QuotaUtils.indexByKey(quotas));
        sortedViews = new TreeSet<>(quotaViewByKey.values());
    }

    @Override
    @NotNull
    public Set<QuotaView> getAll() {
        return sortedViews;
    }

    @Override
    @NotNull
    public Set<QuotaView> getQuotas(final QuotaSpec quotaSpec) {
        return sortedViews.stream()
                .filter(q -> q.getSpec().equals(quotaSpec))
                .collect(Collectors.toSet());
    }

    @Override
    @NotNull
    public Set<QuotaView> getQuotasByResources(final Collection<Resource> resources) {
        return sortedViews.stream()
                .filter(q -> resources.contains(q.getResource()))
                .collect(Collectors.toSet());
    }

    @Override
    @NotNull
    public Set<QuotaView> getQuotas(@NotNull final Collection<Project> projects) {
        return sortedViews.stream()
                .filter(q -> projects.contains(q.getProject()))
                .collect(Collectors.toSet());
    }

    @Override
    public @NotNull QuotaView getQuota(final Quota.@NotNull Key key) {
        final QuotaView quotaView = quotaViewByKey.get(key);
        if (quotaView == null) {
            throw new EmptyResultDataAccessException("No quota with key " + key, 1);
        }
        return quotaView;
    }
}
