package ru.yandex.qe.dispenser.domain.dao.quota;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaSpec;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.dao.InMemoryLongKeyDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectDao;
import ru.yandex.qe.dispenser.domain.dao.quota.spec.QuotaSpecDao;
import ru.yandex.qe.dispenser.domain.dao.resource.segmentation.ResourceSegmentationDao;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentDao;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.support.QuotaDiff;
import ru.yandex.qe.dispenser.domain.util.Page;
import ru.yandex.qe.dispenser.domain.util.PageInfo;


public class QuotaDaoImpl extends InMemoryLongKeyDaoImpl<Quota> implements IntegratedQuotaDao {
    @Autowired
    private ProjectDao directProjectDao;
    @Autowired
    private QuotaSpecDao directQuotaSpecDao;
    @Autowired
    private ResourceSegmentationDao resourceSegmentationDao;
    @Autowired
    private SegmentDao segmentDao;

    @NotNull
    @Override
    public Set<Quota> getQuotas(@NotNull final Collection<Project> projects) {
        return filter(quota -> projects.contains(quota.getProject())).collect(Collectors.toSet());
    }

    @Override
    public @NotNull Set<Quota> getNonEmptyQuotas(@NotNull final Collection<Project> projects) {
        return getQuotas(projects);
    }

    @NotNull
    @Override
    public Set<Quota> getQuotasByResources(@NotNull final Collection<Resource> resources) {
        return filter(q -> resources.contains(q.getResource())).collect(Collectors.toSet());
    }

    @NotNull
    @Override
    public Set<Quota> getQuotas(final @NotNull QuotaSpec quotaSpec) {
        return filter(q -> q.getSpec().equals(quotaSpec)).collect(Collectors.toSet());
    }

    @NotNull
    @Override
    public Set<Quota> getQuotasForUpdate(final @NotNull QuotaSpec quotaSpec) {
        return getQuotas(quotaSpec);
    }

    private Optional<Predicate<Quota>> toPredicate(final QuotaFilterParams quotaFilterParams) {

        final List<Predicate<Quota>> filters = new ArrayList<>();

        final Set<Project> projects = quotaFilterParams.getProjects();
        if (!projects.isEmpty()) {
            filters.add(q -> projects.contains(q.getProject()));
        }

        final Set<QuotaSpec> quotaSpec = quotaFilterParams.getQuotaSpecs();
        if (!quotaSpec.isEmpty()) {
            filters.add(q -> quotaSpec.contains(q.getSpec()));
        }

        final Set<Segment> segments = quotaFilterParams.getSegments();
        if (!segments.isEmpty()) {
            filters.add(q -> q.getSegments().containsAll(segments));
        }

        return filters.stream()
                .reduce(Predicate::and);
    }


    @Override
    public Page<Quota> readPage(final QuotaFilterParams quotaFilterParams, final PageInfo pageInfo) {
        final Optional<Predicate<Quota>> predicate = toPredicate(quotaFilterParams);

        final List<Quota> filteredQuotas = predicate.map(this::filter)
                .orElseGet(() -> getAll().stream())
                .sorted(Comparator.comparingLong(LongIndexBase::getId))
                .collect(Collectors.toList());

        final long count = filteredQuotas.size();
        final long offset = pageInfo.getOffset();

        if (offset > filteredQuotas.size()) {
            return Page.of(Stream.empty(), count);
        }
        final long toIndex = Math.min(offset + pageInfo.getPageSize(), filteredQuotas.size());
        final List<Quota> pageQuotas = filteredQuotas.subList((int) offset, (int) toIndex);

        return Page.of(pageQuotas, count);
    }

    @NotNull
    @Override
    public Set<Quota> getQuotasForUpdate(@NotNull final Collection<Quota.Key> keys) {
        return filter(q -> keys.contains(q.getKey())).collect(Collectors.toSet());
    }

    @Override
    public @NotNull Set<Quota> getProjectsQuotasForUpdate(final @NotNull Collection<Project> projects) {
        return getQuotas(projects);
    }

    @Override
    public boolean update(@NotNull final Quota obj) {
        return obj == update(obj, old -> obj);
    }

    public Quota update(@NotNull final Quota obj, @NotNull final Function<Quota, Quota> modification) {
        final long id = primaryKeyProducer.apply(obj);
        obj.setId(id);
        final Quota[] result = new Quota[1];
        if (id2obj.computeIfPresent(id, (pk, t) -> {
            final Quota modified = modification.apply(t);
            result[0] = modified;
            return modified;
        }) == null) {
            throw new IllegalArgumentException(obj.getClass().getSimpleName() + " object not exists to update!");
        }
        return result[0];
    }

    @NotNull
    @Override
    public Set<Quota> changeAll(@NotNull final Collection<QuotaDiff> quotaDiffs) {
        final Set<Quota> modifiedQuotas = new HashSet<>();
        for (final QuotaDiff qd : quotaDiffs) {
            for (final Quota q : getQuotas(qd.getResource(), Collections.singleton(qd.getProject()), qd.getSegments())) {
                update(q, actual -> {
                    final Quota modified = Quota.builder(actual).change(qd.getDiff()).build();
                    modifiedQuotas.add(modified);
                    return modified;
                });
            }
        }
        return modifiedQuotas;
    }

    @Override
    public void setLastOverquotingTs(@NotNull final Quota quota, @Nullable final Long ts) {
        update(quota, (q) -> Quota.builder(q).lastOverquotingTs(ts).build());
    }

    @Override
    public void applyChanges(@NotNull final Map<Quota.Key, Long> max, @NotNull final Map<Quota.Key, Long> ownActual,
                             @NotNull final Map<Quota.Key, Long> ownMax) {
        if (max.isEmpty() && ownActual.isEmpty() && ownMax.isEmpty()) {
            return;
        }
        getAll().forEach(q -> {
            final Quota.Key key = q.getKey();
            final Long newMax = !max.isEmpty() ? max.get(key) : null;
            final Long newOwnActual = !ownActual.isEmpty() ? ownActual.get(key) : null;
            final Long newOwnMax = !ownMax.isEmpty() ? ownMax.get(key) : null;
            if (newMax != null || newOwnActual != null || newOwnMax != null) {
                final Quota.Builder builder = Quota.builder(q);
                if (newMax != null) {
                    builder.max(newMax);
                }
                if (newOwnActual != null) {
                    builder.ownActual(newOwnActual);
                }
                if (newOwnMax != null) {
                    builder.ownMax(newOwnMax);
                }
                update(builder.build());
            }
        });
    }

    @NotNull
    @Override
    public ProjectDao getDirectProjectDao() {
        return directProjectDao;
    }

    @NotNull
    @Override
    public QuotaSpecDao getDirectQuotaSpecDao() {
        return directQuotaSpecDao;
    }

    @NotNull
    @Override
    public ResourceSegmentationDao getResourceSegmentationDao() {
        return resourceSegmentationDao;
    }

    @NotNull
    @Override
    public SegmentDao getSegmentDao() {
        return segmentDao;
    }
}
