package ru.yandex.qe.dispenser.domain.dao.quota;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaSpec;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;
import ru.yandex.qe.dispenser.domain.util.Page;
import ru.yandex.qe.dispenser.domain.util.PageInfo;

public interface QuotaReader {
    @NotNull
    Set<Quota> getAll();

    @NotNull
    Set<Quota> getQuotas(@NotNull Collection<Project> projects);

    @NotNull
    Set<Quota> getNonEmptyQuotas(@NotNull Collection<Project> projects);

    @NotNull
    Set<Quota> getQuotasByResources(@NotNull Collection<Resource> resources);

    @NotNull
    default Set<Quota> getQuotas(final @NotNull Project project) {
        return getQuotas(Collections.singleton(project));
    }

    @NotNull
    Set<Quota> getQuotas(@NotNull QuotaSpec quotaSpec);

    @NotNull
    default Set<Quota> getQuotas(final @NotNull Resource resource, @NotNull final Collection<Project> projects, @NotNull final Set<Segment> segments) {
        return getQuotas(projects).stream()
                .filter(q -> q.getSpec().getKey().getResource().equals(resource))
                .filter(q -> q.getSegments().equals(segments))
                .collect(Collectors.toSet());
    }

    @NotNull
    default Set<Quota> getQuotas(final @NotNull Quota.Key originKey, @NotNull final Collection<Project> projects) {
        final Set<Quota.Key> quotaKeys = projects.stream()
                .map(originKey::withProject)
                .collect(Collectors.toSet());

        return getQuotas(projects).stream()
                .filter(q -> quotaKeys.contains(q.getKey()))
                .collect(Collectors.toSet());
    }

    @NotNull
    default Set<Quota> getQuotas(final @NotNull Project project, @NotNull final Collection<Resource> resources) {
        final Set<String> resourceKeys = CollectionUtils.map(resources, r -> r.getKey().getPublicKey());
        return CollectionUtils.filter(getQuotas(project), q -> resources.contains(q.getSpec().getKey().getResource()));
    }

    @NotNull
    default Set<Quota> getQuotasBySpecs(final @NotNull Collection<QuotaSpec> specs) {
        return specs.stream()
                .flatMap(spec -> getQuotas(spec).stream())
                .collect(Collectors.toSet());
    }

    @NotNull
    default Quota getQuota(@NotNull final Quota.Key quotaKey) {
        return getQuotas(quotaKey.getProject()).stream()
                .filter(q -> q.getKey().equals(quotaKey))
                .findAny()
                .orElseThrow(() -> new EmptyResultDataAccessException("No quota for key '" + quotaKey + "'", 1));
    }

    @NotNull
    default Quota getQuota(final long id) {
        return getAll().stream()
                .filter(q -> q.getId() == id)
                .findAny()
                .orElseThrow(() -> new EmptyResultDataAccessException("No quota for id '" + id + "'", 1));
    }

    Page<Quota> readPage(QuotaFilterParams quotaFilterParams, PageInfo pageInfo);

    interface QuotaFilterParams {
        Set<QuotaSpec> getQuotaSpecs();

        Set<Project> getProjects();

        Set<Segment> getSegments();
    }

}
