package ru.yandex.qe.dispenser.domain.dao.quota;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Stopwatch;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Multimap;
import com.google.common.collect.Table;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiQuota;
import ru.yandex.qe.dispenser.api.v1.DiQuotaKey;
import ru.yandex.qe.dispenser.api.v1.DiQuotaLightView;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.Entity;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaAggregateViewImpl;
import ru.yandex.qe.dispenser.domain.QuotaSpec;
import ru.yandex.qe.dispenser.domain.QuotaView;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.support.EntityOwnership;
import ru.yandex.qe.dispenser.domain.support.QuotaDiff;
import ru.yandex.qe.dispenser.domain.util.ApplicationContextProvider;
import ru.yandex.qe.dispenser.domain.util.MoreCollectors;
import ru.yandex.qe.dispenser.solomon.Solomon;

public enum QuotaUtils {
    ;
    public static final Logger LOG = LoggerFactory.getLogger(QuotaUtils.class);

    @NotNull
    public static Collection<QuotaDiff> expandToReal(@NotNull final Collection<QuotaDiff> quotaDiffs) {
        final Collection<QuotaDiff> expandedDiffs = new ArrayList<>(quotaDiffs);
        quotaDiffs.stream().filter(qd -> qd.getProject().isPersonal()).forEach(qd -> {
            expandedDiffs.add(new QuotaDiff(qd.getResource(), qd.getProject().getParent(), qd.getDiff(), qd.getCause(), qd.getSegments()));
        });
        return expandedDiffs;
    }

    @NotNull
    public static Multimap<Resource, Project> resourceProjects(@NotNull final Collection<QuotaDiff> diffs) {
        return diffs.stream().collect(MoreCollectors.toLinkedMultimap(QuotaDiff::getResource, QuotaDiff::getProject));
    }

    @NotNull
    public static Set<Quota.Key> getQuotaKeys(@NotNull final Collection<QuotaDiff> diffs) {

        final Set<Resource> resources = diffs.stream()
                .map(QuotaDiff::getResource)
                .collect(Collectors.toSet());

        final Multimap<Resource, QuotaSpec> resourceSpecs = Hierarchy.get().getQuotaSpecReader().getByResources(resources);

        return diffs.stream()
                .flatMap(diff -> resourceSpecs.get(diff.getResource())
                        .stream()
                        .map(spec -> new Quota.Key(spec, diff.getProject(), diff.getSegments()))
                ).collect(Collectors.toSet());
    }

    @NotNull
    public static Map<Quota.Key, QuotaView> aggregateWithActualValuesInternal(@NotNull final Map<Quota.Key, Quota> quotaMap) {
        final Map<Quota.Key, Quota> aggreatedMaxValues = aggregateTotalSegmentValues(quotaMap);
        return aggregateActualValues(aggreatedMaxValues.values());
    }

    public static Map<Quota.Key, Quota> aggregateTotalSegmentValues(@NotNull final Map<Quota.Key, Quota> quotaMap) {
        aggregateTotalSegmentMaxValues(quotaMap);
        return quotaMap;
    }

    @NotNull
    public static Map<Quota.Key, Quota> indexByKey(final @NotNull Collection<Quota> subTreeQuotas) {
        return subTreeQuotas.stream()
                .collect(Collectors.toMap(Quota::getKey, Function.identity()));
    }

    @NotNull
    private static Map<Quota.Key, QuotaView> aggregateActualValues(@NotNull final Collection<Quota> subTreeQuotas) {
        LOG.debug("QuotaUtils.aggregateActualValues execution started");
        final Stopwatch stopwatch = Stopwatch.createStarted();

        final Map<Quota.Key, Quota> aggregatedQuotasByKey = new HashMap<>();
        final Set<Quota.Key> quotasWithOwnActualValue = new HashSet<>();

        final Map<Quota.Key, Long> valuesToIncrement = new HashMap<>();

        final Set<Quota.Key> quotaKeys = new HashSet<>();

        for (final Quota quota : subTreeQuotas) {
            final Quota.Key key = quota.getKey();
            quotaKeys.add(key);

            final long ownActual = quota.getOwnActual();
            aggregatedQuotasByKey.put(key, quota);
            valuesToIncrement.put(key, ownActual);

            if (ownActual > 0) {
                quotasWithOwnActualValue.add(key);
            }
        }

        quotasWithOwnActualValue.forEach(key -> getQuotasForAllAffectedSegments(valuesToIncrement, key));

        final Map<Quota.Key, Long> aggregatedActualValue = incrementRealQuotas(valuesToIncrement, quotaKeys);

        final Map<Quota.Key, QuotaView> quotasWithActualValues = aggregatedActualValue.keySet().stream()
                .map(aggregatedQuotasByKey::get)
                .map(quota -> new QuotaAggregateViewImpl(quota, aggregatedActualValue.get(quota.getKey())))
                .collect(Collectors.toMap(QuotaView::getKey, Function.identity()));

        LOG.debug("QuotaUtils.aggregateActualValues execution finished, execution time {}", stopwatch.elapsed(TimeUnit.MILLISECONDS));
        return quotasWithActualValues;
    }

    private static void aggregateTotalSegmentMaxValues(@NotNull final Map<Quota.Key, Quota> originalQuotas) {
        LOG.debug("QuotaUtils.aggregateTotalSegmentMaxValues execution started");
        final Stopwatch stopwatch = Stopwatch.createStarted();

        final Table<Project, QuotaSpec, Long> totalValues = HashBasedTable.create();

        for (final Quota quota : originalQuotas.values()) {
            if (!quota.getSegments().isEmpty() && !quota.getKey().isAggregation()) {
                final Long total = totalValues.get(quota.getProject(), quota.getSpec());
                totalValues.put(quota.getProject(), quota.getSpec(), quota.getMax() + (total == null ? 0 : total));
            }
        }

        for (final Table.Cell<Project, QuotaSpec, Long> cell : totalValues.cellSet()) {
            final Quota.Key totalKey = Quota.Key.totalOf(cell.getColumnKey(), cell.getRowKey());

            final Quota changedQuota = Quota.builder(originalQuotas.get(totalKey))
                    .max(cell.getValue())
                    .build();

            originalQuotas.put(totalKey, changedQuota);
        }

        LOG.debug("QuotaUtils.aggregateTotalSegmentMaxValues execution finished, execution time {}", stopwatch.elapsed(TimeUnit.MILLISECONDS));
    }

    private static Map<Quota.Key, Long> incrementRealQuotas(@NotNull final Map<Quota.Key, Long> valuesToIncrementByKey,
                                                            @NotNull final Set<Quota.Key> quotaKeys
    ) {
        final Map<Quota.Key, Long> totalActualValuesByKey = quotaKeys.stream()
                .collect(Collectors.toMap(Function.identity(), key -> 0L));

        valuesToIncrementByKey.forEach((quotaKey, diff) -> {

            if (diff > 0) {

                quotaKey.getProject().getPathToRoot().forEach(p -> {
                    final Quota.Key key = quotaKey.withProject(p);
                    Optional.ofNullable(totalActualValuesByKey.get(key))
                            .ifPresent(actual -> totalActualValuesByKey.put(key, actual + diff));
                });

            }
        });

        return totalActualValuesByKey;
    }

    private static void getQuotasForAllAffectedSegments(@NotNull final Map<Quota.Key, Long> realQuotas,
                                                        @NotNull final Quota.Key quotaKey) {
        final long diff = realQuotas.get(quotaKey);
        final Set<Segment> segments = quotaKey.getSegments();

        if (segments.isEmpty()) {
            return;
        }

        final boolean isAggregationQuota = segments.stream()
                .anyMatch(Segment::isAggregationSegment);

        if (isAggregationQuota) {
            return;
        }

        final Set<Segment> totalSegment = segments.stream()
                .map(s -> Segment.totalOf(s.getKey().getSegmentation()))
                .collect(Collectors.toSet());

        final Stream<Set<Segment>> segmentSets = segments.stream()
                .map(s -> {
                    final HashSet<Segment> newSegments = new HashSet<>(segments);
                    newSegments.remove(s);
                    newSegments.add(Segment.totalOf(s.getKey().getSegmentation()));
                    return newSegments;
                });

        Stream.concat(segmentSets, Stream.of(totalSegment))
                .distinct()
                .map(quotaKey::withSegments)
                .filter(realQuotas::containsKey)
                .forEach(key -> realQuotas.put(key, realQuotas.get(key) + diff));
    }

    @NotNull
    public static Table<Entity, Project, EntityOwnership> ownershipTable(@NotNull final Collection<EntityOwnership> subTreeOwnerships) {
        final Table<Entity, Project, EntityOwnership> ownershipTable = HashBasedTable.create();
        subTreeOwnerships.stream()
                .filter(o -> o.getUsages() != 0)
                .forEach(personalOwnership -> incrementRealOwnerships(ownershipTable, personalOwnership));
        return ownershipTable;
    }

    private static void incrementRealOwnerships(@NotNull final Table<Entity, Project, EntityOwnership> realOwnerships,
                                                @NotNull final EntityOwnership personalOwnership) {
        personalOwnership.getProject().getPathToRoot().forEach(p -> {
            final Entity e = personalOwnership.getEntity();
            final int diff = personalOwnership.getUsages();
            final EntityOwnership o = Optional.ofNullable(realOwnerships.get(e, p))
                    .orElse(new EntityOwnership(e, p, 0))
                    .increment(diff);
            realOwnerships.put(e, p, o);
        });
    }

    @NotNull
    public static String html(final int percent) {
        return percent == Integer.MAX_VALUE ? "&infin;" : Integer.toString(percent);
    }

    public static Set<Quota> lockNearestQuotas(final QuotaDao quotaDao, final Quota.ChangeHolder changes) {
        final Set<Quota.Key> quotaKeysWithMaxChanges = changes.getQuotaKeysWithMaxChanges();
        final Set<Quota.Key> changedQuotaKeys = changes.getChangedQuotaKeys();

        // Lock all quotas involved in check: quotas for project itself, its subprojects and parent project
        final Set<Quota.Key> quotaKeysToLock = Stream.of(
                changedQuotaKeys.stream(),
                quotaKeysWithMaxChanges.stream().flatMap(key -> key.getProject().getExistingSubProjects().stream().map(key::withProject)),
                quotaKeysWithMaxChanges.stream().filter(key -> !key.getProject().isRoot()).map(key -> key.withProject(key.getProject().getParent()))
        ).flatMap(s -> s).collect(Collectors.toSet());
        return quotaDao.getQuotasForUpdate(quotaKeysToLock);
    }

    public static List<QuotaView> lockAndCheckValues(final QuotaDao quotaDao, final Quota.ChangeHolder changes) {
        lockNearestQuotas(quotaDao, changes);
        return checkValues(quotaDao, changes);
    }

    public static List<QuotaView> checkValues(@NotNull final QuotaDao quotaDao,
                                              @NotNull final Quota.ChangeHolder changes) {
        final Set<Quota.Key> quotaKeysWithMaxChanges = changes.getQuotaKeysWithMaxChanges();
        final Set<Quota.Key> changedQuotaKeys = changes.getChangedQuotaKeys();

        final Set<QuotaSpec> quotaSpecs = changedQuotaKeys.stream()
                .map(Quota.Key::getSpec)
                .collect(Collectors.toSet());

        final Set<Quota> quotas = changes.applyToQuotas(quotaDao.getQuotasBySpecs(quotaSpecs));
        final Map<Quota.Key, Quota> quotaByKey = indexByKey(quotas);
        final Map<Quota.Key, QuotaView> quotaViewByKey = aggregateWithActualValuesInternal(quotaByKey);

        final Set<Quota.Key> quotaKeysWithOwnMaxChanges = changes.getOwnMax().keySet();

        for (final Quota.Key quotaKey : quotaKeysWithMaxChanges) {
            final QuotaView quotaView = quotaViewByKey.get(quotaKey);
            final long newMaxValue = quotaView.getMax();
            final long newActualValue = quotaView.getTotalActual();

            final Service.Settings serviceSettings = quotaKey.getSpec().getResource().getService().getSettings();

            if (serviceSettings.accountActualValuesInQuotaDistribution()) {
                if (newMaxValue < newActualValue) {
                    throw new IllegalArgumentException(error("New quota's max", "actual", newMaxValue, newActualValue, quotaView));
                }
                if (quotaKeysWithOwnMaxChanges.contains(quotaKey) && quotaView.getEffectiveOwnMax() < quotaView.getOwnActual()) {
                    throw new IllegalArgumentException(error("New quota's own max", "own actual", quotaView.getEffectiveOwnMax(),
                            quotaView.getOwnActual(), quotaView));
                }
            }

            final Function<Project, QuotaView> projectQuotaProducer = project -> {
                final Quota.Key projectQuotaKey = quotaKey.withProject(project);
                return quotaViewByKey.get(projectQuotaKey);
            };

            checkSubprojectsOccupiedValuesSum(quotaKey.getProject(), projectQuotaProducer);
            if (!quotaKey.getProject().isRoot()) {
                checkSubprojectsOccupiedValuesSum(quotaKey.getProject().getParent(), projectQuotaProducer);
            }
        }
        return changedQuotaKeys.stream()
                .map(quotaViewByKey::get)
                .collect(Collectors.toList());
    }

    public static long getMax(final Service.Settings serviceSettings, final long maxValue, final long actualValue) {
        return serviceSettings.accountActualValuesInQuotaDistribution() ? Math.max(maxValue, actualValue) : maxValue;
    }

    private static void checkSubprojectsOccupiedValuesSum(@NotNull final Project parentProject,
                                                          @NotNull final Function<Project, QuotaView> projectQuotaProducer) {
        final QuotaView parentQuota = projectQuotaProducer.apply(parentProject);

        final Service.Settings serviceSettings = parentQuota.getSpec().getResource().getService().getSettings();

        // Perform check only if service uses project hierarchy (see DISPENSER-1776)
        if (!serviceSettings.usesProjectHierarchy()) {
            return;
        }

        final long maxValue = parentQuota.getMax();
        final long subprojectsSum = parentQuota.getProject().getExistingSubProjects().stream()
                .mapToLong(p -> {
                    final QuotaView quota = projectQuotaProducer.apply(p);
                    return getMax(serviceSettings, quota.getMax(), quota.getTotalActual());
                })
                .sum() + getMax(serviceSettings, parentQuota.getEffectiveOwnMax(), parentQuota.getOwnActual());
        if (maxValue < subprojectsSum) {
            final String error = error("Quota's max", "subprojects usage", maxValue, subprojectsSum, parentQuota);
            throw new IllegalArgumentException(error);
        }
    }

    public static String humanizedDiff(final long first, final long second, final DiUnit unit) {
        final StringBuilder diffBuilder = new StringBuilder();
        if (first < second) {
            diffBuilder.append('-');
        }
        final long diff = first - second;
        final DiAmount.Humanized humanize = DiAmount.of(Math.abs(diff), unit).humanize();
        diffBuilder.append(humanize.toString());
        if (humanize.getUnit() != unit) {
            diffBuilder.append(" (").append(diff).append(' ').append(unit.getAbbreviation()).append(')');
        }
        return diffBuilder.toString();
    }

    private static String error(final String leftName, final String rightName, final long left, final long right, final QuotaView quota) {
        final DiUnit baseUnit = quota.getResource().getType().getBaseUnit();
        final String resource = quota.getKey().getSpec().getResource().getName();
        final String project = quota.getProject().getName();

        final DiAmount humanizedLeft = DiAmount.of(left, baseUnit).humanize();
        final DiAmount humanizedRight = DiAmount.of(right, baseUnit).humanize();

        return String.format("%s for resource '%s' for project '%s' (%s) is less than %s (%s). Diff: %s", StringUtils.capitalize(leftName),
                resource, project, humanizedLeft, rightName, humanizedRight, humanizedDiff(right, left, baseUnit));
    }

    @NotNull
    public static Quota.Key parseKey(@NotNull final DiQuotaKey key) {
        final Hierarchy hierarchy = Hierarchy.get();
        final Project project = hierarchy.getProjectReader().readExisting(key.getProjectKey());
        final Service service = hierarchy.getServiceReader().read(key.getServiceKey());
        final Resource resource = hierarchy.getResourceReader().read(new Resource.Key(key.getResourceKey(), service));
        final QuotaSpec spec = hierarchy.getQuotaSpecReader().read(resource, key.getQuotaSpecKey());

        return new Quota.Key(spec, project, SegmentUtils.getCompleteSegmentSet(resource, key.getSegmentKeys()));
    }

    @NotNull
    public static Collection<Quota.Key> getQuotaKeys(@NotNull final Collection<QuotaSpec> specs,
                                                     @NotNull final Collection<Project> projects,
                                                     @NotNull final Collection<Set<Segment>> segmentsSets) {

        final ArrayList<Quota.Key> result = new ArrayList<>(specs.size() * projects.size() * segmentsSets.size());

        for (final Set<Segment> segments : segmentsSets) {
            for (final Project project : projects) {
                for (final QuotaSpec spec : specs) {
                    result.add(new Quota.Key(spec, project, segments));
                }
            }
        }
        return result;
    }


    @NotNull
    public static Optional<Map<Resource, QuotaSpec>> getQuotaSpecByResource(final Collection<Resource> resources) {
        final Multimap<Resource, QuotaSpec> resourceQuotaSpecMapping = Hierarchy.get().getQuotaSpecReader().getByResources(resources);

        for (final Resource resource : resources) {
            final int quotaSpecCount = resourceQuotaSpecMapping.get(resource).size();
            if (quotaSpecCount > 1) {
                throw new IllegalArgumentException(String.format("Resources '%s' has more than 1 quota spec (%d)", resource.getKey(), quotaSpecCount));
            }
            // do not apply any changes if at least one resource has no quotas
            if (quotaSpecCount == 0) {
                return Optional.empty();
            }
        }

        final Map<Resource, QuotaSpec> result = resourceQuotaSpecMapping.entries().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

        return Optional.of(result);
    }

    @NotNull
    public static DiQuotaLightView toLightView(final QuotaView quotaView) {
        return DiQuotaLightView.withKey(quotaView.getKey().toView())
                .withMax(quotaView.getMax())
                .withOwnMax(quotaView.getOwnMax())
                .withOwnActual(quotaView.getOwnActual())
                .withActual(quotaView.getTotalActual())
                .withLastOverquotingTs(quotaView.getLastOverquotingTs())
                .build();
    }

    @NotNull
    public static DiQuota toView(final QuotaView quotaView) {
        final DiUnit baseUnit = quotaView.getResource().getType().getBaseUnit();

        return DiQuota.builder(quotaView.getSpec().toView(), quotaView.getProject().toView())
                .max(DiAmount.of(quotaView.getMax(), baseUnit))
                .actual(DiAmount.of(quotaView.getTotalActual(), baseUnit))
                .ownMax(DiAmount.of(quotaView.getEffectiveOwnMax(), baseUnit))
                .ownActual(DiAmount.of(quotaView.getOwnActual(), baseUnit))
                .lastOverquotingTs(quotaView.getLastOverquotingTs())
                .statisticsLink(ApplicationContextProvider.demandBean(Solomon.class).getStatisticsLink(quotaView.getKey()))
                .segments(SegmentUtils.getNonAggregationSegmentKeys(quotaView.getSegments()))
                .build();
    }
}
