package ru.yandex.qe.dispenser.domain.dao.quota;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.StringJoiner;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import com.google.common.collect.SetMultimap;
import com.google.common.collect.Sets;
import com.healthmarketscience.sqlbuilder.BinaryCondition;
import com.healthmarketscience.sqlbuilder.CustomSql;
import com.healthmarketscience.sqlbuilder.InCondition;
import com.healthmarketscience.sqlbuilder.SelectQuery;
import com.healthmarketscience.sqlbuilder.UnaryCondition;
import com.healthmarketscience.sqlbuilder.custom.postgresql.PgLimitClause;
import com.healthmarketscience.sqlbuilder.custom.postgresql.PgOffsetClause;
import com.healthmarketscience.sqlbuilder.dbspec.basic.DbColumn;
import com.healthmarketscience.sqlbuilder.dbspec.basic.DbSchema;
import com.healthmarketscience.sqlbuilder.dbspec.basic.DbSpec;
import com.healthmarketscience.sqlbuilder.dbspec.basic.DbTable;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaSpec;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.ResourceSegmentation;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.dao.SqlUtils;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectDao;
import ru.yandex.qe.dispenser.domain.dao.project.SqlProjectDao;
import ru.yandex.qe.dispenser.domain.dao.quota.segment.SqlQuotaSegmentDao;
import ru.yandex.qe.dispenser.domain.dao.quota.spec.QuotaSpecDao;
import ru.yandex.qe.dispenser.domain.dao.quota.spec.SqlQuotaSpecDao;
import ru.yandex.qe.dispenser.domain.dao.resource.segmentation.ResourceSegmentationDao;
import ru.yandex.qe.dispenser.domain.dao.resource.segmentation.SqlResourceSegmentationDao;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.support.QuotaDiff;
import ru.yandex.qe.dispenser.domain.util.MathUtils;
import ru.yandex.qe.dispenser.domain.util.MoreCollectors;
import ru.yandex.qe.dispenser.domain.util.Page;
import ru.yandex.qe.dispenser.domain.util.PageInfo;

import static ru.yandex.qe.dispenser.domain.util.CollectionUtils.ids;

public class SqlQuotaDao extends SqlDaoBase implements IntegratedQuotaDao {
    private static final Logger LOG = LoggerFactory.getLogger(SqlQuotaDao.class);

    public static final String GET_ALL_QUOTAS_QUERY = "SELECT * FROM quota LEFT JOIN quota_segment ON quota.id = quota_segment.quota_id";
    public static final String GET_ALL_QUOTAS_BY_ID_QUERY_TEMPLATE = GET_ALL_QUOTAS_QUERY + " WHERE quota.id IN (%s) ORDER BY quota.id";
    public static final String GET_ALL_QUOTAS_QUERY_NON_EMPTY = "SELECT * FROM quota LEFT JOIN quota_segment ON quota.id = quota_segment.quota_id WHERE max_value > 0 OR actual_value > 0 OR own_max_value > 0 OR last_overquoting_ts IS NOT NULL";

    private static final String GET_QUOTAS_WITH_SEGMENTS_QUERY = "SELECT * FROM quota JOIN quota_segment ON quota.id = quota_segment.quota_id";
    private static final String GET_ALL_ONLY_QUOTAS_QUERY = "SELECT *, null as segmentation_id FROM quota";

    public static final String GET_ALL_QUOTAS_WITH_SPEC_QUERY =
            GET_ALL_QUOTAS_QUERY + " JOIN quota_spec ON quota_spec.id = quota.quota_spec_id";


    private static final String GET_PROJECTS_QUOTAS_QUERY = GET_ALL_QUOTAS_QUERY + " WHERE quota.project_id IN (:projectIds)";
    private static final String GET_QUOTAS_BY_SPEC_QUERY = GET_ALL_QUOTAS_QUERY + " WHERE quota.quota_spec_id = :quotaSpecId";
    private static final String GET_QUOTAS_BY_RESOURCE_PROJECTS_QUERY =
            GET_ALL_QUOTAS_WITH_SPEC_QUERY + " WHERE quota_spec.resource_id = :resourceId AND quota.project_id IN (:projectIds)";
    private static final String GET_QUOTAS_BY_RESOURCES =
            GET_ALL_QUOTAS_WITH_SPEC_QUERY + " WHERE quota_spec.resource_id IN (:resourceIds)";
    private static final String SELECT_BY_ID_QUERY = GET_ALL_QUOTAS_QUERY + " WHERE id = :id";
    private static final String SELECT_BY_IDS_QUERY = GET_ALL_QUOTAS_QUERY + " WHERE id IN (:ids)";
    private static final String UPDATE_QUERY = "UPDATE quota SET max_value = :maxValue, own_max_value = :ownMaxValue WHERE id = :id";
    private static final String REMOVE_QUERY = "DELETE FROM quota WHERE id = :id";

    private static final String CLEAR_QUERY = "TRUNCATE quota CASCADE";

    private static final String GET_QUOTAS_BY_SPEC_PROJECTS_NO_SEGMENTS_QUERY_FOR_UPDATE = GET_ALL_ONLY_QUOTAS_QUERY
            + " WHERE %s ORDER BY quota.id FOR UPDATE";
    private static final String GET_QUOTAS_BY_SPEC_PROJECTS_WITH_SEGMENTS_QUERY_FOR_UPDATE = GET_QUOTAS_WITH_SEGMENTS_QUERY
            + " WHERE %s ORDER BY quota.id FOR UPDATE";
    private static final String GET_QUOTAS_BY_PROJECTS_WITH_SEGMENTS_QUERY_FOR_UPDATE = GET_QUOTAS_WITH_SEGMENTS_QUERY
            + " WHERE quota.project_id IN (:projectIds) AND quota.quota_spec_id IN (:segmentQuotaSpecIds) FOR UPDATE";
    private static final String GET_QUOTAS_BY_PROJECTS_NO_SEGMENTS_QUERY_FOR_UPDATE = GET_ALL_ONLY_QUOTAS_QUERY
            + " WHERE quota.project_id IN (:projectIds) AND quota.quota_spec_id NOT IN (:segmentQuotaSpecIds) FOR UPDATE";
    private static final String GET_QUOTAS_BY_SPEC_WITH_SEGMENTS_QUERY_FOR_UPDATE = GET_QUOTAS_WITH_SEGMENTS_QUERY
            + " WHERE quota.quota_spec_id = :quotaSpecId FOR UPDATE";
    private static final String GET_QUOTAS_BY_SPEC_NO_SEGMENTS_QUERY_FOR_UPDATE = GET_ALL_ONLY_QUOTAS_QUERY
            + " WHERE quota.quota_spec_id = :quotaSpecId FOR UPDATE";

    private static final String GET_QUOTAS_BY_PROJECTS_QUERY_FOR_UPDATE = GET_ALL_ONLY_QUOTAS_QUERY
            + " WHERE quota.project_id IN (:projectIds) FOR UPDATE";

    private static final String CHANGE_QUOTAS_QUERY = "UPDATE quota SET actual_value = actual_value + :diff WHERE id = :id";
    private static final String SET_QUOTAS_QUERY = "UPDATE quota SET actual_value = :ownActualValue WHERE id = :id";
    private static final String UPDATE_LOTS_QUERY = "UPDATE quota SET last_overquoting_ts = :lastOverquotingTs WHERE id = :id";

    private static final DbSchema SCHEMA = new DbSchema(new DbSpec(), "public");
    private static final DbTable QUOTA_TABLE = new DbTable(SCHEMA, "quota", "q");

    public static final String BASE_QUOTA_UPDATE_QUERY = "UPDATE quota SET %s FROM (%s) AS quota_ids where quota.id = quota_ids.id";
    public static final String BASE_SEGMENTED_QUOTA_UPDATE_SEARCH_SUBQUERY = "SELECT q.id FROM quota q JOIN quota_segment qs ON q.id = qs.quota_id WHERE q.project_id = :project_id AND q.quota_spec_id = :quota_spec_id AND (%s) GROUP BY q.id HAVING COUNT(qs.id) = :segments_count";
    public static final String BASE_QUOTA_UPDATE_SEARCH_SUBQUERY = "SELECT q.id FROM quota q WHERE q.project_id = :project_id AND q.quota_spec_id = :quota_spec_id";

    static {
        QUOTA_TABLE.addColumn("id");
        QUOTA_TABLE.addColumn("quota_spec_id");
        QUOTA_TABLE.addColumn("project_id");
        QUOTA_TABLE.addColumn("max_value");
        QUOTA_TABLE.addColumn("actual_value");
        QUOTA_TABLE.addColumn("last_overquoting_ts");
        QUOTA_TABLE.addColumn("own_max_value");
    }

    @Autowired
    private ProjectDao directProjectDao;
    @Autowired
    private QuotaSpecDao directQuotaSpecDao;
    @Autowired
    private ResourceSegmentationDao resourceSegmentationDao;
    @Autowired
    private SegmentDao segmentDao;

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<Quota> changeAll(@NotNull final Collection<QuotaDiff> quotaDiffs) {
        if (quotaDiffs.isEmpty()) {
            return Collections.emptySet();
        }

        final Set<Quota> lockedQuotas = getQuotasForUpdate(QuotaUtils.getQuotaKeys(quotaDiffs));

        final Map<Long, Long> id2diff = new HashMap<>();
        quotaDiffs.forEach(qd -> lockedQuotas.stream().filter(qd::isFor).forEach(q -> {
            MathUtils.increment(id2diff, q.getId(), qd.getDiff());
        }));

        final List<SqlParameterSource> increments = new ArrayList<>();
        id2diff.forEach((id, diff) -> {
            if (diff != 0) {
                increments.add(new MapSqlParameterSource("id", id).addValue("diff", diff));
            }
        });

        jdbcTemplate.batchUpdate(CHANGE_QUOTAS_QUERY, increments);

        return lockedQuotas;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public void setLastOverquotingTs(@NotNull final Quota quota, @Nullable final Long ts) {
        assert quota.getId() > 0;
        final Map<String, Object> params = toParams(quota);
        params.put("lastOverquotingTs", SqlUtils.toTimestamp(ts));
        jdbcTemplate.update(UPDATE_LOTS_QUERY, params);
    }

    @Override
    public void applyChanges(@NotNull final Map<Quota.Key, Long> max, @NotNull final Map<Quota.Key, Long> ownActual,
                             @NotNull final Map<Quota.Key, Long> ownMax) {
        if (max.isEmpty() && ownActual.isEmpty() && ownMax.isEmpty()) {
            return;
        }
        final Set<Quota.Key> changedKeys = new HashSet<>();
        changedKeys.addAll(max.keySet());
        changedKeys.addAll(ownActual.keySet());
        changedKeys.addAll(ownMax.keySet());
        final Map<QuotaSegmentsClass, Set<Quota.Key>> bySegmentCounts = changedKeys.stream()
                .collect(Collectors.groupingBy(k -> new QuotaSegmentsClass(k.getSegments()), Collectors.toSet()));
        bySegmentCounts.forEach((quotaSegmentsClass, segmentClassKeys) -> {
            final Map<QuotaUpdatesClass, Set<Quota.Key>> byUpdatesClass = segmentClassKeys.stream()
                    .collect(Collectors.groupingBy(k -> new QuotaUpdatesClass(k, max, ownActual, ownMax), Collectors.toSet()));
            if (byUpdatesClass.size() > 3) {
                // Just do 3 sequential updates for each field
                final Set<Quota.Key> maxUpdateKeys = segmentClassKeys.stream().filter(max::containsKey).collect(Collectors.toSet());
                final Map<Quota.Key, Long> maxForUpdate = max.entrySet().stream().filter(e -> maxUpdateKeys.contains(e.getKey()))
                        .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
                final Set<Quota.Key> ownActualUpdateKeys = segmentClassKeys.stream().filter(ownActual::containsKey).collect(Collectors.toSet());
                final Map<Quota.Key, Long> ownActualForUpdate = ownActual.entrySet().stream().filter(e -> ownActualUpdateKeys.contains(e.getKey()))
                        .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
                final Set<Quota.Key> ownMaxUpdateKeys = segmentClassKeys.stream().filter(ownMax::containsKey).collect(Collectors.toSet());
                final Map<Quota.Key, Long> ownMaxForUpdate = ownMax.entrySet().stream().filter(e -> ownMaxUpdateKeys.contains(e.getKey()))
                        .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
                doQuotaUpdates(maxForUpdate, Collections.emptyMap(), Collections.emptyMap(), quotaSegmentsClass, QuotaUpdatesClass.MAX, maxUpdateKeys);
                doQuotaUpdates(Collections.emptyMap(), ownActualForUpdate, Collections.emptyMap(), quotaSegmentsClass, QuotaUpdatesClass.OWN_ACTUAL, ownActualUpdateKeys);
                doQuotaUpdates(Collections.emptyMap(), Collections.emptyMap(), ownMaxForUpdate, quotaSegmentsClass, QuotaUpdatesClass.OWN_MAX, ownMaxUpdateKeys);
            } else {
                // Do three or less combined updates
                byUpdatesClass.forEach((quotaUpdatesClass, updateClassKeys) -> {
                    doQuotaUpdates(max, ownActual, ownMax, quotaSegmentsClass, quotaUpdatesClass, updateClassKeys);
                });
            }
        });
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public Set<Quota> getAll() {
        return queryForSet(GET_ALL_QUOTAS_QUERY).stream().filter(q -> q != Quota.FAKE).collect(Collectors.toSet());
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public Set<Quota> getQuotas(@NotNull final Collection<Project> projects) {
        if (projects.isEmpty()) {
            return Collections.emptySet();
        }
        final Set<Long> projectIds = ids(projects);
        if (projects.size() < 100) { // TODO
            return queryForSet(GET_PROJECTS_QUOTAS_QUERY, ImmutableMap.of("projectIds", projectIds)).stream().filter(q -> q != Quota.FAKE).collect(Collectors.toSet());
        }
        return queryForSet(GET_ALL_QUOTAS_QUERY, Collections.emptyMap(), rch -> {
            final long projectId = rch.getLong("project_id");
            return projectIds.contains(projectId);
        });
    }

    @Override
    public @NotNull Set<Quota> getNonEmptyQuotas(@NotNull final Collection<Project> projects) {
        if (projects.isEmpty()) {
            return Collections.emptySet();
        }
        final Set<Long> projectIds = ids(projects);
        return queryForSet(GET_ALL_QUOTAS_QUERY_NON_EMPTY, Collections.emptyMap(), rch -> {
            final long projectId = rch.getLong("project_id");
            return projectIds.contains(projectId);
        });
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public Set<Quota> getQuotasByResources(@NotNull final Collection<Resource> resources) {
        if (resources.isEmpty()) {
            return Collections.emptySet();
        }
        return queryForSet(GET_QUOTAS_BY_RESOURCES, ImmutableMap.of("resourceIds", ids(resources))).stream().filter(q -> q != Quota.FAKE).collect(Collectors.toSet());
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<Quota> getQuotas(final @NotNull QuotaSpec quotaSpec) {
        return getQuotasBySpec(quotaSpec, GET_QUOTAS_BY_SPEC_QUERY);
    }

    private Set<Quota> getQuotasBySpec(final @NotNull QuotaSpec quotaSpec, final String query) {
        return queryForSet(query, ImmutableMap.of("quotaSpecId", quotaSpec.getId()))
                .stream()
                .filter(q -> q != Quota.FAKE)
                .collect(Collectors.toSet());
    }

    @NotNull
    private Set<Quota> getQuotasForUpdate(@NotNull final String query, @NotNull final Collection<Quota.Key> quotaKeys) {
        if (quotaKeys.isEmpty()) {
            return Collections.emptySet();
        }

        final Multimap<Set<Project>, QuotaSpec> mergedProjectResources = HashMultimap.create();

        final Multimap<QuotaSpec, Project> specificationProjects = quotaKeys.stream()
                .collect(MoreCollectors.toLinkedMultimap(Quota.Key::getSpec, Quota.Key::getProject));

        specificationProjects.asMap().forEach((spec, projects) -> mergedProjectResources.put(ImmutableSet.copyOf(projects), spec));

        final StringJoiner conditions = new StringJoiner(" OR ");
        final Map<String, Object> params = new HashMap<>();
        int i = 0;
        for (final Set<Project> projects : mergedProjectResources.keySet()) {
            final Collection<QuotaSpec> specs = mergedProjectResources.get(projects);
            if (!specs.isEmpty() && !projects.isEmpty()) {
                conditions.add(String.format("(quota.quota_spec_id IN (:quotaSpecIds%d) AND quota.project_id IN (:projectIds%d))", i, i));
                params.put("quotaSpecIds" + i, ids(specs));
                params.put("projectIds" + i, ids(projects));
                i++;
            }
        }

        if (params.isEmpty()) {
            return Collections.emptySet();
        }

        return queryForSet(String.format(query, conditions), params)
                .stream()
                .filter(q -> q != Quota.FAKE)
                .collect(Collectors.toSet());
    }

    @NotNull
    @Override
    public Set<Quota> getQuotasForUpdate(@NotNull final Collection<Quota.Key> keys) {
        final Map<Boolean, List<Quota.Key>> quotaSegmentTypes = keys.stream()
                .collect(Collectors.partitioningBy(key -> !key.getSegments().isEmpty()));

        final Set<Quota> quotaWithoutSegments = getQuotasForUpdate(GET_QUOTAS_BY_SPEC_PROJECTS_NO_SEGMENTS_QUERY_FOR_UPDATE, quotaSegmentTypes.get(false));

        final List<Quota.Key> segmentedQuotaKeys = quotaSegmentTypes.get(true);

        final Set<Quota> quotaWithSegments = getQuotasForUpdate(GET_QUOTAS_BY_SPEC_PROJECTS_WITH_SEGMENTS_QUERY_FOR_UPDATE, segmentedQuotaKeys)
                .stream()
                .filter(q -> segmentedQuotaKeys.contains(q.getKey()))
                .collect(Collectors.toSet());

        return Sets.union(quotaWithoutSegments, quotaWithSegments);
    }

    @Override
    public @NotNull Set<Quota> getProjectsQuotasForUpdate(final @NotNull Collection<Project> projects) {
        if (projects.isEmpty()) {
            return Collections.emptySet();
        }

        final Set<Resource> resourcesWithSegmentations = Hierarchy.get().getResourceSegmentationReader().getAll().stream()
                .map(ResourceSegmentation::getResource)
                .collect(Collectors.toSet());

        final Collection<QuotaSpec> specsWithSegments = Hierarchy.get().getQuotaSpecReader().getByResources(resourcesWithSegmentations).values();

        final Map<String, Object> params = ImmutableMap.of(
                "projectIds", ids(projects),
                "segmentQuotaSpecIds", ids(specsWithSegments)
        );

        final Stream<Quota> quotas;
        if (specsWithSegments.isEmpty()) {
            quotas = queryForSet(GET_QUOTAS_BY_PROJECTS_QUERY_FOR_UPDATE, params).stream();
        } else {
            quotas = Stream.concat(
                    queryForSet(GET_QUOTAS_BY_PROJECTS_WITH_SEGMENTS_QUERY_FOR_UPDATE, params).stream(),
                    queryForSet(GET_QUOTAS_BY_PROJECTS_NO_SEGMENTS_QUERY_FOR_UPDATE, params).stream()
            );
        }

        return quotas
                .filter(q -> q != Quota.FAKE)
                .collect(Collectors.toSet());
    }

    @NotNull
    @Override
    public Set<Quota> getQuotasForUpdate(final @NotNull QuotaSpec quotaSpec) {
        final Set<Segmentation> segmentations = Hierarchy.get().getResourceSegmentationReader().getSegmentations(quotaSpec.getResource());

        final Map<String, Object> params = ImmutableMap.of(
                "quotaSpecId", quotaSpec.getId()
        );

        final Stream<Quota> quotas;
        if (segmentations.isEmpty()) {
            quotas = queryForSet(GET_QUOTAS_BY_SPEC_NO_SEGMENTS_QUERY_FOR_UPDATE, params).stream();
        } else {
            quotas = queryForSet(GET_QUOTAS_BY_SPEC_WITH_SEGMENTS_QUERY_FOR_UPDATE, params).stream();
        }

        return quotas
                .filter(q -> q != Quota.FAKE)
                .collect(Collectors.toSet());
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<Quota> getQuotas(final @NotNull Resource resource, @NotNull final Collection<Project> projects, @NotNull final Set<Segment> segments) {
        if (projects.isEmpty()) {
            return Collections.emptySet();
        }
        final Map<String, Object> params = ImmutableMap.of("resourceId", resource.getId(), "projectIds", ids(projects));
        return queryForSet(GET_QUOTAS_BY_RESOURCE_PROJECTS_QUERY, params).stream()
                .filter(q -> q != Quota.FAKE)
                .filter(q -> q.getSegments().equals(segments)) //TODO sql select
                .collect(Collectors.toSet());
    }

    @Override
    public Page<Quota> readPage(final QuotaFilterParams quotaFilterParams, final PageInfo pageInfo) {

        final SelectQuery countQuery = createQueryWithoutColumns(quotaFilterParams);

        final boolean isMultipleSegments = quotaFilterParams.getSegments().size() > 1;
        final String field = isMultipleSegments ? "distinct q.id" : "q.id";

        countQuery.addCustomColumns(new CustomSql("count(" + field + ") as count"));
        final long count = jdbcTemplate.queryForObject(countQuery.toString(), Collections.emptyMap(), (r, e) -> r.getLong("count"));

        if (count == 0) {
            return Page.empty();
        }

        final SelectQuery idsQuery = createQueryWithoutColumns(quotaFilterParams);
        if (isMultipleSegments) {
            idsQuery.setIsDistinct(true);
        }
        idsQuery.addCustomColumns(QUOTA_TABLE.findColumn("id"));
        idsQuery.addCustomization(new PgLimitClause(pageInfo.getPageSize()));
        idsQuery.addCustomization(new PgOffsetClause(pageInfo.getOffset()));

        final String query = String.format(GET_ALL_QUOTAS_BY_ID_QUERY_TEMPLATE, idsQuery.toString());
        final List<Quota> quotas = queryForOrderedQuotas(query, Collections.emptyMap());

        return Page.of(quotas, count);
    }

    public static SelectQuery createIdQuery(final QuotaFilterParams quotaFilterParams) {
        final SelectQuery query = createQueryWithoutColumns(quotaFilterParams);
        query.addColumns(QUOTA_TABLE.findColumn("id"));
        return query;
    }

    @NotNull
    private static SelectQuery createQueryWithoutColumns(final QuotaFilterParams quotaFilterParams) {
        final SelectQuery selectQuery = new SelectQuery();
        selectQuery.addFromTable(QUOTA_TABLE);

        final Set<Project> projects = quotaFilterParams.getProjects();

        if (!projects.isEmpty()) {
            selectQuery.addCondition(new InCondition(QUOTA_TABLE.findColumn("project_id"), ids(projects)));
        }

        final DbTable projectTable = new DbTable(SCHEMA, "project", "p");
        final DbColumn projectId = projectTable.addColumn("id");
        final DbColumn personId = projectTable.addColumn("person_id");

        selectQuery.addCustomJoin(SelectQuery.JoinType.LEFT_OUTER, QUOTA_TABLE, projectTable,
                new BinaryCondition(BinaryCondition.Op.EQUAL_TO,
                        projectId,
                        QUOTA_TABLE.findColumn("project_id"))
        );

        selectQuery.addCondition(new UnaryCondition(UnaryCondition.Op.IS_NULL, personId));

        final Set<QuotaSpec> quotaSpec = quotaFilterParams.getQuotaSpecs();

        if (!quotaSpec.isEmpty()) {
            selectQuery.addCondition(new InCondition(QUOTA_TABLE.findColumn("quota_spec_id"), ids(quotaSpec)));
        }

        final Set<Segment> segments = quotaFilterParams.getSegments();

        if (segments.size() == 1) {

            final DbTable qs = new DbTable(SCHEMA, "quota_segment", "qs");
            final DbColumn segmentId = qs.addColumn("segment_id");
            final DbColumn quotaId = qs.addColumn("quota_id");

            final SelectQuery subQuery = new SelectQuery();
            subQuery.addFromTable(qs);
            subQuery.addColumns(quotaId);
            subQuery.addCondition(new BinaryCondition(BinaryCondition.Op.EQUAL_TO, segmentId, segments.iterator().next().getId()));

            selectQuery.addCondition(new InCondition(QUOTA_TABLE.findColumn("id"), subQuery));
        } else {

            final HashMap<Segmentation, List<Long>> segmentIdsBySegmentation = segments.stream()
                    .collect(Collectors.groupingBy(Segment::getSegmentation,
                            HashMap::new,
                            Collectors.mapping(LongIndexBase::getId, Collectors.toList())));

            int counter = 0;
            for (final Segmentation segmentation : segmentIdsBySegmentation.keySet()) {
                counter += 1;
                final DbTable qs = new DbTable(SCHEMA, "quota_segment", "qs" + counter);
                final DbColumn segmentId = qs.addColumn("segment_id");
                final DbColumn quotaId = qs.addColumn("quota_id");
                selectQuery.addCustomJoin(SelectQuery.JoinType.LEFT_OUTER, QUOTA_TABLE, qs,
                        new BinaryCondition(BinaryCondition.Op.EQUAL_TO,
                                quotaId,
                                QUOTA_TABLE.findColumn("id"))
                );
                selectQuery.addCondition(new InCondition(segmentId, segmentIdsBySegmentation.get(segmentation)));
            }

        }

        return selectQuery;
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Quota create(final @NotNull Quota quota) {
        createAll(Collections.singleton(quota));
        return getQuota(quota.getKey());
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public void createAll(@NotNull final Collection<Quota> quotas) {
        if (quotas.isEmpty()) {
            return;
        }
        final String[] sqls = quotas.stream()
                .sorted()
                .flatMap(q -> getInsertSql(q).stream())
                .toArray(String[]::new);

        jdbcTemplate.batchUpdate(sqls);
    }

    private Collection<String> getInsertSql(@NotNull final Quota quota) {
        final boolean hasSegments = !quota.getSegments().isEmpty();

        final String insertQuotaSql = "INSERT INTO quota (quota_spec_id, project_id, max_value, own_max_value, actual_value) values ("
                + quota.getSpec().getId()
                + ", "
                + quota.getProject().getId()
                + ", "
                + quota.getMax()
                + ", "
                + quota.getOwnMax()
                + ", "
                + quota.getOwnActual()
                + ") ON CONFLICT DO NOTHING";

        if (!hasSegments) {
            return Collections.singleton(insertQuotaSql);
        }

        final StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("INSERT INTO quota_segment (quota_id, segmentation_id, segment_id) VALUES (");

        final StringJoiner valueJoiner = new StringJoiner("), (");
        for (final Segment segment : quota.getSegments()) {
            valueJoiner.add(
                    "currval('quota_id_seq'), " + segment.getKey().getSegmentation().getId() +
                            ", " + (segment.isAggregationSegment() ? "NULL" : segment.getId())
            );
        }

        stringBuilder.append(valueJoiner).append(")");
        return Arrays.asList(insertQuotaSql, stringBuilder.toString());
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public @NotNull Quota read(@NotNull final Long id) throws EmptyResultDataAccessException {
        final Quota q = queryForObject(SELECT_BY_ID_QUERY, Collections.singletonMap("id", id));
        if (q == null) {
            throw new EmptyResultDataAccessException("no quota with id " + id, 1);
        }
        return q;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean update(final @NotNull Quota quota) {
        return jdbcTemplate.update(UPDATE_QUERY, toParams(quota)) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean delete(final @NotNull Quota quota) {
        return jdbcTemplate.update(REMOVE_QUERY, toParams(quota)) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean clear() {
        return jdbcTemplate.update(CLEAR_QUERY) > 0;
    }

    @Override
    public void createZeroQuotasFor(final @NotNull QuotaSpec quotaSpec) {
        final SqlProjectDao projectDao = (SqlProjectDao) getDirectProjectDao();
        projectDao.lockForChanges(); // no projects should be added or deleted during quota creation
        ((SqlResourceSegmentationDao) getResourceSegmentationDao()).lockForChanges();

        IntegratedQuotaDao.super.createZeroQuotasFor(quotaSpec);
    }

    @Override
    public void createZeroQuotasFor(final @NotNull Collection<Project> projects) {
        final SqlQuotaSpecDao quotaSpecDao = (SqlQuotaSpecDao) getDirectQuotaSpecDao();
        quotaSpecDao.lockForChanges(); // no quota specs should be added or deleted during quota creation
        ((SqlResourceSegmentationDao) getResourceSegmentationDao()).lockForChanges();

        IntegratedQuotaDao.super.createZeroQuotasFor(projects);
    }

    @Override
    public void updateQuotaSegments(final @NotNull Collection<Resource> resources) {
        ((SqlResourceSegmentationDao) getResourceSegmentationDao()).lockForChanges();

        IntegratedQuotaDao.super.updateQuotaSegments(resources);
    }

    @NotNull
    private SqlParameterSource toActualParams(@NotNull final Quota quota, final long actual) {
        return new MapSqlParameterSource("id", quota.getId())
                .addValue("ownActualValue", actual);
    }

    @NotNull
    private static Quota toQuota(@NotNull final ResultSet rs, @NotNull final Set<Segment> segments) throws SQLException {
        try {
            final QuotaSpec quotaSpec = Hierarchy.get().getQuotaSpecReader().read(rs.getLong("quota_spec_id"));
            final Project project = Hierarchy.get().getProjectReader().read(rs.getLong("project_id"));

            return Quota.builder(rs.getLong("id"), quotaSpec, project, segments)
                    .max(rs.getLong("max_value"))
                    .ownMax(rs.getLong("own_max_value"))
                    .ownActual(Math.max(rs.getLong("actual_value"), 0))
                    .lastOverquotingTs(SqlUtils.toTime(rs.getTimestamp("last_overquoting_ts"))) // TODO: ordering is important
                    .build();
        } catch (EmptyResultDataAccessException ignore) {
            return Quota.FAKE;
        }
    }

    @NotNull
    @Override
    public ProjectDao getDirectProjectDao() {
        return directProjectDao;
    }

    @Override
    public @NotNull QuotaSpecDao getDirectQuotaSpecDao() {
        return directQuotaSpecDao;
    }

    @NotNull
    @Override
    public ResourceSegmentationDao getResourceSegmentationDao() {
        return resourceSegmentationDao;
    }

    @NotNull
    @Override
    public SegmentDao getSegmentDao() {
        return segmentDao;
    }

    @NotNull
    private Set<Quota> queryForSet(@NotNull final String query) {
        return queryForSet(query, Collections.emptyMap(), rs -> true);
    }

    @NotNull
    private Set<Quota> queryForSet(@NotNull final String query, @NotNull final Map<String, ?> params) {
        return queryForSet(query, params, rs -> true);
    }

    @NotNull
    private Set<Quota> queryForSet(@NotNull final String query, @NotNull final Map<String, ?> params,
                                   @NotNull final ResultSetFilter filter) {
        final Set<Quota> quotas = new HashSet<>();

        final SetMultimap<Long, Segment> quotaSegmentById = HashMultimap.create();
        final Map<Long, Quota> quotaUniqResultSet = new HashMap<>();

        jdbcTemplate.query(query, params, rs -> {
            if (!filter.test(rs)) {
                return;
            }
            final long id = rs.getLong("id");
            rs.getLong("segmentation_id");
            final boolean hasSegments = !rs.wasNull();
            if (hasSegments) {
                final Segment segment = SqlQuotaSegmentDao.toSegment(rs, 0);
                quotaSegmentById.put(id, segment);
            }
            if (!quotaUniqResultSet.containsKey(id)) {
                final Quota quota = toQuota(rs, Collections.emptySet());
                if (!hasSegments) {
                    quotas.add(quota);
                } else {
                    quotaUniqResultSet.put(id, quota);
                }
            }
        });

        quotaUniqResultSet.forEach((id, quota) -> {
            quotas.add(Quota.builder(quota, quotaSegmentById.get(id)).build());
        });
        return quotas;
    }

    @NotNull
    private List<Quota> queryForOrderedQuotas(@NotNull final String query, @NotNull final Map<String, ?> params) {
        final SortedQuotaCollector collector = new SortedQuotaCollector();
        jdbcTemplate.query(query, params, collector::add);
        return collector.getResult();
    }

    @Nullable
    private Quota queryForObject(@NotNull final String query, @NotNull final Map<String, ?> params) {
        final Set<Quota> quotas = queryForSet(query, params);
        return quotas.isEmpty() ? null : quotas.iterator().next();
    }

    private interface ResultSetFilter {
        boolean test(ResultSet var1) throws SQLException;
    }

    private static class SortedQuotaCollector {
        private final Set<Segment> segments = new HashSet<>();
        private final List<Quota> result = new ArrayList<>();

        private Long quotaId = null;
        private Quota quota = null;

        public void add(final ResultSet rs) throws SQLException {
            final long id = rs.getLong("id");
            if (quotaId == null) {
                setQuota(id, rs);
            }
            if (quotaId != id) {
                finalizeQuota();
                setQuota(id, rs);
            }
            rs.getLong("segmentation_id");
            final boolean hasSegments = !rs.wasNull();
            if (hasSegments) {
                final Segment segment = SqlQuotaSegmentDao.toSegment(rs, 0);
                segments.add(segment);
            }
        }

        private void finalizeQuota() {
            if (quota != null) {
                if (segments.isEmpty()) {
                    result.add(quota);
                } else {
                    result.add(Quota.builder(quota, new HashSet<>(segments)).build());
                    segments.clear();
                }
            }
        }

        private void setQuota(final long id, final ResultSet rs) throws SQLException {
            quota = toQuota(rs, Collections.emptySet());
            quotaId = id;
        }

        public List<Quota> getResult() {
            finalizeQuota();
            return result;
        }
    }

    private void doQuotaUpdates(@NotNull final Map<Quota.Key, Long> max, @NotNull final Map<Quota.Key, Long> ownActual,
                                @NotNull final Map<Quota.Key, Long> ownMax, @NotNull final QuotaSegmentsClass quotaSegmentsClass,
                                @NotNull final QuotaUpdatesClass quotaUpdatesClass, @NotNull final Set<Quota.Key> keys) {
        final String query = prepareQuotaUpdateQuery(quotaSegmentsClass, quotaUpdatesClass);
        final List<SqlParameterSource> parameters = prepareQuotaUpdateParameters(max, ownActual, ownMax, quotaSegmentsClass, quotaUpdatesClass, keys);
        jdbcTemplate.batchUpdate(query, parameters.toArray(new MapSqlParameterSource[0]));
    }

    private List<SqlParameterSource> prepareQuotaUpdateParameters(@NotNull final Map<Quota.Key, Long> max, @NotNull final Map<Quota.Key, Long> ownActual,
                                                                  @NotNull final Map<Quota.Key, Long> ownMax, @NotNull final QuotaSegmentsClass quotaSegmentsClass,
                                                                  @NotNull final QuotaUpdatesClass quotaUpdatesClass, @NotNull final Set<Quota.Key> keys) {
        final List<SqlParameterSource> result = new ArrayList<>();
        keys.forEach(key -> {
            final MapSqlParameterSource parameters = new MapSqlParameterSource();
            addQuotaUpdateValuesParameters(max, ownActual, ownMax, quotaUpdatesClass, key, parameters);
            addQuotaUpdateSearchParameters(quotaSegmentsClass, key, parameters);
            result.add(parameters);
        });
        return result;
    }

    private void addQuotaUpdateSearchParameters(@NotNull final QuotaSegmentsClass quotaSegmentsClass, @NotNull final Quota.Key key,
                                                @NotNull final MapSqlParameterSource parameters) {
        parameters.addValue("project_id", key.getProject().getId());
        parameters.addValue("quota_spec_id", key.getSpec().getId());
        if (!quotaSegmentsClass.isSegmentsPresent()) {
            return;
        }
        final List<Segment> nonAggregationSegments = key.getSegments().stream().filter(s -> !s.isAggregationSegment()).collect(Collectors.toList());
        for (int i = 0; i < nonAggregationSegments.size(); i++) {
            parameters.addValue("non_agg_segment_id_" + i, nonAggregationSegments.get(i).getId());
            parameters.addValue("non_agg_segmentation_id_" + i, nonAggregationSegments.get(i).getSegmentation().getId());
        }
        final List<Segment> aggregationSegments = key.getSegments().stream().filter(Segment::isAggregationSegment).collect(Collectors.toList());
        for (int i = 0; i < aggregationSegments.size(); i++) {
            parameters.addValue("agg_segmentation_id_" + i, aggregationSegments.get(i).getSegmentation().getId());
        }
        parameters.addValue("segments_count", key.getSegments().size());
    }

    private void addQuotaUpdateValuesParameters(@NotNull final Map<Quota.Key, Long> max, @NotNull final Map<Quota.Key, Long> ownActual,
                                                @NotNull final Map<Quota.Key, Long> ownMax, @NotNull final QuotaUpdatesClass quotaUpdatesClass,
                                                @NotNull final Quota.Key key, @NotNull final MapSqlParameterSource parameters) {
        if (quotaUpdatesClass.isMaxPresent()) {
            parameters.addValue("max_value", max.get(key));
        }
        if (quotaUpdatesClass.isOwnActualPresent()) {
            parameters.addValue("actual_value", ownActual.get(key));
        }
        if (quotaUpdatesClass.isOwnMaxPresent()) {
            parameters.addValue("own_max_value", ownMax.get(key));
        }
    }

    private String prepareQuotaUpdateQuery(@NotNull final QuotaSegmentsClass quotaSegmentsClass,
                                           @NotNull final QuotaUpdatesClass quotaUpdatesClass) {
        final String quotaUpdateValues = getQuotaUpdateValues(quotaUpdatesClass);
        final String quotaSearchSubquery = getQuotaUpdateSearchSubquery(quotaSegmentsClass);
        return String.format(BASE_QUOTA_UPDATE_QUERY, quotaUpdateValues, quotaSearchSubquery);
    }

    private String getQuotaUpdateSearchSubquery(@NotNull final SqlQuotaDao.@NotNull QuotaSegmentsClass quotaSegmentsClass) {
        if (!quotaSegmentsClass.isSegmentsPresent()) {
            return BASE_QUOTA_UPDATE_SEARCH_SUBQUERY;
        }
        final StringBuilder quotaSearchConditions = new StringBuilder();
        for (int i = 0; i < quotaSegmentsClass.getNonAggregationSegmentsCount(); i++) {
            quotaSearchConditions.append("(qs.segment_id = :non_agg_segment_id_");
            quotaSearchConditions.append(i);
            quotaSearchConditions.append(" AND qs.segmentation_id = :non_agg_segmentation_id_");
            quotaSearchConditions.append(i);
            quotaSearchConditions.append(")");
            if (i < quotaSegmentsClass.getNonAggregationSegmentsCount() - 1) {
                quotaSearchConditions.append(" OR ");
            }
        }
        if (quotaSegmentsClass.getNonAggregationSegmentsCount() > 0 && quotaSegmentsClass.getAggregationSegmentsCount() > 0) {
            quotaSearchConditions.append(" OR ");
        }
        for (int i = 0; i < quotaSegmentsClass.getAggregationSegmentsCount(); i++) {
            quotaSearchConditions.append("(qs.segment_id IS NULL AND qs.segmentation_id = :agg_segmentation_id_");
            quotaSearchConditions.append(i);
            quotaSearchConditions.append(")");
            if (i < quotaSegmentsClass.getAggregationSegmentsCount() - 1) {
                quotaSearchConditions.append(" OR ");
            }
        }
        return String.format(BASE_SEGMENTED_QUOTA_UPDATE_SEARCH_SUBQUERY, quotaSearchConditions.toString());
    }

    @NotNull
    private String getQuotaUpdateValues(@NotNull final QuotaUpdatesClass quotaUpdatesClass) {
        final StringBuilder result = new StringBuilder();
        if (quotaUpdatesClass.isMaxPresent()) {
            result.append("max_value = :max_value");
        }
        if (quotaUpdatesClass.isOwnActualPresent()) {
            if (result.length() > 0) {
                result.append(", ");
            }
            result.append("actual_value = :actual_value");
        }
        if (quotaUpdatesClass.isOwnMaxPresent()) {
            if (result.length() > 0) {
                result.append(", ");
            }
            result.append("own_max_value = :own_max_value");
        }
        return result.toString();
    }

    private static class QuotaSegmentsClass {

        private final long aggregationSegmentsCount;
        private final long nonAggregationSegmentsCount;

        private QuotaSegmentsClass(@NotNull final Set<Segment> segments) {
            this.aggregationSegmentsCount = segments.stream().filter(Segment::isAggregationSegment).count();
            this.nonAggregationSegmentsCount = segments.stream().filter(s -> !s.isAggregationSegment()).count();
        }

        public long getAggregationSegmentsCount() {
            return aggregationSegmentsCount;
        }

        public long getNonAggregationSegmentsCount() {
            return nonAggregationSegmentsCount;
        }

        public boolean isSegmentsPresent() {
            return aggregationSegmentsCount > 0 || nonAggregationSegmentsCount > 0;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final QuotaSegmentsClass that = (QuotaSegmentsClass) o;
            return aggregationSegmentsCount == that.aggregationSegmentsCount
                    && nonAggregationSegmentsCount == that.nonAggregationSegmentsCount;
        }

        @Override
        public int hashCode() {
            return Objects.hash(aggregationSegmentsCount, nonAggregationSegmentsCount);
        }

    }

    private static class QuotaUpdatesClass {

        private static final QuotaUpdatesClass MAX = new QuotaUpdatesClass(true, false, false);
        private static final QuotaUpdatesClass OWN_ACTUAL = new QuotaUpdatesClass(false, true, false);
        private static final QuotaUpdatesClass OWN_MAX = new QuotaUpdatesClass(false, false, true);

        private final boolean maxPresent;
        private final boolean ownActualPresent;
        private final boolean ownMaxPresent;

        private QuotaUpdatesClass(@NotNull final Quota.Key key, @NotNull final Map<Quota.Key, Long> max,
                                  @NotNull final Map<Quota.Key, Long> ownActual, @NotNull final Map<Quota.Key, Long> ownMax) {
            this.maxPresent = max.containsKey(key);
            this.ownActualPresent = ownActual.containsKey(key);
            this.ownMaxPresent = ownMax.containsKey(key);
        }

        private QuotaUpdatesClass(final boolean maxPresent, final boolean ownActualPresent, final boolean ownMaxPresent) {
            this.maxPresent = maxPresent;
            this.ownActualPresent = ownActualPresent;
            this.ownMaxPresent = ownMaxPresent;
        }

        public boolean isMaxPresent() {
            return maxPresent;
        }

        public boolean isOwnActualPresent() {
            return ownActualPresent;
        }

        public boolean isOwnMaxPresent() {
            return ownMaxPresent;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final QuotaUpdatesClass that = (QuotaUpdatesClass) o;
            return maxPresent == that.maxPresent
                    && ownActualPresent == that.ownActualPresent
                    && ownMaxPresent == that.ownMaxPresent;
        }

        @Override
        public int hashCode() {
            return Objects.hash(maxPresent, ownActualPresent, ownMaxPresent);
        }

    }

}
