package ru.yandex.qe.dispenser.domain.dao.quota.request;

import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;

public final class QuotaChangeInRequest {

    private final long id;
    private final long requestId;
    @NotNull
    private final Resource resource;
    @NotNull
    private final Project project;
    @NotNull
    private final Set<Segment> segments;
    private final long amount;
    private final long amountReady;
    private final long amountAllocated;
    private final long amountAllocating;

    private QuotaChangeInRequest(final long id,
                                 final long requestId,
                                 @NotNull final Resource resource,
                                 @NotNull final Project project,
                                 @NotNull final Set<Segment> segments,
                                 final long amount,
                                 final long amountReady,
                                 final long amountAllocated,
                                 final long amountAllocating) {
        this.id = id;
        this.requestId = requestId;
        this.resource = resource;
        this.project = project;
        this.segments = segments;
        this.amount = amount;
        this.amountReady = amountReady;
        this.amountAllocated = amountAllocated;
        this.amountAllocating = amountAllocating;
    }

    public static Builder builder() {
        return new Builder();
    }

    public long getId() {
        return id;
    }

    public long getRequestId() {
        return requestId;
    }

    @NotNull
    public Resource getResource() {
        return resource;
    }

    @NotNull
    public Project getProject() {
        return project;
    }

    @NotNull
    public Set<Segment> getSegments() {
        return segments;
    }

    public long getAmount() {
        return amount;
    }

    public long getAmountReady() {
        return amountReady;
    }

    public long getAmountAllocated() {
        return amountAllocated;
    }

    public long getAmountAllocating() {
        return amountAllocating;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final QuotaChangeInRequest that = (QuotaChangeInRequest) o;
        return id == that.id &&
                requestId == that.requestId &&
                amount == that.amount &&
                amountReady == that.amountReady &&
                amountAllocated == that.amountAllocated &&
                amountAllocating == that.amountAllocating &&
                resource.equals(that.resource) &&
                project.equals(that.project) &&
                segments.equals(that.segments);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, requestId, resource, project, segments, amount, amountReady, amountAllocated, amountAllocating);
    }

    @Override
    public String toString() {
        return "QuotaChangeInRequest{" +
                "id=" + id +
                ", requestId=" + requestId +
                ", resource=" + resource +
                ", project=" + project +
                ", segments=" + segments +
                ", amount=" + amount +
                ", amountReady=" + amountReady +
                ", amountAllocated=" + amountAllocated +
                ", amountAllocating=" + amountAllocating +
                '}';
    }

    public static final class Builder {
        @Nullable
        private Long id;
        @Nullable
        private Long requestId;
        @Nullable
        private Resource resource;
        @Nullable
        private Project project;
        @NotNull
        private Set<Segment> segments = new HashSet<>();
        @Nullable
        private Long amount;
        @Nullable
        private Long amountReady;
        @Nullable
        private Long amountAllocated;
        @Nullable
        private Long amountAllocating;

        private Builder() {
        }

        @NotNull
        public Builder id(final long id) {
            this.id = id;
            return this;
        }

        @NotNull
        public Builder requestId(final long requestId) {
            this.requestId = requestId;
            return this;
        }

        @NotNull
        public Builder resource(@NotNull final Resource resource) {
            Objects.requireNonNull(resource, "Resource is required.");
            this.resource = resource;
            return this;
        }

        @NotNull
        public Builder project(final Project project) {
            Objects.requireNonNull(project, "Project is required.");
            this.project = project;
            return this;
        }

        @NotNull
        public Builder addSegment(@NotNull final Segment segment) {
            Objects.requireNonNull(segment, "Segment is required.");
            this.segments.add(segment);
            return this;
        }

        @NotNull
        public Builder segments(@NotNull final Set<Segment> segments) {
            Objects.requireNonNull(segments, "Segments are required.");
            this.segments = segments;
            return this;
        }

        @NotNull
        public Builder amount(final long amount) {
            this.amount = amount;
            return this;
        }

        @NotNull
        public Builder amountReady(final long amountReady) {
            this.amountReady = amountReady;
            return this;
        }

        @NotNull
        public Builder amountAllocated(final long amountAllocated) {
            this.amountAllocated = amountAllocated;
            return this;
        }

        @NotNull
        public Builder amountAllocating(final long amountAllocating) {
            this.amountAllocating = amountAllocating;
            return this;
        }

        @Nullable
        public Long getId() {
            return id;
        }

        @Nullable
        public Long getRequestId() {
            return requestId;
        }

        @Nullable
        public Resource getResource() {
            return resource;
        }

        @NotNull
        public Set<Segment> getSegments() {
            return segments;
        }

        @Nullable
        public Long getAmount() {
            return amount;
        }

        @Nullable
        public Long getAmountReady() {
            return amountReady;
        }

        @Nullable
        public Long getAmountAllocated() {
            return amountAllocated;
        }

        @Nullable
        public Long getAmountAllocating() {
            return amountAllocating;
        }

        @NotNull
        public QuotaChangeInRequest build() {
            Objects.requireNonNull(id, "Id is required.");
            Objects.requireNonNull(requestId, "Request id is required.");
            Objects.requireNonNull(resource, "Resource is required.");
            Objects.requireNonNull(project, "Project is required.");
            Objects.requireNonNull(amount, "Amount is required.");
            Objects.requireNonNull(amountReady, "Amount ready is required.");
            Objects.requireNonNull(amountAllocated, "Amount allocated is required.");
            Objects.requireNonNull(amountAllocating, "Amount allocating is required.");
            return new QuotaChangeInRequest(id, requestId, resource, project, segments,
                    amount, amountReady, amountAllocated, amountAllocating);
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Builder builder = (Builder) o;
            return Objects.equals(id, builder.id) &&
                    Objects.equals(requestId, builder.requestId) &&
                    Objects.equals(resource, builder.resource) &&
                    Objects.equals(project, builder.project) &&
                    segments.equals(builder.segments) &&
                    Objects.equals(amount, builder.amount) &&
                    Objects.equals(amountReady, builder.amountReady) &&
                    Objects.equals(amountAllocated, builder.amountAllocated) &&
                    Objects.equals(amountAllocating, builder.amountAllocating);
        }

        @Override
        public int hashCode() {
            return Objects.hash(id, requestId, resource, project, segments, amount, amountReady, amountAllocated, amountAllocating);
        }

        @Override
        public String toString() {
            return "Builder{" +
                    "id=" + id +
                    ", requestId=" + requestId +
                    ", resource=" + resource +
                    ", project=" + project +
                    ", segments=" + segments +
                    ", amount=" + amount +
                    ", amountReady=" + amountReady +
                    ", amountAllocated=" + amountAllocated +
                    ", amountAllocating=" + amountAllocating +
                    '}';
        }

    }

}
