package ru.yandex.qe.dispenser.domain.dao.quota.request;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;

import com.google.common.collect.Sets;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeRequestImportantFilter;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeRequestUnbalancedFilter;
import ru.yandex.qe.dispenser.api.v1.DiResourcePreorderReasonType;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.InMemoryLongKeyDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.bot.MappedPreOrder;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.MappedPreOrderDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.change.BotPreOrderChangeDao;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.request.ExtendedPreOrderRequest;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.request.PreOrderRequestDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.dao.goal.Goal;
import ru.yandex.qe.dispenser.domain.dao.goal.GoalDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.goal.OkrAncestors;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.util.Page;
import ru.yandex.qe.dispenser.domain.util.PageInfo;

@ParametersAreNonnullByDefault
public class QuotaChangeRequestDaoImpl extends InMemoryLongKeyDaoImpl<QuotaChangeRequest> implements QuotaChangeRequestDao {

    private final AtomicLong changeCounter = new AtomicLong();
    private final CampaignDao campaignDao;
    private final BotPreOrderChangeDao botPreOrderChangeDao;
    private final Map<Long, Long> requestIdByChangeId = new HashMap<>();

    private final PreOrderRequestDaoImpl preOrderRequestDao;
    private final MappedPreOrderDaoImpl preOrderDao;
    private final GoalDaoImpl goalDao;

    @Inject
    public QuotaChangeRequestDaoImpl(final CampaignDao campaignDao,
                                     final PreOrderRequestDaoImpl preOrderRequestDao,
                                     final BotPreOrderChangeDao botPreOrderChangeDao,
                                     final MappedPreOrderDaoImpl preOrderDao,
                                     final GoalDaoImpl goalDao) {
        this.campaignDao = campaignDao;
        this.preOrderRequestDao = preOrderRequestDao;
        this.botPreOrderChangeDao = botPreOrderChangeDao;
        this.preOrderDao = preOrderDao;
        this.goalDao = goalDao;
    }

    @NotNull
    @Override
    public QuotaChangeRequest read(final long id) throws EmptyResultDataAccessException {
        return read(Long.valueOf(id));
    }

    @NotNull
    @Override
    public Map<Long, QuotaChangeRequest> read(final Collection<Long> ids) {
        return ids.stream()
                .map(id2obj::get)
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(LongIndexBase::getId, Function.identity()));
    }

    @NotNull
    @Override
    public Optional<QuotaChangeRequest> findByTicketKey(final String ticketKey) {
        return filter(req -> Objects.equals(req.getTrackerIssueKey(), ticketKey))
                .findFirst();
    }

    @NotNull
    @Override
    public Map<String, QuotaChangeRequest> findByTicketKeys(final Collection<String> ticketKeys) {
        return filter(req -> ticketKeys.contains(req.getTrackerIssueKey()))
                .collect(Collectors.toMap(QuotaChangeRequest::getTrackerIssueKey, Function.identity()));
    }

    @NotNull
    @Override
    public Map<String, QuotaChangeRequest> findByTicketKeysForUpdate(final Collection<String> ticketKeys) {
        return findByTicketKeys(ticketKeys);
    }

    @Override
    @NotNull
    public List<QuotaChangeRequest> read(final QuotaChangeRequestFilter filter) {
        return readStream(filter)
                .collect(Collectors.toList());
    }

    @Override
    public @NotNull Page<QuotaChangeRequest> readPage(
            final QuotaChangeRequestFilter filter,
            final PageInfo pageInfo) {
        final List<QuotaChangeRequest> allItems = read(filter);
        final long count = allItems.size();
        final int offset = (int) pageInfo.getOffset();

        if (offset > allItems.size()) {
            return Page.of(Stream.empty(), count);
        }

        final int toIndex = Math.min(offset + (int) pageInfo.getPageSize(), allItems.size());
        final List<QuotaChangeRequest> items = allItems.subList(offset, toIndex);

        return Page.of(items, count);
    }

    @Override
    @NotNull
    public Map<Resource, Long> readAggregation(final QuotaChangeRequestFilter filter) {
        final Map<Resource, Long> result = new HashMap<>();
        readStream(filter)
                .flatMap(r -> r.getChanges().stream())
                .forEach(c -> result.put(c.getResource(), result.getOrDefault(c.getResource(), 0L) + c.getAmount()));

        return result;
    }

    @Override
    public @NotNull List<QuotaChangeRequest> readRequestsWithFilteredChanges(final QuotaChangeRequestFilter filter) {

        final List<Predicate<QuotaChangeRequest.Change>> predicates = new ArrayList<>(2);

        final Set<Service> services = filter.getServices();
        if (!services.isEmpty()) {
            predicates.add(c -> services.contains(c.getResource().getService()));
        }

        final Set<Long> orderIds = filter.getOrderIds();
        if (!orderIds.isEmpty()) {
            predicates.add(c -> c.getKey().getBigOrder() != null && orderIds.contains(c.getKey().getBigOrder().getId()));
        }

        Stream<QuotaChangeRequest> filteredRequestStream = readStream(filter);

        if (!predicates.isEmpty()) {

            final Predicate<QuotaChangeRequest.Change> changePredicate = predicates.stream()
                    .reduce(Predicate::and)
                    .get();

            filteredRequestStream = filteredRequestStream
                    .map(r -> r.copyBuilder()
                            .changes(r.getChanges().stream()
                                    .filter(changePredicate)
                                    .collect(Collectors.toList()))
                            .build());
        }

        return filteredRequestStream.collect(Collectors.toList());

    }

    @NotNull
    private Optional<Predicate<QuotaChangeRequest>> preparePredicate(final QuotaChangeRequestFilter filter) {
        final List<Predicate<QuotaChangeRequest>> predicates = new ArrayList<>(3);

        if (!filter.getProjects().isEmpty()) {
            predicates.add(request -> filter.getProjects().contains(request.getProject()));
        }

        final Set<Person> authors = filter.getAuthors();
        if (!authors.isEmpty()) {
            predicates.add(request -> authors.contains(request.getAuthor()));
        }

        final Set<QuotaChangeRequest.Status> statuses = filter.getStatus();
        if (!statuses.isEmpty()) {
            predicates.add(request -> statuses.contains(request.getStatus()));
        }

        final Set<QuotaChangeRequest.Type> types = filter.getType();
        if (!types.isEmpty()) {
            predicates.add(request -> types.contains(request.getType()));
        }

        final Set<Service> services = filter.getServices();
        if (!services.isEmpty()) {
            predicates.add(request -> request.getChanges().stream().anyMatch(c -> services.contains(c.getResource().getService())));
        }

        final Set<Long> orderIds = filter.getOrderIds();
        if (!orderIds.isEmpty()) {
            predicates.add(request -> request.getChanges().stream().filter(c -> c.getKey().getBigOrder() != null).anyMatch(c -> orderIds.contains(c.getKey().getBigOrder().getId())));
        }

        final Set<Long> campaignIds = filter.getCampaignIds();
        if (!campaignIds.isEmpty()) {
            predicates.add(request -> campaignIds.contains(request.getCampaignId()));
        }

        final Set<Long> campaignOrderIds = filter.getCampaignOrderIds();
        if (!campaignOrderIds.isEmpty()) {
            final List<Campaign.CampaignOrder> campaignOrders = campaignDao.getCampaignOrders(campaignOrderIds);
            if (campaignOrders.size() != campaignOrderIds.size()) {
                final Set<Long> actualCampaignOrderIds = campaignOrders.stream().map(Campaign.CampaignOrder::getId).collect(Collectors.toSet());
                final Set<Long> missingCampaignOrderIds = Sets.difference(campaignOrderIds, actualCampaignOrderIds);
                throw new IllegalArgumentException("Invalid campaign order ids: " + missingCampaignOrderIds.stream()
                        .map(Object::toString).collect(Collectors.joining(", ")));
            }
            predicates.add(request -> campaignOrders.stream()
                    .anyMatch(o -> Long.valueOf(o.getCampaignId()).equals(request.getCampaignId())
                            && request.getChanges().stream().anyMatch(c -> c.getKey().getBigOrder() != null && c.getKey().getBigOrder().getId() == o.getBigOrderId())));
        }
        final Set<Long> goalIds = filter.getGoalIds();
        if (!goalIds.isEmpty()) {
            predicates.add(request -> request.getGoal() != null && goalIds.contains(request.getGoal().getId()));
        }

        final Set<DiResourcePreorderReasonType> reasonTypes = filter.getReasonTypes();
        if (!reasonTypes.isEmpty()) {
            predicates.add(request -> reasonTypes.contains(request.getResourcePreorderReasonType()));
        }

        final Set<Long> excludedChangeRequestIds = filter.getExcludedChangeRequestIds();
        if (!excludedChangeRequestIds.isEmpty()) {
            predicates.add(request -> !excludedChangeRequestIds.contains(request.getId()));
        }

        final Optional<Boolean> withoutTicket = filter.withoutTicket();
        if (withoutTicket.isPresent()) {
            if (withoutTicket.get()) {
                predicates.add(request -> request.getTrackerIssueKey() == null);
            } else {
                predicates.add(request -> request.getTrackerIssueKey() != null);
            }
        }

        final Set<Long> changeRequestIds = filter.getChangeRequestIds();
        if (!changeRequestIds.isEmpty()) {
            predicates.add(request -> changeRequestIds.contains(request.getId()));
        }

        final Set<Long> preOrderIds = filter.getPreOrderIds();
        if (!preOrderIds.isEmpty()) {
            final Set<Long> requestIds = new HashSet<>(botPreOrderChangeDao.getRequestsIdsByPreOrderIds(preOrderIds).values());
            predicates.add(request -> requestIds.contains(request.getId()));
        }

        final Long owningCostGreaterOrEquals = filter.getOwningCostGreaterOrEquals();
        if (owningCostGreaterOrEquals != null) {
            predicates.add(request -> request.getRequestOwningCost() >= owningCostGreaterOrEquals);
        }

        final Long owningCostLessOrEquals = filter.getOwningCostLessOrEquals();
        if (owningCostLessOrEquals != null) {
            predicates.add(request -> request.getRequestOwningCost() <= owningCostLessOrEquals);
        }

        final DiQuotaChangeRequestImportantFilter importantFilter = filter.getImportantFilter();
        if (importantFilter != null) {
            switch (importantFilter) {
                case IMPORTANT:
                case NOT_IMPORTANT:
                    predicates.add(request -> request.isImportantRequest() ==
                            (importantFilter == DiQuotaChangeRequestImportantFilter.IMPORTANT));
                    break;
                case BOTH:
                    break;
                default:
                    throw new IllegalStateException("Unsupported importance filter!");
            }
        }

        final DiQuotaChangeRequestUnbalancedFilter unbalancedFilter = filter.getUnbalancedFilter();
        if (unbalancedFilter != null) {
            switch (unbalancedFilter) {
                case BALANCED:
                case UNBALANCED:
                    predicates.add(request -> request.isUnbalanced() ==
                            (unbalancedFilter == DiQuotaChangeRequestUnbalancedFilter.UNBALANCED));
                    break;
                case BOTH:
                    break;
                default:
                    throw new IllegalStateException("Unsupported unbalanced filter!");
            }
        }

        Set<Campaign.Type> campaignTypes = filter.getCampaignTypes();
        if (!campaignTypes.isEmpty()) {
            predicates.add(request -> request.getCampaign() != null &&
                    campaignTypes.contains(request.getCampaign().getType()));
        }

        prepareSummaryPredicate(filter, predicates);

        return predicates.stream()
                .reduce(Predicate::and);
    }

    private void prepareSummaryPredicate(final QuotaChangeRequestFilter filter, final List<Predicate<QuotaChangeRequest>> predicates) {
        // Does not match actual full text search in DB, just a stub implementation
        filter.getSummary().ifPresent(summary -> {
            final String trimmedQuery = summary.trim();
            if (trimmedQuery.isEmpty()) {
                return;
            }
            final List<String> queryWords = Arrays.asList(trimmedQuery.split("\\s+"));
            if (queryWords.isEmpty()) {
                return;
            }
            predicates.add(request -> {
                if (request.getSummary() == null) {
                    return false;
                }
                final boolean exactMatch = summary.equals(request.getSummary());
                final Set<String> summaryWords = new HashSet<>(Arrays.asList(request.getSummary().split("\\s+")));
                final boolean wordMatch = summaryWords.containsAll(queryWords);
                return exactMatch || wordMatch;
            });
        });
    }

    @NotNull
    @Override
    public List<QuotaChangeRequest.Change> setChanges(@NotNull final QuotaChangeRequest request, final List<QuotaChangeRequest.Change> resultChanges) {
        request.getChanges().forEach(c -> requestIdByChangeId.remove(c.getId()));

        update(request.copyBuilder()
                .changes(resultChanges.stream()
                        .peek(c -> {
                            if (c.getId() < 0) {
                                c.setId(changeCounter.incrementAndGet());
                            }
                            requestIdByChangeId.put(c.getId(), request.getId());
                        })
                        .collect(Collectors.toList()))
                .build());

        return resultChanges;
    }

    @NotNull
    @Override
    public Stream<QuotaChangeRequest> readStream(final QuotaChangeRequestFilter filter) {
        return preparePredicate(filter)
                .map(this::filter)
                .orElseGet(() -> getAll().stream())
                .sorted(getSortingComparator(filter)
                        .thenComparingLong(QuotaChangeRequest::getId));
    }

    private Comparator<QuotaChangeRequest> getSortingComparator(final QuotaChangeRequestFilter filter) {
        final Comparator<QuotaChangeRequest> quotaChangeRequestComparator;

        switch (filter.getSortBy()) {
            case UPDATED_AT:
                quotaChangeRequestComparator = Comparator.comparingLong(QuotaChangeRequest::getUpdated);
                break;
            case CREATED_AT:
                quotaChangeRequestComparator = Comparator.comparingLong(QuotaChangeRequest::getCreated);
                break;
            case COST:
                quotaChangeRequestComparator = Comparator.comparingDouble(QuotaChangeRequest::getCost);
                break;
            case REQUEST_OWNING_COST:
                quotaChangeRequestComparator = Comparator.comparingLong(QuotaChangeRequest::getRequestOwningCost);
                break;
            default:
                throw new IllegalArgumentException("Unsupported sorting column: " + filter.getSortBy());
        }
        return filter.getSortOrder() == SortOrder.DESC
                ? quotaChangeRequestComparator.reversed() :
                quotaChangeRequestComparator;
    }

    @Override
    public Set<Long> getBigOrderIdsForRequestsInStatuses(final Set<QuotaChangeRequest.Status> statuses) {
        return filter(r -> r.getType() == QuotaChangeRequest.Type.RESOURCE_PREORDER && statuses.contains(r.getStatus()))
                .flatMap(r -> r.getChanges().stream()
                        .filter(c -> c.getKey().getBigOrder() != null)
                        .map(c -> c.getKey().getBigOrder().getId()))
                .collect(Collectors.toSet());
    }

    @Override
    public void moveToProject(final Collection<Long> requestIds, final Project project) {
        read(requestIds).values()
                .forEach(request -> update(request.copyBuilder().updated(Instant.now().toEpochMilli()).project(project).build()));
    }

    @NotNull
    @Override
    public QuotaChangeRequest create(final @NotNull QuotaChangeRequest newInstance) {
        final List<QuotaChangeRequest.Change> changes = newInstance.getChanges();
        for (final QuotaChangeRequest.Change change : changes) {
            change.setId(changeCounter.incrementAndGet());
        }
        final QuotaChangeRequest quotaChangeRequest = super.create(newInstance);
        for (final QuotaChangeRequest.Change change : changes) {
            requestIdByChangeId.put(change.getId(), quotaChangeRequest.getId());
        }
        return quotaChangeRequest;
    }

    @Override
    public Set<Long> getRequestsIdsByChangeIds(final Set<Long> changeIds) {
        final Set<Long> requestIds = new HashSet<>();
        for (final QuotaChangeRequest quotaChangeRequest : getAll()) {
            for (final QuotaChangeRequest.Change change : quotaChangeRequest.getChanges()) {
                if (changeIds.contains(change.getId())) {
                    requestIds.add(quotaChangeRequest.getId());
                    break;
                }
            }
        }
        return requestIds;
    }

    @Override
    public boolean hasRequestsInCampaign(final long campaignId) {
        return filter(r -> Long.valueOf(campaignId).equals(r.getCampaignId())).findFirst().isPresent();
    }

    @Override
    public boolean hasRequestsInCampaignForOrdersOtherThan(final long campaignId, final Set<Long> orderIds) {
        return filter(r -> Long.valueOf(campaignId).equals(r.getCampaignId()))
                .anyMatch(r -> r.getChanges().stream()
                        .filter(c -> c.getKey().getBigOrder() != null)
                        .anyMatch(c -> !orderIds.contains(c.getKey().getBigOrder().getId())));
    }

    @Override
    @NotNull
    public List<RequestPreOrderAggregationEntry> readPreOrderAggregation(final QuotaChangeRequestFilter filter) {

        final Map<Triple<Long, Long, Long>, Pair<Double, Double>> aggregation = new HashMap<>();

        for (final QuotaChangeRequest request : readStream(filter).collect(Collectors.toList())) {
            final Collection<ExtendedPreOrderRequest> preorders = preOrderRequestDao.getRequestPreorders(request.getId());
            for (final ExtendedPreOrderRequest preorder : preorders) {
                final MappedPreOrder order = preOrderDao.read(preorder.getPreOrderId());
                final Triple<Long, Long, Long> key = Triple.of(request.getProject().getId(),
                        order.getService().getId(),
                        order.getServerId());

                final Pair<Double, Double> previous = aggregation.get(key);
                final Pair<Double, Double> value;
                if (previous != null) {
                    value = Pair.of(previous.getLeft() + preorder.getServersQuantity(),
                            previous.getRight() + preorder.getCost());
                } else {
                    value = Pair.of(preorder.getServersQuantity(),
                            preorder.getCost());
                }
                aggregation.put(key, value);
            }
        }

        return aggregation.entrySet().stream()
                .map(e -> new RequestPreOrderAggregationEntry(
                        e.getKey().getLeft(),
                        e.getKey().getMiddle(),
                        e.getKey().getRight(),
                        e.getValue().getLeft(),
                        e.getValue().getRight()
                ))
                .collect(Collectors.toList());
    }


    @Override
    public boolean update(final Collection<QuotaChangeRequest> values) {
        if (values.isEmpty()) {
            return false;
        }
        final List<Boolean> hasUpdates = values.stream()
                .map(this::update)
                .collect(Collectors.toList());

        return hasUpdates.stream().anyMatch(x -> x);
    }

    @Override
    @NotNull
    public Stream<Pair<QuotaChangeRequest, QuotaChangeRequest>> setStatuses(final QuotaChangeRequestFilter filter,
                                                                            final QuotaChangeRequest.Status status,
                                                                            final long updateTimeMillis) {
        final List<QuotaChangeRequest> requests = readStream(filter).collect(Collectors.toList());

        final List<Pair<QuotaChangeRequest, QuotaChangeRequest>> result = new ArrayList<>(requests.size());
        final List<QuotaChangeRequest> toUpdate = new ArrayList<>(requests.size());
        for (final QuotaChangeRequest request : requests) {
            final QuotaChangeRequest updatedRequest = request.copyBuilder()
                    .status(status)
                    .updated(updateTimeMillis)
                    .build();
            toUpdate.add(updatedRequest);
            result.add(Pair.of(updatedRequest, request));
        }

        update(toUpdate);

        return result.stream();
    }

    @NotNull
    @Override
    public Stream<ReportQuotaChangeRequest> readReport(@NotNull final QuotaChangeRequestFilter filter,
                                                       final boolean showGoalQuestions,
                                                       final boolean filterEmptyChanges,
                                                       final boolean showGoalHierarchy) {
        Stream<QuotaChangeRequest> stream = readRequestsWithFilteredChanges(filter).stream();
        if (filterEmptyChanges) {
            stream = stream.map(r -> r.copyBuilder()
                    .changes(r.getChanges().stream().filter(c -> c.getAmount() > 0).collect(Collectors.toList()))
                    .build()
            );
        }
        return stream
                .map(req -> {
                    final Map<Long, String> requestGoalAnswersMap;
                    if (showGoalQuestions) {
                        requestGoalAnswersMap = Optional.ofNullable(req.getRequestGoalAnswers())
                                .orElse(Collections.emptyMap());
                    } else {
                        requestGoalAnswersMap = Collections.emptyMap();
                    }

                    final Map<OkrAncestors.OkrType, ReportQuotaChangeRequest.Goal> goalHierarchy;
                    if (showGoalHierarchy && req.getGoal() != null) {
                        final OkrAncestors okrParents = req.getGoal().getOkrParents();
                        goalHierarchy = new EnumMap<>(OkrAncestors.OkrType.class);
                        for (final OkrAncestors.OkrType okrType : OkrAncestors.OkrType.values()) {
                            final Long goalId = okrParents.getGoalId(okrType);
                            if (goalId != null) {
                                final Goal goal = goalDao.read(goalId);
                                goalHierarchy.put(okrType, new ReportQuotaChangeRequest.Goal(goal.getId(), goal.getName()));
                            }
                        }
                    } else {
                        goalHierarchy = Collections.emptyMap();
                    }

                    return new ReportQuotaChangeRequest(req, requestGoalAnswersMap, goalHierarchy);
                });
    }

    @Override
    public void updateChanges(final Collection<QuotaChangeRequest.Change> changesToUpdate) {
        for (final QuotaChangeRequest.Change change : changesToUpdate) {
            final QuotaChangeRequest request = read(requestIdByChangeId.get(change.getId()));
            update(request.copyBuilder()
                    .changes(request.getChanges().stream()
                            .map(c -> c.getId() == change.getId() ? change : c)
                            .collect(Collectors.toList()))
                    .build());
        }
    }

    @Override
    public void updateChangesOwningCost(Collection<QuotaChangeRequest.Change> changesToUpdate) {
        for (final QuotaChangeRequest.Change change : changesToUpdate) {
            final QuotaChangeRequest request = read(requestIdByChangeId.get(change.getId()));
            update(request.copyBuilder()
                    .changes(request.getChanges().stream()
                            .map(c -> !c.getKey().equals(change.getKey()) ? c : c.copyBuilder()
                                    .owningCost(change.getOwningCost()).build())
                            .collect(Collectors.toList()))
                    .build());
        }
    }

    @Override
    public void updateRequestsOwningCost(Map<Long, Long> requestOwningCostMap) {
        requestOwningCostMap.forEach((id, requestOwningCost) -> {
            QuotaChangeRequest request = read(id);
            update(request.copyBuilder()
                    .requestOwningCost(requestOwningCost)
                    .build());
        });
    }

    @Override
    public void updateUnbalanced(Map<Long, Boolean> unbalancedByRequestId) {
        unbalancedByRequestId.forEach((id, unbalanced) -> {
            QuotaChangeRequest request = read(id);
            update(request.copyBuilder()
                    .unbalanced(unbalanced)
                    .build());
        });
    }

    @Override
    public void setReadyForAllocationState(final Collection<Long> requestIds, final boolean isReadyForAllocation) {
        final List<QuotaChangeRequest> updatedRequests = read(requestIds).values()
                .stream()
                .map(r -> r.copyBuilder()
                        .readyForAllocation(isReadyForAllocation)
                        .updated(System.currentTimeMillis())
                        .build())
                .collect(Collectors.toList());

        update(updatedRequests);
    }

    @Override
    public Map<Long, QuotaChangeRequest> readForUpdate(final Collection<Long> ids) {
        return read(ids);
    }

    @Override
    public void setChangesReadyAmount(final Map<Long, Long> readyByChangeId) {
        for (final Long changeId : readyByChangeId.keySet()) {
            final QuotaChangeRequest request = read(requestIdByChangeId.get(changeId));
            update(request.copyBuilder()
                    .changes(request.getChanges().stream()
                            .map(c -> c.getId() == changeId ? c.copyBuilder()
                                    .amountReady(readyByChangeId.get(changeId))
                                    .build() : c)
                            .collect(Collectors.toList()))
                    .build());
        }
    }

    @Override
    public void setChangesAllocatedAmount(final Map<Long, Long> allocatedByChangeId) {
        for (final Long changeId : allocatedByChangeId.keySet()) {
            final QuotaChangeRequest request = read(requestIdByChangeId.get(changeId));
            update(request.copyBuilder()
                    .changes(request.getChanges().stream()
                            .map(c -> c.getId() == changeId ? c.copyBuilder()
                                    .amountAllocated(allocatedByChangeId.get(changeId))
                                    .build() : c)
                            .collect(Collectors.toList()))
                    .build());
        }
    }

    @Override
    public void setChangesAllocatingAmount(final Map<Long, Long> allocatingByChangeId) {
        for (final Long changeId : allocatingByChangeId.keySet()) {
            final QuotaChangeRequest request = read(requestIdByChangeId.get(changeId));
            update(request.copyBuilder()
                    .changes(request.getChanges().stream()
                            .map(c -> c.getId() == changeId ? c.copyBuilder()
                                    .amountAllocating(allocatingByChangeId.get(changeId))
                                    .build() : c)
                            .collect(Collectors.toList()))
                    .build());
        }
    }

    @Override
    public void incrementChangesReadyAmount(@NotNull final Map<Long, Long> readyIncrementByChangeId) {
        for (final Long changeId : readyIncrementByChangeId.keySet()) {
            final QuotaChangeRequest request = read(requestIdByChangeId.get(changeId));
            update(request.copyBuilder()
                    .changes(request.getChanges().stream()
                            .map(c -> c.getId() == changeId ? c.copyBuilder()
                                    .amountReady(c.getAmountReady() + readyIncrementByChangeId.get(changeId))
                                    .build() : c)
                            .collect(Collectors.toList()))
                    .build());
        }
    }

    @Override
    public void incrementChangesAllocatedAmount(@NotNull final Map<Long, Long> allocatedIncrementByChangeId) {
        for (final Long changeId : allocatedIncrementByChangeId.keySet()) {
            final QuotaChangeRequest request = read(requestIdByChangeId.get(changeId));
            update(request.copyBuilder()
                    .changes(request.getChanges().stream()
                            .map(c -> c.getId() == changeId ? c.copyBuilder()
                                    .amountAllocated(c.getAmountAllocated() + allocatedIncrementByChangeId.get(changeId))
                                    .amountAllocating(c.getAmountAllocated() + allocatedIncrementByChangeId.get(changeId))
                                    .build() : c)
                            .collect(Collectors.toList()))
                    .build());
        }
    }

    @Override
    public void incrementChangesReadyAndAllocatedAmount(@NotNull final Map<Long, Pair<Long, Long>> readyAndAllocatedIncrementByChangeId) {
        for (final Long changeId : readyAndAllocatedIncrementByChangeId.keySet()) {
            final QuotaChangeRequest request = read(requestIdByChangeId.get(changeId));
            update(request.copyBuilder()
                    .changes(request.getChanges().stream()
                            .map(c -> c.getId() == changeId ? c.copyBuilder()
                                    .amountReady(c.getAmountReady() + readyAndAllocatedIncrementByChangeId.get(changeId).getLeft())
                                    .amountAllocated(c.getAmountAllocated() + readyAndAllocatedIncrementByChangeId.get(changeId).getRight())
                                    .amountAllocating(c.getAmountAllocated() + readyAndAllocatedIncrementByChangeId.get(changeId).getRight())
                                    .build() : c)
                            .collect(Collectors.toList()))
                    .build());
        }
    }

    @NotNull
    @Override
    public List<QuotaChangeInRequest> selectSegmentedChangesForQuotaDistributionUpdate(
            final long campaignId, final long bigOrderId, final long serviceId,
            @NotNull final QuotaChangeRequest.Type type,
            @NotNull final Set<QuotaChangeRequest.Status> statuses,
            @NotNull final Set<ResourceSegments> resourceSegments, final boolean lock) {
        // WARNING! Slow implementation, never use in production
        return filter(r -> Objects.equals(r.getCampaignId(), campaignId)
                && r.getType() == type
                && statuses.contains(r.getStatus()))
                .flatMap(r -> r.getChanges().stream()
                        .filter(c -> c.getResource().getService().getId() == serviceId && Objects.equals(c.getKey().getBigOrder().getId(), bigOrderId))
                        .map(c -> QuotaChangeInRequest.builder()
                                .id(c.getId())
                                .requestId(r.getId())
                                .resource(c.getResource())
                                .project(r.getProject())
                                .segments(c.getSegments())
                                .amount(c.getAmount())
                                .amountReady(c.getAmountReady())
                                .amountAllocated(c.getAmountAllocated())
                                .amountAllocating(c.getAmountAllocating())
                                .build()))
                .filter(c -> resourceSegments.contains(new ResourceSegments(c.getResource().getId(),
                        c.getSegments().stream().map(LongIndexBase::getId).collect(Collectors.toSet())))
                        && c.getAmount() > c.getAmountReady())
                .sorted(Comparator.comparing(QuotaChangeInRequest::getId))
                .collect(Collectors.toList());
    }

    @NotNull
    @Override
    public List<QuotaChangeInRequest> selectNonSegmentedChangesForQuotaDistributionUpdate(
            final long campaignId, final long bigOrderId, final long serviceId,
            @NotNull final QuotaChangeRequest.Type type,
            @NotNull final Set<QuotaChangeRequest.Status> statuses,
            @NotNull final Set<Long> resourceIds, final boolean lock) {
        // WARNING! Slow implementation, never use in production
        return filter(r -> Objects.equals(r.getCampaignId(), campaignId)
                && r.getType() == type
                && statuses.contains(r.getStatus()))
                .flatMap(r -> r.getChanges().stream()
                        .filter(c -> c.getResource().getService().getId() == serviceId && Objects.equals(c.getKey().getBigOrder().getId(), bigOrderId))
                        .map(c -> QuotaChangeInRequest.builder()
                                .id(c.getId())
                                .requestId(r.getId())
                                .resource(c.getResource())
                                .project(r.getProject())
                                .segments(c.getSegments())
                                .amount(c.getAmount())
                                .amountReady(c.getAmountReady())
                                .amountAllocated(c.getAmountAllocated())
                                .amountAllocating(c.getAmountAllocating())
                                .build()))
                .filter(c -> resourceIds.contains(c.getResource().getId()) && c.getAmount() > c.getAmountReady())
                .sorted(Comparator.comparing(QuotaChangeInRequest::getId))
                .collect(Collectors.toList());
    }

    @NotNull
    @Override
    public Map<Long, List<QuotaChangeRequest.Change>> selectChangesByRequestIds(@NotNull final List<Long> requestIds) {
        // WARNING! Slow implementation, never use in production
        return filter(r -> requestIds.contains(r.getId()))
                .collect(Collectors.toMap(LongIndexBase::getId, QuotaChangeRequest::getChanges));
    }

    @NotNull
    @Override
    public Map<Long, Pair<QuotaChangeRequest.Status, String>> selectStatusAndIssueByRequestIds(@NotNull final List<Long> requestIds) {
        // WARNING! Slow implementation, never use in production
        return filter(r -> requestIds.contains(r.getId()))
                .collect(Collectors.toMap(LongIndexBase::getId, r -> Pair.of(r.getStatus(), r.getTrackerIssueKey())));
    }

    @Override
    public void updateRequestCost(final Set<Long> requestIds) {
        updateAll(preOrderRequestDao.getRequestTotalSumById(requestIds).entrySet()
                .stream()
                .map(e -> id2obj.get(e.getKey()).copyBuilder().cost(e.getValue()).build())
                .collect(Collectors.toList())
        );

    }

    @Override
    public List<QuotaChangeRequest.Change> readChangesByRequestAndProviderForUpdate(long requestId, long serviceId) {
        final QuotaChangeRequest quotaChangeRequest = read(requestId);
        return quotaChangeRequest.getChanges()
                .stream()
                .filter(c -> c.getResource().getService().getId() == serviceId)
                .collect(Collectors.toList());
    }

    @NotNull
    @Override
    public List<QuotaChangeRequest> readRequestsByCampaignsForUpdate(Collection<Long> campaignIds,
                                                                     @Nullable Long fromId, long limit) {
        if (campaignIds.isEmpty()) {
            return List.of();
        }

        return filter(v -> v.getCampaignId() != null && campaignIds.contains(v.getCampaignId())
                && (fromId == null || v.getId() > fromId))
                .sorted(Comparator.comparing(LongIndexBase::getId)).limit(limit)
                .collect(Collectors.toList());
    }

    @Override
    public List<Tuple2<Long, Long>> readRequestOwningCostByCampaignId(@Nullable Long fromId, Long campaignId, long limit,
                                                                      Set<QuotaChangeRequest.Status> validStatuses) {
        if (validStatuses.isEmpty()) {
            return List.of();
        }

        return filter(v -> v.getCampaignId() != null && campaignId.equals(v.getCampaignId())
                && (fromId == null || v.getId() > fromId) && validStatuses.contains(v.getStatus())
                && v.getRequestOwningCost() > 0)
                .sorted(Comparator.comparing(LongIndexBase::getId)).limit(limit)
                .map(qcr -> Tuple2.tuple(qcr.getId(), qcr.getRequestOwningCost()))
                .collect(Collectors.toList());
    }

    @Override
    public List<Long> getIdsFirstPageByCampaigns(Collection<? extends Long> campaignIds, long limit) {
        if (campaignIds.isEmpty()) {
            return List.of();
        }
        return filter(v -> v.getCampaignId() != null && campaignIds.contains(v.getCampaignId()))
                .sorted(Comparator.comparing(LongIndexBase::getId)).limit(limit).map(LongIndexBase::getId)
                .collect(Collectors.toList());
    }

    @Override
    public List<Long> getIdsNextPageByCampaigns(Collection<? extends Long> campaignIds, long from, long limit) {
        if (campaignIds.isEmpty()) {
            return List.of();
        }
        return filter(v -> v.getCampaignId() != null && campaignIds.contains(v.getCampaignId()) && v.getId() > from)
                .sorted(Comparator.comparing(LongIndexBase::getId)).limit(limit).map(LongIndexBase::getId)
                .collect(Collectors.toList());
    }

    @Override
    public void setShowAllocationNote(Collection<Long> requestIds, boolean showAllocationNote) {
        final List<QuotaChangeRequest> updated = read(requestIds).values()
                .stream()
                .map(r -> r.copyBuilder()
                        .showAllocationNote(showAllocationNote)
                        .build())
                .collect(Collectors.toList());
        update(updated);
    }
}
