package ru.yandex.qe.dispenser.domain.dao.quota.request;

import java.math.BigDecimal;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.SetMultimap;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeRequestImportantFilter;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeRequestUnbalancedFilter;
import ru.yandex.qe.dispenser.api.v1.DiResourcePreorderReasonType;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.QuotaRequestChangeBuilder;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.dao.SqlUtils;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.collector.DefaultRequestCollector;
import ru.yandex.qe.dispenser.domain.dao.quota.request.collector.ReportRequestCollector;
import ru.yandex.qe.dispenser.domain.dao.quota.request.collector.RequestCollector;
import ru.yandex.qe.dispenser.domain.dao.quota.request.collector.StatusUpdatedRequestCollector;
import ru.yandex.qe.dispenser.domain.dao.resource.ResourceReader;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentReader;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.util.MapBuilder;
import ru.yandex.qe.dispenser.domain.util.Page;
import ru.yandex.qe.dispenser.domain.util.PageInfo;

import static ru.yandex.qe.dispenser.domain.util.CollectionUtils.ids;

@ParametersAreNonnullByDefault
public class SqlQuotaChangeRequestDao extends SqlDaoBase implements QuotaChangeRequestDao {

    private static final String REQUEST_CHANGE_SEGMENT_FROM_CLAUSE = "FROM quota_request JOIN quota_request_change ON quota_request.id = quota_request_change.quota_request_id LEFT JOIN quota_request_change_segment ON quota_request_change_segment.quota_request_change_id = quota_request_change.id";

    private static final String FROM_CLAUSE = REQUEST_CHANGE_SEGMENT_FROM_CLAUSE
            + " LEFT JOIN campaign ON quota_request.campaign_id = campaign.id LEFT JOIN bot_big_order ON quota_request_change.order_id = bot_big_order.id LEFT JOIN campaign_big_order ON quota_request.campaign_id = campaign_big_order.campaign_id AND quota_request_change.order_id = campaign_big_order.big_order_id LEFT JOIN goal ON quota_request.goal_id = goal.id";

    public static final String SELECT_ALL_FIELDS_CLAUSE = "SELECT quota_request.*, quota_request_change.id as quota_request_change_id, quota_request_change.resource_id, quota_request_change.order_id, quota_request_change.amount, quota_request_change.amount_ready, quota_request_change.amount_allocated, quota_request_change.amount_allocating, quota_request_change.owning_cost, quota_request_change_segment.segment_id, bot_big_order.date as bot_big_order_date, campaign.key as campaign_key, campaign.name as campaign_name, campaign.status as campaign_status, campaign.request_modification_disabled_for_non_managers as campaign_request_modification_disabled_for_non_managers, campaign.deleted as campaign_deleted, campaign.allowed_request_modification_when_closed as campaign_allowed_request_modification_when_closed, campaign.allowed_modification_on_missing_additional_fields as campaign_allowed_modification_on_missing_additional_fields, campaign.forced_allowing_request_status_change as campaign_forced_allowing_request_status_change, campaign.single_provider_request_mode as campaign_single_provider_request_mode, campaign.allowed_request_creation_for_provider_admin as campaign_allowed_request_creation_for_provider_admin, campaign.request_creation_disabled as campaign_request_creation_disabled, campaign.allowed_request_creation_for_capacity_planner as allowed_request_creation_for_capacity_planner, campaign.campaign_type as campaign_campaign_type, campaign.request_modification_disabled as campaign_request_modification_disabled, campaign_big_order.deleted as campaign_big_order_deleted, goal.name as goal_name, goal.name as goal_name, goal.status as goal_status, goal.importance as goal_importance, goal.value_stream_goal_id as goal_value_stream_goal_id, goal.umbrella_goal_id as goal_umbrella_goal_id, goal.contour_goal_id as goal_contour_goal_id ";

    public static final String GET_ALL_QUERY = SELECT_ALL_FIELDS_CLAUSE + FROM_CLAUSE;

    public static final String GET_AGGREGATION_QUERY_TEMPLATE =
            "SELECT quota_request_change.resource_id, sum(quota_request_change.amount) as amount "
                    + REQUEST_CHANGE_SEGMENT_FROM_CLAUSE
                    + " %s  GROUP BY quota_request_change.resource_id";

    public static final String GET_IDS = "SELECT quota_request.id FROM quota_request";
    public static final String COUNT_ALL_QUERY = "SELECT COUNT(*) FROM quota_request";
    public static final String GET_BY_ID_QUERY = GET_ALL_QUERY + " WHERE quota_request.id IN (:quotaRequestId)";
    public static final String GET_BY_ID_FOR_UPDATE_QUERY = GET_BY_ID_QUERY + " FOR UPDATE OF quota_request";
    public static final String GET_IDS_FIRST_PAGE_BY_CAMPAIGNS = "SELECT id FROM quota_request WHERE campaign_id IN (:campaignIds) ORDER BY id ASC LIMIT :limit";
    public static final String GET_IDS_NEXT_PAGE_BY_CAMPAIGNS = "SELECT id FROM quota_request WHERE campaign_id IN (:campaignIds) AND id > :fromId ORDER BY id ASC LIMIT :limit";
    public static final String GET_BY_TICKET_KEY_QUERY = GET_ALL_QUERY + " WHERE quota_request.ticket_key = :ticketKey";
    public static final String GET_BY_TICKET_KEYS_QUERY = GET_ALL_QUERY + " WHERE quota_request.ticket_key IN (:ticketKeys)";
    public static final String GET_BY_TICKET_KEYS_QUERY_FOR_UPDATE = GET_BY_TICKET_KEYS_QUERY + " FOR UPDATE OF quota_request";

    public static final String GET_ORDER_IDS_BY_STATUSES_TEMPLATE = "SELECT DISTINCT qrc.order_id FROM quota_request qr JOIN quota_request_change qrc ON qrc.quota_request_id = qr.id WHERE qr.status IN (%s)";

    public static final String BASE_INSERT_QUERY =
            "INSERT INTO quota_request (author_id, description, comment, calculations, status, ticket_key, created, updated, project_id, type, source_project_id, chart_links, chart_links_absence_explanation,additional_properties, campaign_id, resource_preorder_reason_type, goal_id, ready_for_allocation, cost, summary, request_owning_cost, important_request, unbalanced, campaign_type)"
                    + " VALUES (:authorId, :description, :comment, :calculations, cast(:status as quota_request_status), :trackerIssueKey, :created, :updated, :projectId, cast(:type as quota_request_type), :sourceProjectId, :chartLinks, :chartLinksAbsenceExplanation, :additionalProperties, :campaign_id, cast(:resourcePreorderReasonType as resource_preorder_reason_type), :goalId, :readyForAllocation, :cost, :summary, :requestOwningCost, :importantRequest, :unbalanced, cast(:campaign_type as campaign_type))";

    public static final String CHANGE_INSERT_QUERY = "INSERT INTO quota_request_change (quota_request_id, resource_id, order_id,  amount, amount_ready, amount_allocated, amount_allocating, owning_cost) VALUES (:quotaRequestId, :resourceId, :orderId, :amount, :amountReady, :amountAllocated, :amountAllocating, :owningCost)";
    public static final String CHANGE_DELETE_QUERY = "DELETE FROM quota_request_change WHERE id IN (:quotaRequestChangeId)";
    public static final String CHANGE_UPDATE_QUERY = "UPDATE quota_request_change SET amount = :amount, amount_ready = :amountReady, amount_allocated = :amountAllocated, amount_allocating = :amountAllocating, owning_cost = :owningCost WHERE id = :quotaRequestChangeId";

    public static final String CHANGE_OWNING_COST = "UPDATE quota_request_change AS qrc SET " +
            "    owning_cost = c.owning_cost " +
            "FROM (values " +
            "       :values" +
            "     ) as c(id, owning_cost) " +
            "WHERE c.id = qrc.id";

    public static final String CHANGE_SEGMENT_INSERT_QUERY = "INSERT INTO quota_request_change_segment (quota_request_change_id, segment_id) VALUES (:quotaRequestChangeId, :segmentId)";

    public static final String UPDATE_QUERY =
            "UPDATE quota_request SET author_id = :authorId, description = :description, comment = :comment, calculations = :calculations, status = cast(:status as quota_request_status), ticket_key = :trackerIssueKey, created = :created, updated = :updated, chart_links = :chartLinks, chart_links_absence_explanation = :chartLinksAbsenceExplanation, additional_properties = :additionalProperties, resource_preorder_reason_type = cast(:resourcePreorderReasonType as resource_preorder_reason_type), goal_id = :goalId, ready_for_allocation = :readyForAllocation, request_goal_answers = :requestGoalAnswers, cost = :cost, summary = :summary, project_id = :projectId, request_owning_cost = :requestOwningCost, important_request = :importantRequest, unbalanced = :unbalanced WHERE id = :quotaRequestId";

    public static final String UPDATE_AND_RETURN_QUERY_TEMPLATE =
            "WITH quota_request AS (UPDATE quota_request SET status = cast(:newStatus as quota_request_status), updated = :newUpdated FROM quota_request qr %s AND qr.id = quota_request.id RETURNING quota_request.*, qr.status as old_status, qr.updated as old_updated) "
                    + GET_ALL_QUERY;

    public static final String DELETE_QUERY = "DELETE FROM quota_request WHERE id = :quotaRequestId";

    public static final String CLEAR_QUERY = "DELETE FROM quota_request";

    public static final String CHART_LINKS_DELIMITER = "||";
    private static final String MOVE_TO_PROJECT_QUERY = "UPDATE quota_request SET project_id = :project, updated = :updated WHERE id IN(:ids)";

    private static final String GET_REQUEST_IDS_BY_CHANGE_ID = "SELECT qrc.quota_request_id FROM quota_request_change qrc WHERE qrc.id IN (:changeIds)";
    private static final String EXISTS_BY_CAMPAIGN_ID = "SELECT EXISTS(SELECT 1 FROM quota_request WHERE campaign_id = :campaign_id) AS exists";
    private static final String EXISTS_BY_CAMPAIGN_ID_AND_ORDER_ID_NOT_IN = "SELECT EXISTS(SELECT 1 FROM quota_request qr JOIN quota_request_change qrc ON qr.id = qrc.quota_request_id WHERE qr.campaign_id = :campaign_id AND qrc.order_id NOT IN (:order_ids)) as exists";

    private static final String CHANGE_SET_ALLOCATED_QUERY = "UPDATE quota_request_change SET amount_allocated = :amount WHERE id = :quotaRequestChangeId";
    private static final String CHANGE_SET_ALLOCATING_QUERY = "UPDATE quota_request_change SET amount_allocating = :amount WHERE id = :quotaRequestChangeId";

    private static final String CHANGE_SET_READY_QUERY = "UPDATE quota_request_change SET amount_ready = :amount WHERE id = :quotaRequestChangeId";

    private static final String CHANGE_INCREMENT_ALLOCATED_QUERY = "UPDATE quota_request_change SET " +
            "amount_allocating = amount_allocated + :amount_allocated, amount_allocated = amount_allocated + :amount_allocated WHERE id = :quotaRequestChangeId";

    private static final String CHANGE_INCREMENT_READY_QUERY = "UPDATE quota_request_change SET " +
            "amount_ready = amount_ready + :amount_ready WHERE id = :quotaRequestChangeId";

    private static final String CHANGE_INCREMENT_READY_AND_ALLOCATED_QUERY = "UPDATE quota_request_change SET " +
            "amount_ready = amount_ready + :amount_ready, amount_allocating = amount_allocated + :amount_allocated, amount_allocated = amount_allocated + :amount_allocated WHERE id = :quotaRequestChangeId";

    public static final TypeReference<Map<String, String>> PROPERTIES_TYPE = new TypeReference<>() {
    };

    private static final String SET_READY_FOR_ALLOCATION_STATE_QUERY = "UPDATE quota_request SET ready_for_allocation = :readyForAllocation, updated = :updated WHERE id IN (:quotaRequestId)";
    private static final String SET_SHOW_ALLOCATION_NOTE_QUERY = "UPDATE quota_request SET show_allocation_note = :showAllocationNote WHERE id IN (:quotaRequestId)";

    private static final String SELECT_SEGMENTED_CHANGES_FOR_UPDATE_BASE =
            "SELECT qrc.id AS change_id, qr.id AS request_id, qr.project_id, " +
                    "qrc.resource_id, qrc.amount, qrc.amount_ready, qrc.amount_allocated, qrc.amount_allocating, qrc.owning_cost, qrcs.segment_id " +
                    "FROM quota_request qr JOIN quota_request_change qrc ON qrc.quota_request_id=qr.id " +
                    "JOIN quota_request_change_segment qrcs ON qrcs.quota_request_change_id=qrc.id " +
                    "WHERE qr.campaign_id = :campaign_id AND qrc.order_id = :order_id " +
                    "AND qr.type = cast (:request_type as quota_request_type) AND qr.status in (%s) AND qrc.amount > qrc.amount_ready AND (%s) " +
                    "ORDER BY qr.id ASC, qrc.id ASC, qrcs.segment_id ASC%s";
    public static final String RESOURCE_SEGMENT_CONDITION = "(qrc.resource_id = :resource_id_%d AND qrcs.segment_id = :segment_id_%d)";
    public static final String RESOURCE_SEGMENT_SUBQUERY_CONDITION = "(qrc.resource_id = :resource_id_%d AND exists (" +
            "SELECT s.quota_request_change_id FROM quota_request_change_segment s WHERE s.quota_request_change_id = qrc.id AND s.segment_id IN (%s) " +
            "GROUP BY s.quota_request_change_id HAVING COUNT(s.segment_id) = :segments_count_%d))";
    private static final String SELECT_NON_SEGMENTED_CHANGES_FOR_UPDATE_BASE =
            "SELECT qrc.id AS change_id, qr.id AS request_id, qr.project_id, " +
                    "qrc.resource_id, qrc.amount, qrc.amount_ready, qrc.amount_allocated, qrc.amount_allocating, qrc.owning_cost " +
                    "FROM quota_request qr JOIN quota_request_change qrc ON qrc.quota_request_id=qr.id " +
                    "WHERE qr.campaign_id = :campaign_id AND qrc.order_id = :order_id " +
                    "AND qr.type = cast (:request_type as quota_request_type) AND qr.status in (%s) " +
                    "AND qrc.amount > qrc.amount_ready AND qrc.resource_id IN (%s) " +
                    "ORDER BY qr.id ASC, qrc.id ASC%s";
    private static final String SELECT_CHANGES_BY_REQUEST =
            "SELECT qrc.id AS change_id, qrc.quota_request_id AS request_id, " +
                    "qrc.resource_id, qrc.amount, qrc.amount_ready, qrc.amount_allocated, qrc.amount_allocating, qrc.owning_cost, qrcs.segment_id " +
                    "FROM quota_request_change qrc " +
                    "LEFT JOIN quota_request_change_segment qrcs ON qrcs.quota_request_change_id=qrc.id " +
                    "WHERE qrc.quota_request_id IN (:quota_request_ids) " +
                    "ORDER BY qrc.quota_request_id ASC, qrc.id ASC, qrcs.segment_id ASC";
    private static final String SELECT_CHANGES_BY_REQUEST_AND_SERVICE_FOR_UPDATE =
            "SELECT qrc.id AS change_id, qrc.quota_request_id AS request_id, qrc.order_id, bo.date, " +
                    "qrc.resource_id, qrc.amount, qrc.amount_ready, qrc.amount_allocated, qrc.amount_allocating, qrc.owning_cost, qrcs.segment_id " +
                    "FROM quota_request_change qrc " +
                    "LEFT JOIN quota_request_change_segment qrcs ON qrcs.quota_request_change_id=qrc.id " +
                    "LEFT JOIN resource r ON qrc.resource_id=r.id " +
                    "LEFT JOIN bot_big_order bo ON qrc.order_id=bo.id " +
                    "WHERE qrc.quota_request_id = :requestId AND r.service_id = :serviceId " +
                    "ORDER BY qrc.quota_request_id ASC, qrc.id ASC, qrcs.segment_id ASC FOR UPDATE OF qrc";


    private static final String SELECT_REQUESTS_BY_CAMPAIGN_IDS_FOR_UPDATE =
            "WITH request_ids AS (" +
                    "    SELECT id" +
                    "    FROM quota_request" +
                    "    WHERE campaign_id in (:ids)" +
                    "      AND (cast(:fromId as int8) is null OR id > :fromId)" +
                    "    ORDER BY id" +
                    "    LIMIT :limit" +
                    ") " +
                    SELECT_ALL_FIELDS_CLAUSE +
                    "FROM quota_request" +
                    "         JOIN quota_request_change ON quota_request.id = quota_request_change.quota_request_id" +
                    "         LEFT JOIN quota_request_change_segment" +
                    "                   ON quota_request_change_segment.quota_request_change_id = " +
                    "quota_request_change.id" +
                    "         LEFT JOIN campaign ON quota_request.campaign_id = campaign.id" +
                    "         LEFT JOIN bot_big_order ON quota_request_change.order_id = bot_big_order.id " +
                    "         LEFT JOIN campaign_big_order ON quota_request.campaign_id = campaign_big_order.campaign_id AND quota_request_change.order_id = campaign_big_order.big_order_id " +
                    "         LEFT JOIN goal ON quota_request.goal_id = goal.id " +
                    "WHERE quota_request.id IN (SELECT id from request_ids)" +
                    "    ORDER BY quota_request.id " +
                    "    FOR UPDATE OF quota_request_change";

    private static final String SELECT_REQUESTS_OWNING_COST_BY_CAMPAIGN_ID =
            "WITH request_ids AS (" +
                    "    SELECT id" +
                    "    FROM quota_request" +
                    "    WHERE campaign_id = (:id)" +
                    "      AND (cast(:fromId as int8) is null OR id > :fromId)" +
                    "      AND request_owning_cost > 0" +
                    "      AND status in (%s)" +
                    "    ORDER BY id" +
                    "    LIMIT :limit" +
                    ") " +
                    "SELECT id, request_owning_cost "+
                    "FROM quota_request " +
                    "WHERE quota_request.id IN (SELECT id from request_ids) " +
                    "ORDER BY id";

    private static final String SELECT_STATUS_AND_ISSUE_BY_REQUEST =
            "SELECT id, status, ticket_key FROM quota_request WHERE id IN (:ids) ORDER BY id ASC";

    private static final String UPDATE_COST_QUERY = "UPDATE quota_request SET cost = total.cost FROM (SELECT poqr.quota_request_id request_id, SUM" +
            "(poqr.cost) \"cost\" FROM pre_order_quota_request poqr WHERE poqr.quota_request_id IN (:id) GROUP BY poqr.quota_request_id) total " +
            "WHERE total.request_id = quota_request.id";

    private static final String REPORT_AGGREGATION_QUERY = "SELECT qr.project_id, bpo.service_id, bpo.server_id, SUM(poqr.server_quantity) " +
            "total_server_quantity, SUM(poqr.cost) total_cost FROM quota_request qr JOIN pre_order_quota_request poqr ON qr.id = poqr" +
            ".quota_request_id " + "JOIN bot_pre_order bpo ON bpo.id = poqr.pre_order_id WHERE qr.id IN (%s) " + "GROUP BY qr.project_id, bpo" +
            ".service_id, bpo.server_id";

    public static final String CHANGE_REQUEST_OWNING_COST = "UPDATE quota_request AS qr SET " +
            "    request_owning_cost = c.request_owning_cost " +
            "FROM (values " +
            "     :values " +
            "     ) as c(id, request_owning_cost) " +
            "WHERE c.id = qr.id";

    public static final String UPDATE_UNBALANCED = "UPDATE quota_request AS qr SET " +
            "    unbalanced = c.unbalanced " +
            "FROM (values " +
            "     :values " +
            "     ) as c(id, unbalanced) " +
            "WHERE c.id = qr.id";

    private final CampaignDao campaignDao;
    private final HierarchySupplier hierarchySupplier;

    @Inject
    public SqlQuotaChangeRequestDao(final CampaignDao campaignDao, final HierarchySupplier hierarchySupplier) {
        this.campaignDao = campaignDao;
        this.hierarchySupplier = hierarchySupplier;
    }

    @Override
    @NotNull
    public Set<QuotaChangeRequest> getAll() {
        return query(GET_ALL_QUERY, Collections.emptyMap()).collect(Collectors.toSet());
    }

    @NotNull
    @Override
    public QuotaChangeRequest read(final long id) throws EmptyResultDataAccessException {
        return query(GET_BY_ID_QUERY, Collections.singletonMap("quotaRequestId", id))
                .findFirst()
                .orElseThrow(() -> new EmptyResultDataAccessException("No quota change request with id " + id, 1));
    }

    @NotNull
    @Override
    public Map<Long, QuotaChangeRequest> read(final Collection<Long> ids) {
        if (ids.isEmpty()) {
            return Collections.emptyMap();
        }
        return query(GET_BY_ID_QUERY, Collections.singletonMap("quotaRequestId", ids))
                .collect(Collectors.toMap(LongIndexBase::getId, Function.identity()));
    }

    @NotNull
    @Override
    public Optional<QuotaChangeRequest> findByTicketKey(final String ticketKey) {
        return query(GET_BY_TICKET_KEY_QUERY, Collections.singletonMap("ticketKey", ticketKey)).findFirst();
    }

    @Override
    public Map<String, QuotaChangeRequest> findByTicketKeys(final Collection<String> ticketKeys) {
        if (ticketKeys.isEmpty()) {
            return Collections.emptyMap();
        }
        return query(GET_BY_TICKET_KEYS_QUERY, Collections.singletonMap("ticketKeys", ticketKeys))
                .collect(Collectors.toMap(QuotaChangeRequest::getTrackerIssueKey, Function.identity()));
    }

    @Override
    public Map<String, QuotaChangeRequest> findByTicketKeysForUpdate(final Collection<String> ticketKeys) {
        if (ticketKeys.isEmpty()) {
            return Collections.emptyMap();
        }
        return query(GET_BY_TICKET_KEYS_QUERY_FOR_UPDATE, Collections.singletonMap("ticketKeys", ticketKeys))
                .collect(Collectors.toMap(QuotaChangeRequest::getTrackerIssueKey, Function.identity()));
    }

    @Override
    @NotNull
    public List<QuotaChangeRequest> read(final QuotaChangeRequestFilter filter) {
        return readStream(filter).collect(Collectors.toList());
    }

    @Override
    @NotNull
    public List<QuotaChangeRequest> readRequestsWithFilteredChanges(final QuotaChangeRequestFilter filter) {
        final Map<String, Object> queryParams = new HashMap<>();
        final StringBuilder queryBuilder = new StringBuilder(GET_ALL_QUERY);

        final String where = conditionsToWhere(makeFilters(filter, queryParams, true));
        queryBuilder.append(where);

        return query(queryBuilder.toString(), queryParams).collect(Collectors.toList());
    }

    @Override
    @NotNull
    public Stream<QuotaChangeRequest> readStream(final QuotaChangeRequestFilter filter) {
        final Map<String, Object> queryParams = new HashMap<>();
        final StringBuilder queryBuilder = new StringBuilder(GET_ALL_QUERY);

        final String where = conditionsToWhere(makeFilters(filter, queryParams));
        queryBuilder.append(where)
                .append(getOrderBy(filter));

        return query(queryBuilder.toString(), queryParams);
    }

    @NotNull
    private String getOrderBy(final QuotaChangeRequestFilter filter) {
        String result = " ORDER BY ";
        switch (filter.getSortBy()) {
            case CREATED_AT:
                result += "created ";
                break;
            case UPDATED_AT:
                result += "updated ";
                break;
            case COST:
                result += "cost ";
                break;
            case REQUEST_OWNING_COST:
                result += "request_owning_cost ";
                break;
            default:
                throw new IllegalArgumentException("Not supported order field: " + filter.getSortBy());
        }
        result += filter.getSortOrder() + ", id " + filter.getSortOrder();
        return result;
    }

    @Override
    @NotNull
    public Map<Resource, Long> readAggregation(final QuotaChangeRequestFilter filter) {
        final Map<String, Object> queryParams = new HashMap<>();

        final String query = String.format(GET_AGGREGATION_QUERY_TEMPLATE, conditionsToWhere(makeFilters(filter, queryParams)));

        final ResourceReader resourceReader = Hierarchy.get().getResourceReader();

        final HashMap<Resource, Long> result = new HashMap<>();

        jdbcTemplate.query(query, queryParams, (rs) -> {
            final Resource resource = resourceReader.read(rs.getLong("resource_id"));
            final long amount = rs.getLong("amount");
            result.put(resource, amount);
        });

        return result;
    }

    @Override
    @NotNull
    public Stream<ReportQuotaChangeRequest> readReport(final QuotaChangeRequestFilter filter,
                                                       final boolean showGoalQuestions,
                                                       final boolean filterEmptyChange,
                                                       final boolean showGoalHierarchy) {
        final Map<String, Object> queryParams = new HashMap<>();

        final SqlFilters filters = makeFilters(filter, queryParams, true);
        if (filterEmptyChange) {
            filters.addCondition("quota_request_change.amount > 0");
        }

        final List<String> additionalFields = new ArrayList<>();

        if (showGoalHierarchy) {
            filters.addJoin("LEFT JOIN goal okr_vs ON okr_vs.id = goal.value_stream_goal_id");
            additionalFields.add("okr_vs.name as okr_vs_name");
            filters.addJoin("LEFT JOIN goal okr_umb ON okr_umb.id = goal.umbrella_goal_id");
            additionalFields.add("okr_umb.name as okr_umb_name");
            filters.addJoin("LEFT JOIN goal okr_con ON okr_con.id = goal.contour_goal_id");
            additionalFields.add("okr_con.name as okr_con_name");
        }

        final String additionalFieldString;
        if (!additionalFields.isEmpty()) {
            additionalFieldString = ", " + String.join(", ", additionalFields) + " ";
        } else {
            additionalFieldString = "";
        }

        final String queryBuilder = SELECT_ALL_FIELDS_CLAUSE + additionalFieldString + FROM_CLAUSE + conditionsToWhere(filters) +
                getOrderBy(filter);


        return query(queryBuilder, queryParams, new ReportRequestCollector(showGoalQuestions, showGoalHierarchy));
    }

    @Override
    public @NotNull Page<QuotaChangeRequest> readPage(
            final QuotaChangeRequestFilter filter,
            final PageInfo pageInfo) {
        final Map<String, Object> queryParams = new HashMap<>();
        final String where = conditionsToWhere(makeFilters(filter, queryParams));

        queryParams.put("page_offset", pageInfo.getOffset());
        queryParams.put("page_limit", pageInfo.getPageSize());
        final String order = getOrderBy(filter);
        final String pagination = " OFFSET :page_offset LIMIT :page_limit";

        final List<Long> requestIds = jdbcTemplate.query(GET_IDS + where + order + pagination, queryParams, (rs, i) -> rs.getLong("id"));
        final long count = jdbcTemplate.queryForObject(COUNT_ALL_QUERY + where, queryParams, Long.class);

        queryParams.put("requestIds", requestIds);
        if (requestIds.isEmpty()) {
            return Page.of(Stream.empty(), count);
        }

        final String query = GET_ALL_QUERY + " WHERE quota_request.id IN (:requestIds)" + order;
        return Page.of(query(query, queryParams), count);
    }

    @Override
    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public QuotaChangeRequest create(final QuotaChangeRequest quotaChangeRequest) {
        final long id = jdbcTemplate.insert(BASE_INSERT_QUERY, toParams(quotaChangeRequest));
        final QuotaChangeRequest insertedRequest = quotaChangeRequest.copyBuilder()
                .id(id)
                .build();

        insertChanges(id, quotaChangeRequest.getChanges());

        return insertedRequest;
    }

    private void insertChanges(final long requestId, final List<QuotaChangeRequest.Change> changes) {
        for (final QuotaChangeRequest.Change change : changes) {
            final long changeId = jdbcTemplate.insert(CHANGE_INSERT_QUERY, toChangeInsertParams(requestId, change));
            change.setId(changeId);
            jdbcTemplate.batchUpdate(CHANGE_SEGMENT_INSERT_QUERY, toChangeSegmentsParams(change));
        }
    }

    @NotNull
    @Override
    public QuotaChangeRequest read(@NotNull final Long id) throws EmptyResultDataAccessException {
        return read(id.longValue());
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean update(final QuotaChangeRequest quotaChangeRequest) {
        return jdbcTemplate.update(UPDATE_QUERY, toParams(quotaChangeRequest)) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean updateAll(final Collection<QuotaChangeRequest> requests) {
        final int[] updatedArray = jdbcTemplate.batchUpdate(UPDATE_QUERY, requests.stream()
                .map(SqlQuotaChangeRequestDao::toParams)
                .collect(Collectors.toList())
        );

        boolean result = false;
        for (final int updated : updatedArray) {
            result |= updated > 0;
        }
        return result;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean delete(@NotNull final QuotaChangeRequest quotaChangeRequest) {
        return jdbcTemplate.update(DELETE_QUERY, toParams(quotaChangeRequest)) > 0;
    }

    @Override
    @Transactional
    public boolean clear() {
        return jdbcTemplate.update(CLEAR_QUERY) > 0;
    }

    @NotNull
    private static Map<String, Object> toParams(final QuotaChangeRequest quotaChangeRequest) {
        return MapBuilder.<String, Object>of("quotaRequestId", quotaChangeRequest.getId())
                .put("projectId", quotaChangeRequest.getProject().getId())
                .put("authorId", quotaChangeRequest.getAuthor().getId())
                .put("created", SqlUtils.toTimestamp(quotaChangeRequest.getCreated()))
                .put("updated", SqlUtils.toTimestamp(quotaChangeRequest.getUpdated()))
                .put("description", quotaChangeRequest.getDescription())
                .put("comment", quotaChangeRequest.getComment())
                .put("calculations", quotaChangeRequest.getCalculations())
                .put("trackerIssueKey", quotaChangeRequest.getTrackerIssueKey())
                .put("status", quotaChangeRequest.getStatus().name())
                .put("type", quotaChangeRequest.getType().name())
                .put("sourceProjectId", quotaChangeRequest.getSourceProject() != null ? quotaChangeRequest.getSourceProject().getId() : null)
                .put("chartLinks", StringUtils.join(quotaChangeRequest.getChartLinks(), CHART_LINKS_DELIMITER))
                .put("chartLinksAbsenceExplanation", quotaChangeRequest.getChartLinksAbsenceExplanation())
                .put("additionalProperties", SqlUtils.toJsonb(quotaChangeRequest.getAdditionalProperties()))
                .put("campaign_id", quotaChangeRequest.getCampaignId())
                .put("resourcePreorderReasonType", quotaChangeRequest.getResourcePreorderReasonType() != null ? quotaChangeRequest.getResourcePreorderReasonType().name() : null)
                .put("goalId", quotaChangeRequest.getGoal() == null ? null : quotaChangeRequest.getGoal().getId())
                .put("readyForAllocation", quotaChangeRequest.isReadyForAllocation())
                .put("requestGoalAnswers", quotaChangeRequest.getRequestGoalAnswers() != null ? SqlUtils.toJsonb(quotaChangeRequest.getRequestGoalAnswers()) : null)
                .put("cost", quotaChangeRequest.getCost())
                .put("summary", quotaChangeRequest.getSummary())
                .put("requestOwningCost", quotaChangeRequest.getRequestOwningCost())
                .put("importantRequest", quotaChangeRequest.isImportantRequest())
                .put("unbalanced", quotaChangeRequest.isUnbalanced())
                .put("campaign_type", quotaChangeRequest.getCampaign() != null ? quotaChangeRequest.getCampaign().getType().name() : null)
                .build();
    }

    @NotNull
    private static Map<String, ?> toChangeInsertParams(final long quotaRequestId, final QuotaChangeRequest.Change change) {

        final QuotaChangeRequest.BigOrder bigOrder = change.getKey().getBigOrder();
        final HashMap<String, Object> params = new HashMap<>();
        params.put("quotaRequestId", quotaRequestId);
        params.put("resourceId", change.getResource().getId());
        params.put("orderId", bigOrder == null ? null : bigOrder.getId());
        params.put("amount", change.getAmount());
        params.put("amountReady", change.getAmountReady());
        params.put("amountAllocated", change.getAmountAllocated());
        params.put("amountAllocating", change.getAmountAllocating());
        params.put("owningCost", change.getOwningCost().toString());

        return params;
    }

    @NotNull
    private static Map<String, ?> toChangeUpdateParams(final QuotaChangeRequest.Change change) {

        final HashMap<String, Object> params = new HashMap<>();
        params.put("quotaRequestChangeId", change.getId());
        params.put("amount", change.getAmount());
        params.put("amountReady", change.getAmountReady());
        params.put("amountAllocated", change.getAmountAllocated());
        params.put("amountAllocating", change.getAmountAllocating());
        params.put("owningCost", change.getOwningCost().toString());

        return params;
    }

    @NotNull
    private static List<Map<String, ?>> toChangeSegmentsParams(final QuotaChangeRequest.Change quotaRequestChange) {

        return quotaRequestChange.getSegments().stream()
                .map(segment -> ImmutableMap.of(
                        "quotaRequestChangeId", quotaRequestChange.getId(),
                        "segmentId", segment.getId()
                )).collect(Collectors.toList());
    }


    @NotNull
    private <T> T query(final String query, final Map<String, ?> params, final RequestCollector<T> collector) {
        jdbcTemplate.query(query, params, collector::processRow);
        return collector.get();
    }

    @NotNull
    private <T> T query(final String query, final SqlParameterSource paramSource, final RequestCollector<T> collector) {
        jdbcTemplate.query(query, paramSource, collector::processRow);
        return collector.get();
    }

    @NotNull
    private Stream<QuotaChangeRequest> query(final String query, final Map<String, ?> params) {
        return query(query, params, new DefaultRequestCollector());
    }

    @NotNull
    private Stream<QuotaChangeRequest> query(final String query, SqlParameterSource paramSource) {
        return query(query, paramSource, new DefaultRequestCollector());
    }

    @Override
    @NotNull
    public List<QuotaChangeRequest.Change> setChanges(@NotNull final QuotaChangeRequest request, final List<QuotaChangeRequest.Change> newChanges) {
        final Map<QuotaChangeRequest.ChangeKey, QuotaChangeRequest.Change> newChangesByKey = newChanges.stream()
                .collect(Collectors.toMap(QuotaChangeRequest.Change::getKey, Function.identity()));

        final Map<QuotaChangeRequest.ChangeKey, QuotaChangeRequest.Change> existingChangesByKey = request.getChanges().stream()
                .collect(Collectors.toMap(QuotaChangeRequest.Change::getKey, Function.identity()));

        final List<QuotaChangeRequest.Change> toUpdate = Sets.intersection(newChangesByKey.keySet(), existingChangesByKey.keySet()).stream()
                .map(newChangesByKey::get)
                .filter(change -> !change.equals(existingChangesByKey.get(change.getKey())))
                .peek(change -> change.setId(existingChangesByKey.get(change.getKey()).getId()))
                .collect(Collectors.toList());

        updateChanges(toUpdate);

        final List<QuotaChangeRequest.Change> toRemove = Sets.difference(existingChangesByKey.keySet(), newChangesByKey.keySet()).stream()
                .map(existingChangesByKey::get)
                .collect(Collectors.toList());

        removeChanges(toRemove);

        final List<QuotaChangeRequest.Change> toCreate = Sets.difference(newChangesByKey.keySet(), existingChangesByKey.keySet()).stream()
                .map(newChangesByKey::get)
                .collect(Collectors.toList());

        insertChanges(request.getId(), toCreate);

        return newChanges;
    }

    @Override
    public Set<Long> getBigOrderIdsForRequestsInStatuses(final Set<QuotaChangeRequest.Status> statuses) {
        if (statuses.isEmpty()) {
            return Collections.emptySet();
        }

        final Map<String, Object> queryParams = new HashMap<>();

        final String allStatusParams = getStatusParamStatement(statuses, queryParams);

        final String query = String.format(GET_ORDER_IDS_BY_STATUSES_TEMPLATE, allStatusParams);


        return jdbcTemplate.queryForSet(query, queryParams, (rs, e) -> rs.getLong("order_id"));
    }

    @NotNull
    private String getStatusParamStatement(final Set<QuotaChangeRequest.Status> statuses, final Map<String, Object> queryParams) {
        final List<String> statusParams = new ArrayList<>();

        int counter = 0;
        for (final QuotaChangeRequest.Status status : statuses) {
            final String key = "status_" + counter;
            queryParams.put(key, status.name());
            statusParams.add("cast (:" + key + " as quota_request_status)");
            counter += 1;
        }

        return String.join(", ", statusParams);
    }

    @Override
    public void moveToProject(final Collection<Long> requestIds, final Project project) {
        if (requestIds.isEmpty()) {
            return;
        }

        final @NotNull Map<String, Object> params = new HashMap<>();
        params.put("ids", requestIds);
        params.put("project", project.getId());
        params.put("updated", Timestamp.from(Instant.now()));

        jdbcTemplate.update(MOVE_TO_PROJECT_QUERY, params);
    }

    @Override
    public Set<Long> getRequestsIdsByChangeIds(final Set<Long> changedChanges) {
        return jdbcTemplate.queryForSet(GET_REQUEST_IDS_BY_CHANGE_ID, Collections.singletonMap("changeIds", changedChanges), (rs, e) -> rs.getLong("quota_request_id"));
    }

    @Override
    public boolean hasRequestsInCampaign(final long campaignId) {
        return jdbcTemplate.queryForObject(EXISTS_BY_CAMPAIGN_ID, new MapSqlParameterSource("campaign_id", campaignId), (rs, rn) -> {
            return rs.getBoolean("exists");
        });
    }

    @Override
    public boolean hasRequestsInCampaignForOrdersOtherThan(final long campaignId, final Set<Long> orderIds) {
        final MapSqlParameterSource params = new MapSqlParameterSource(ImmutableMap.of("campaign_id", campaignId, "order_ids", orderIds));
        return jdbcTemplate.queryForObject(EXISTS_BY_CAMPAIGN_ID_AND_ORDER_ID_NOT_IN, params, (rs, rn) -> {
            return rs.getBoolean("exists");
        });
    }

    @Override
    public boolean update(final Collection<QuotaChangeRequest> values) {
        if (values.isEmpty()) {
            return false;
        }

        final List<Map<String, ?>> params = values.stream()
                .map(SqlQuotaChangeRequestDao::toParams)
                .collect(Collectors.toList());

        final int[] updates = jdbcTemplate.batchUpdate(UPDATE_QUERY, params);
        for (final int update : updates) {
            if (update > 0) {
                return true;
            }
        }
        return false;
    }

    @Override
    @NotNull
    public Stream<Pair<QuotaChangeRequest, QuotaChangeRequest>> setStatuses(final QuotaChangeRequestFilter filter,
                                                                            final QuotaChangeRequest.Status status,
                                                                            final long updateTimeMillis) {
        final Map<String, Object> queryParams = new HashMap<>();
        queryParams.put("newStatus", status.name());
        queryParams.put("newUpdated", SqlUtils.toTimestamp(updateTimeMillis));

        final String where = conditionsToWhere(makeFilters(filter, queryParams));

        final String query = String.format(UPDATE_AND_RETURN_QUERY_TEMPLATE, where);

        return this.query(query, queryParams, new StatusUpdatedRequestCollector());
    }

    private void removeChanges(final List<QuotaChangeRequest.Change> changes) {
        if (changes.isEmpty()) {
            return;
        }
        jdbcTemplate.update(CHANGE_DELETE_QUERY, Collections.singletonMap("quotaRequestChangeId", ids(changes)));
    }

    @Override
    public void updateChanges(final Collection<QuotaChangeRequest.Change> changes) {
        if (changes.isEmpty()) {
            return;
        }
        jdbcTemplate.batchUpdate(CHANGE_UPDATE_QUERY, changes.stream()
                .map(SqlQuotaChangeRequestDao::toChangeUpdateParams)
                .collect(Collectors.toList()));
    }

    @Override
    public void updateChangesOwningCost(Collection<QuotaChangeRequest.Change> changesToUpdate) {
        if (changesToUpdate.isEmpty()) {
            return;
        }

        String values = changesToUpdate.stream()
                .map(SqlQuotaChangeRequestDao::toChangeUpdateParams)
                .map(changeMap -> "(" + changeMap.get("quotaRequestChangeId") + ", " + changeMap.get("owningCost") + ")")
                .collect(Collectors.joining(","));

        jdbcTemplate.update(CHANGE_OWNING_COST.replace(":values", values));
    }

    @Override
    public void updateRequestsOwningCost(Map<Long, Long> requestOwningCostMap) {
        if (requestOwningCostMap.isEmpty()) {
            return;
        }

        String values = requestOwningCostMap.entrySet().stream()
                .map(e -> "(" + e.getKey() + ", " + e.getValue() + ")")
                .collect(Collectors.joining(","));

        jdbcTemplate.update(CHANGE_REQUEST_OWNING_COST.replace(":values", values));
    }

    @Override
    public void updateUnbalanced(Map<Long, Boolean> unbalancedByRequestId) {
        if (unbalancedByRequestId.isEmpty()) {
            return;
        }

        String values = unbalancedByRequestId.entrySet().stream()
                .map(e -> "(" + e.getKey() + ", " + e.getValue() + ")")
                .collect(Collectors.joining(","));

        jdbcTemplate.update(UPDATE_UNBALANCED.replace(":values", values));
    }

    public String queryForIds(final QuotaChangeRequestFilter filter, final Map<String, Object> queryParams) {
        return GET_IDS + conditionsToWhere(makeFilters(filter, queryParams));
    }

    @Override
    public void setReadyForAllocationState(final Collection<Long> requestIds, final boolean isReadyForAllocation) {
        if (requestIds.isEmpty()) {
            return;
        }
        jdbcTemplate.update(SET_READY_FOR_ALLOCATION_STATE_QUERY, ImmutableMap.of(
                "quotaRequestId", requestIds,
                "readyForAllocation", isReadyForAllocation,
                "updated", Timestamp.from(Instant.now())
        ));
    }

    @Override
    public void setShowAllocationNote(Collection<Long> requestIds, boolean showAllocationNote) {
        if (requestIds.isEmpty()) {
            return;
        }
        jdbcTemplate.update(SET_SHOW_ALLOCATION_NOTE_QUERY, ImmutableMap.of(
                "quotaRequestId", requestIds,
                "showAllocationNote", showAllocationNote
        ));
    }

    @Override
    public Map<Long, QuotaChangeRequest> readForUpdate(final Collection<Long> ids) {
        if (ids.isEmpty()) {
            return Collections.emptyMap();
        }
        return query(GET_BY_ID_FOR_UPDATE_QUERY, Collections.singletonMap("quotaRequestId", ids))
                .collect(Collectors.toMap(LongIndexBase::getId, Function.identity()));
    }

    private void setChangesAmount(final String query, final Map<Long, Long> amountById) {
        if (amountById.isEmpty()) {
            return;
        }
        final List<Map<String, ?>> params = amountById.entrySet().stream()
                .map(e -> ImmutableMap.of(
                        "quotaRequestChangeId", e.getKey(),
                        "amount", e.getValue()
                ))
                .collect(Collectors.toList());

        jdbcTemplate.batchUpdate(query, params);
    }

    @Override
    public void setChangesReadyAmount(final Map<Long, Long> readyByChangeId) {
        setChangesAmount(CHANGE_SET_READY_QUERY, readyByChangeId);
    }

    @Override
    public void setChangesAllocatedAmount(final Map<Long, Long> allocatedByChangeId) {
        setChangesAmount(CHANGE_SET_ALLOCATED_QUERY, allocatedByChangeId);
    }

    @Override
    public void setChangesAllocatingAmount(final Map<Long, Long> allocatingByChangeId) {
        setChangesAmount(CHANGE_SET_ALLOCATING_QUERY, allocatingByChangeId);
    }

    @Override
    public void incrementChangesReadyAmount(@NotNull final Map<Long, Long> readyIncrementByChangeId) {
        if (readyIncrementByChangeId.isEmpty()) {
            return;
        }
        final List<Map<String, ?>> params = readyIncrementByChangeId.entrySet().stream()
                .map(e -> ImmutableMap.of(
                        "quotaRequestChangeId", e.getKey(),
                        "amount_ready", e.getValue()
                ))
                .collect(Collectors.toList());
        jdbcTemplate.batchUpdate(CHANGE_INCREMENT_READY_QUERY, params);
    }

    @Override
    public void incrementChangesAllocatedAmount(@NotNull final Map<Long, Long> allocatedIncrementByChangeId) {
        if (allocatedIncrementByChangeId.isEmpty()) {
            return;
        }
        final List<Map<String, ?>> params = allocatedIncrementByChangeId.entrySet().stream()
                .map(e -> ImmutableMap.of(
                        "quotaRequestChangeId", e.getKey(),
                        "amount_allocated", e.getValue()
                ))
                .collect(Collectors.toList());
        jdbcTemplate.batchUpdate(CHANGE_INCREMENT_ALLOCATED_QUERY, params);
    }

    @Override
    public void incrementChangesReadyAndAllocatedAmount(@NotNull final Map<Long, Pair<Long, Long>> readyAndAllocatedIncrementByChangeId) {
        if (readyAndAllocatedIncrementByChangeId.isEmpty()) {
            return;
        }
        final List<Map<String, ?>> params = readyAndAllocatedIncrementByChangeId.entrySet().stream()
                .map(e -> ImmutableMap.of(
                        "quotaRequestChangeId", e.getKey(),
                        "amount_ready", e.getValue().getLeft(),
                        "amount_allocated", e.getValue().getRight()
                ))
                .collect(Collectors.toList());
        jdbcTemplate.batchUpdate(CHANGE_INCREMENT_READY_AND_ALLOCATED_QUERY, params);
    }

    @NotNull
    @Override
    public List<QuotaChangeInRequest> selectSegmentedChangesForQuotaDistributionUpdate(
            final long campaignId, final long bigOrderId, final long serviceId,
            @NotNull final QuotaChangeRequest.Type type,
            @NotNull final Set<QuotaChangeRequest.Status> statuses,
            @NotNull final Set<ResourceSegments> resourceSegments, final boolean lock) {
        if (resourceSegments.stream().anyMatch(rs -> rs.getSegmentIds().isEmpty())) {
            throw new IllegalArgumentException("Empty segment sets are not allowed");
        }
        if (resourceSegments.isEmpty() || statuses.isEmpty()) {
            return Collections.emptyList();
        }
        final Map<String, Object> queryParams = new HashMap<>();
        queryParams.put("campaign_id", campaignId);
        queryParams.put("order_id", bigOrderId);
        queryParams.put("service_id", serviceId);
        queryParams.put("request_type", type.name());
        final Set<String> statusInSet = new HashSet<>();
        long statusIndex = 0L;
        for (final QuotaChangeRequest.Status status : statuses) {
            statusInSet.add(String.format("CAST(:status_%d AS quota_request_status)", statusIndex));
            final String param = String.format("status_%d", statusIndex);
            queryParams.put(param, status.name());
            statusIndex++;
        }
        final String statusIn = String.join(", ", statusInSet);
        long resourceSegmentIndex = 0L;
        final Set<String> resourceSegmentsConstraintSet = new HashSet<>();
        for (final ResourceSegments rs : resourceSegments) {
            if (rs.getSegmentIds().size() == 1) {
                resourceSegmentsConstraintSet.add(String.format(RESOURCE_SEGMENT_CONDITION,
                        resourceSegmentIndex, resourceSegmentIndex));
                final String resourceParam = String.format("resource_id_%d", resourceSegmentIndex);
                queryParams.put(resourceParam, rs.getResourceId());
                final String segmentParam = String.format("segment_id_%d", resourceSegmentIndex);
                queryParams.put(segmentParam, rs.getSegmentIds().iterator().next());
            } else {
                final String resourceParam = String.format("resource_id_%d", resourceSegmentIndex);
                queryParams.put(resourceParam, rs.getResourceId());
                final Set<String> segmentInSet = new HashSet<>();
                long segmentIndex = 0L;
                for (final Long segmentId : rs.getSegmentIds()) {
                    segmentInSet.add(String.format(":segment_id_%d_%d", resourceSegmentIndex, segmentIndex));
                    final String segmentParam = String.format("segment_id_%d_%d", resourceSegmentIndex, segmentIndex);
                    queryParams.put(segmentParam, segmentId);
                    segmentIndex++;
                }
                final String segmentIn = String.join(", ", segmentInSet);
                final String segmentsCountParam = String.format("segments_count_%d", resourceSegmentIndex);
                queryParams.put(segmentsCountParam, (long) rs.getSegmentIds().size());
                resourceSegmentsConstraintSet.add(String.format(RESOURCE_SEGMENT_SUBQUERY_CONDITION,
                        resourceSegmentIndex, segmentIn, resourceSegmentIndex));
            }
            resourceSegmentIndex++;
        }
        final String resourceSegmentsConstraint = String.join(" OR ", resourceSegmentsConstraintSet);
        final String query = String.format(SELECT_SEGMENTED_CHANGES_FOR_UPDATE_BASE, statusIn, resourceSegmentsConstraint,
                lock ? " FOR UPDATE OF qr, qrc, qrcs" : "");
        final Hierarchy hierarchy = hierarchySupplier.get();
        return jdbcTemplate.query(query, new MapSqlParameterSource(queryParams), rs -> {
            final List<QuotaChangeInRequest> result = new ArrayList<>();
            QuotaChangeInRequest.Builder builder = null;
            Long lastChangeId = null;
            while (rs.next()) {
                final long changeId = rs.getLong("change_id");
                if (lastChangeId == null || !Objects.equals(lastChangeId, changeId)) {
                    if (builder != null) {
                        result.add(builder.build());
                    }
                    lastChangeId = changeId;
                    builder = QuotaChangeInRequest.builder();
                    final long requestId = rs.getLong("request_id");
                    final long resourceId = rs.getLong("resource_id");
                    final long projectId = rs.getLong("project_id");
                    final long amount = rs.getLong("amount");
                    final long amountReady = rs.getLong("amount_ready");
                    final long amountAllocated = rs.getLong("amount_allocated");
                    final long amountAllocating = rs.getLong("amount_allocating");
                    builder.id(changeId)
                            .requestId(requestId)
                            .resource(hierarchy.getResourceReader().read(resourceId))
                            .project(hierarchy.getProjectReader().read(projectId))
                            .amount(amount)
                            .amountReady(amountReady)
                            .amountAllocated(amountAllocated)
                            .amountAllocating(amountAllocating);
                }
                final long segmentId = rs.getLong("segment_id");
                builder.addSegment(hierarchy.getSegmentReader().read(segmentId));
            }
            if (builder != null) {
                result.add(builder.build());
            }
            return result;
        });
    }

    @NotNull
    @Override
    public List<QuotaChangeInRequest> selectNonSegmentedChangesForQuotaDistributionUpdate(
            final long campaignId, final long bigOrderId, final long serviceId,
            @NotNull final QuotaChangeRequest.Type type,
            @NotNull final Set<QuotaChangeRequest.Status> statuses,
            @NotNull final Set<Long> resourceIds, final boolean lock) {
        if (resourceIds.isEmpty() || statuses.isEmpty()) {
            return Collections.emptyList();
        }
        final Map<String, Object> queryParams = new HashMap<>();
        queryParams.put("campaign_id", campaignId);
        queryParams.put("order_id", bigOrderId);
        queryParams.put("service_id", serviceId);
        queryParams.put("request_type", type.name());
        final Set<String> statusInSet = new HashSet<>();
        long statusIndex = 0L;
        for (final QuotaChangeRequest.Status status : statuses) {
            statusInSet.add(String.format("CAST(:status_%d AS quota_request_status)", statusIndex));
            final String param = String.format("status_%d", statusIndex);
            queryParams.put(param, status.name());
            statusIndex++;
        }
        final String statusIn = String.join(", ", statusInSet);
        final Set<String> resourceIdInSet = new HashSet<>();
        long resourceIndex = 0L;
        for (final Long resourceId : resourceIds) {
            resourceIdInSet.add(String.format(":resource_id_%d", resourceIndex));
            final String param = String.format("resource_id_%d", resourceIndex);
            queryParams.put(param, resourceId);
            resourceIndex++;
        }
        final String resourceIdIn = String.join(", ", resourceIdInSet);
        final String query = String.format(SELECT_NON_SEGMENTED_CHANGES_FOR_UPDATE_BASE, statusIn, resourceIdIn,
                lock ? " FOR UPDATE OF qr, qrc" : "");
        final Hierarchy hierarchy = hierarchySupplier.get();
        return jdbcTemplate.query(query, queryParams, (rs, i) -> {
            final QuotaChangeInRequest.Builder builder = QuotaChangeInRequest.builder();
            final long changeId = rs.getLong("change_id");
            final long requestId = rs.getLong("request_id");
            final long resourceId = rs.getLong("resource_id");
            final long projectId = rs.getLong("project_id");
            final long amount = rs.getLong("amount");
            final long amountReady = rs.getLong("amount_ready");
            final long amountAllocated = rs.getLong("amount_allocated");
            final long amountAllocating = rs.getLong("amount_allocating");
            builder.id(changeId)
                    .requestId(requestId)
                    .resource(hierarchy.getResourceReader().read(resourceId))
                    .project(hierarchy.getProjectReader().read(projectId))
                    .amount(amount)
                    .amountReady(amountReady)
                    .amountAllocated(amountAllocated)
                    .amountAllocating(amountAllocating);
            return builder.build();
        });
    }

    @NotNull
    @Override
    public Map<Long, List<QuotaChangeRequest.Change>> selectChangesByRequestIds(@NotNull final List<Long> requestIds) {
        if (requestIds.isEmpty()) {
            return Collections.emptyMap();
        }
        final Hierarchy hierarchy = hierarchySupplier.get();
        final MapSqlParameterSource params = new MapSqlParameterSource(ImmutableMap.of("quota_request_ids", requestIds));
        return jdbcTemplate.query(SELECT_CHANGES_BY_REQUEST, params, rs -> {
            final Map<Long, List<QuotaChangeRequest.Change>> result = new HashMap<>();
            QuotaRequestChangeBuilder builder = null;
            Long lastChangeId = null;
            Set<Segment> lastSegments = new HashSet<>();
            Long lastRequestId = null;
            while (rs.next()) {
                final long changeId = rs.getLong("change_id");
                final long requestId = rs.getLong("request_id");
                if (lastChangeId == null || !Objects.equals(lastChangeId, changeId)) {
                    if (builder != null) {
                        builder.segments(lastSegments);
                        final QuotaChangeRequest.Change change = builder.build();
                        result.computeIfAbsent(lastRequestId, k -> new ArrayList<>()).add(change);
                    }
                    lastChangeId = changeId;
                    lastSegments = new HashSet<>();
                    lastRequestId = requestId;
                    builder = QuotaChangeRequest.Change.builder();
                    final long resourceId = rs.getLong("resource_id");
                    final long amount = rs.getLong("amount");
                    final long amountReady = rs.getLong("amount_ready");
                    final long amountAllocated = rs.getLong("amount_allocated");
                    final long amountAllocating = rs.getLong("amount_allocating");
                    final BigDecimal owningCost = new BigDecimal(rs.getString("owning_cost"));
                    builder.id(changeId)
                            .resource(hierarchy.getResourceReader().read(resourceId))
                            .amount(amount)
                            .amountReady(amountReady)
                            .amountAllocated(amountAllocated)
                            .amountAllocating(amountAllocating)
                            .owningCost(owningCost);
                }
                final Long segmentId = getLong(rs, "segment_id");
                if (segmentId != null) {
                    lastSegments.add(hierarchy.getSegmentReader().read(segmentId));
                }
            }
            if (builder != null) {
                builder.segments(lastSegments);
                final QuotaChangeRequest.Change change = builder.build();
                result.computeIfAbsent(lastRequestId, k -> new ArrayList<>()).add(change);
            }
            return result;
        });
    }

    @NotNull
    @Override
    public Map<Long, Pair<QuotaChangeRequest.Status, String>> selectStatusAndIssueByRequestIds(@NotNull final List<Long> requestIds) {
        if (requestIds.isEmpty()) {
            return Collections.emptyMap();
        }
        final MapSqlParameterSource params = new MapSqlParameterSource(ImmutableMap.of("ids", requestIds));
        return jdbcTemplate.query(SELECT_STATUS_AND_ISSUE_BY_REQUEST, params, rs -> {
            final Map<Long, Pair<QuotaChangeRequest.Status, String>> result = new HashMap<>();
            while (rs.next()) {
                final long id = rs.getLong("id");
                final String issueKey = rs.getString("ticket_key");
                final QuotaChangeRequest.Status status = QuotaChangeRequest.Status.valueOf(rs.getString("status"));
                result.put(id, Pair.of(status, issueKey));
            }
            return result;
        });
    }

    @Override
    public void updateRequestCost(final Set<Long> requestIds) {
        if (requestIds.isEmpty()) {
            return;
        }
        jdbcTemplate.update(UPDATE_COST_QUERY, Collections.singletonMap("id", requestIds));
    }

    @Override
    public List<QuotaChangeRequest.Change> readChangesByRequestAndProviderForUpdate(long requestId, long serviceId) {
        final Map<String, Long> params = Map.of("requestId", requestId,
                "serviceId", serviceId);
        return jdbcTemplate.query(SELECT_CHANGES_BY_REQUEST_AND_SERVICE_FOR_UPDATE, new MapSqlParameterSource(params), rs -> {
            SetMultimap<Long, Segment> segmentIdByChangeId = HashMultimap.create();
            Map<Long, QuotaRequestChangeBuilder> changeBuilderById = new HashMap<>();

            final ResourceReader resourceReader = Hierarchy.get().getResourceReader();
            final SegmentReader segmentReader = Hierarchy.get().getSegmentReader();
            while (rs.next()) {
                final long changeId = rs.getLong("change_id");
                final long resourceId = rs.getLong("resource_id");
                final long amount = rs.getLong("amount");
                final long amountReady = rs.getLong("amount_ready");
                final long amountAllocated = rs.getLong("amount_allocated");
                final long amountAllocating = rs.getLong("amount_allocating");
                final BigDecimal owningCost = new BigDecimal(rs.getString("owning_cost"));
                final Long segmentId = getLong(rs, "segment_id");
                final Resource resource = resourceReader.read(resourceId);
                final long orderId = rs.getLong("order_id");
                final LocalDate date = LocalDate.parse(rs.getString("date"), DateTimeFormatter.ISO_LOCAL_DATE);

                final QuotaRequestChangeBuilder builder = QuotaChangeRequest.Change.builder()
                        .id(changeId)
                        .resource(resource)
                        .order(new QuotaChangeRequest.BigOrder(orderId, date, true))
                        .amount(amount)
                        .amountReady(amountReady)
                        .amountAllocated(amountAllocated)
                        .amountAllocating(amountAllocating)
                        .owningCost(owningCost);

                if (segmentId != null) {
                    final Segment segment = segmentReader.read(segmentId);
                    segmentIdByChangeId.put(changeId, segment);
                }

                changeBuilderById.put(changeId, builder);
            }
            return changeBuilderById.entrySet()
                    .stream()
                    .map(e -> e.getValue().segments(segmentIdByChangeId.get(e.getKey())).build())
                    .collect(Collectors.toList());
        });
    }

    @NotNull
    @Override
    public List<QuotaChangeRequest> readRequestsByCampaignsForUpdate(Collection<Long> campaignIds, @Nullable Long fromId,
                                                                     long limit) {
        if (campaignIds.isEmpty()) {
            return List.of();
        }

        Map<String, Object> paramsMap = new HashMap<>();
        paramsMap.put("ids", campaignIds);
        paramsMap.put("fromId", fromId);
        paramsMap.put("limit", limit);

        return query(SELECT_REQUESTS_BY_CAMPAIGN_IDS_FOR_UPDATE, paramsMap).collect(Collectors.toList());
    }

    @Override
    public List<Tuple2<Long, Long>> readRequestOwningCostByCampaignId(@Nullable Long fromId, Long campaignId, long limit,
                                                                      Set<QuotaChangeRequest.Status> validStatuses) {
        if (validStatuses.isEmpty()) {
            return List.of();
        }

        Map<String, Object> paramsMap = new HashMap<>();
        paramsMap.put("id", campaignId);
        paramsMap.put("fromId", fromId);
        paramsMap.put("limit", limit);

        RequestCollector<Stream<Tuple2<Long, Long>>> streamRequestCollector = new RequestCollector<>() {
            private final List<Tuple2<Long, Long>> requestOwningCost = new ArrayList<>();

            @Override
            public void processRow(ResultSet rs) throws SQLException {
                long id = rs.getLong("id");
                long requestOwningCost = rs.getLong("request_owning_cost");
                this.requestOwningCost.add(Tuple2.tuple(id, requestOwningCost));
            }

            @NotNull
            @Override
            public Stream<Tuple2<Long, Long>> get() {
                return requestOwningCost.stream();
            }
        };

        String statusParamStatement = getStatusParamStatement(validStatuses, paramsMap);
        return query(String.format(SELECT_REQUESTS_OWNING_COST_BY_CAMPAIGN_ID, statusParamStatement), paramsMap, streamRequestCollector)
                .collect(Collectors.toList());
    }

    @Override
    public List<Long> getIdsFirstPageByCampaigns(Collection<? extends Long> campaignIds, long limit) {
        if (campaignIds.isEmpty()) {
            return List.of();
        }
        return jdbcTemplate.query(GET_IDS_FIRST_PAGE_BY_CAMPAIGNS,
                Map.of("campaignIds", campaignIds, "limit", limit), (rs, i) -> rs.getLong("id"));
    }

    @Override
    public List<Long> getIdsNextPageByCampaigns(Collection<? extends Long> campaignIds, long from, long limit) {
        if (campaignIds.isEmpty()) {
            return List.of();
        }
        return jdbcTemplate.query(GET_IDS_NEXT_PAGE_BY_CAMPAIGNS,
                Map.of("campaignIds", campaignIds, "fromId", from, "limit", limit),
                (rs, i) -> rs.getLong("id"));
    }

    private SqlFilters makeFilters(final QuotaChangeRequestFilter filter, final Map<String, Object> queryParams) {
        return makeFilters(filter, queryParams, false);
    }

    private SqlFilters makeFilters(final QuotaChangeRequestFilter filter, final Map<String, Object> queryParams, final boolean filterChanges) {
        final SqlFilters sqlFilters = new SqlFilters();

        final Set<Person> authors = filter.getAuthors();
        if (!authors.isEmpty()) {
            queryParams.put("authorId", ids(authors));
            sqlFilters.addCondition("quota_request.author_id IN (:authorId)");
        }

        final Set<QuotaChangeRequest.Status> statuses = filter.getStatus();
        if (!statuses.isEmpty()) {
            final Set<String> statusNames = statuses.stream().map(Enum::name).collect(Collectors.toSet());
            int counter = 0;
            final List<String> statusParams = new ArrayList<>(statusNames.size());
            for (final String statusName : statusNames) {
                final String key = "status_" + counter;
                queryParams.put(key, statusName);
                statusParams.add("cast (:" + key + " as quota_request_status)");
                counter += 1;
            }

            sqlFilters.addCondition("quota_request.status IN (" + String.join(", ", statusParams) + ")");
        }

        final Set<QuotaChangeRequest.Type> types = filter.getType();
        if (!types.isEmpty()) {
            final Set<String> typeNames = types.stream().map(Enum::name).collect(Collectors.toSet());
            int counter = 0;
            final List<String> statusParams = new ArrayList<>(typeNames.size());
            for (final String typeName : typeNames) {
                final String key = "type" + counter;
                queryParams.put(key, typeName);
                statusParams.add("cast (:" + key + " as quota_request_type)");
                counter += 1;
            }

            sqlFilters.addCondition("quota_request.type IN (" + String.join(", ", statusParams) + ")");
        }

        final Set<Project> projects = filter.getProjects();
        if (!projects.isEmpty()) {
            queryParams.put("projectId", ids(projects));
            sqlFilters.addCondition("quota_request.project_id IN (:projectId)");
        }

        final Set<Long> campaignIds = filter.getCampaignIds();
        if (!campaignIds.isEmpty()) {
            queryParams.put("campaign_ids", campaignIds);
            sqlFilters.addCondition("quota_request.campaign_id IN (:campaign_ids)");
        }
        addGoalFilter(filter, queryParams, sqlFilters);
        addReasonTypeFilter(filter, queryParams, sqlFilters);

        final Set<Long> excludedChangeRequestIds = filter.getExcludedChangeRequestIds();
        if (!excludedChangeRequestIds.isEmpty()) {
            queryParams.put("excluded_ids", excludedChangeRequestIds);
            sqlFilters.addCondition("quota_request.id NOT IN (:excluded_ids)");
        }

        if (filter.withoutTicket().isPresent()) {
            sqlFilters.addCondition("quota_request.ticket_key " + (filter.withoutTicket().get() ? "IS" : "NOT IS") + " NULL");
        }

        final SqlFilters subQueryFilters = new SqlFilters();

        final Set<Service> services = filter.getServices();
        if (!services.isEmpty()) {
            queryParams.put("serviceId", ids(services));
            final SqlFilters targetFilters = filterChanges ? sqlFilters : subQueryFilters;

            targetFilters.addCondition("r.service_id IN (:serviceId)");
            targetFilters.addJoin("JOIN resource r ON r.id = quota_request_change.resource_id");
        }

        final Set<Long> orderIds = filter.getOrderIds();
        if (!orderIds.isEmpty()) {
            queryParams.put("orderId", orderIds);
            final SqlFilters targetFilters = filterChanges ? sqlFilters : subQueryFilters;

            targetFilters.addCondition("quota_request_change.order_id IN (:orderId)");
        }

        final Set<Long> preOrderIds = filter.getPreOrderIds();
        if (!preOrderIds.isEmpty()) {
            queryParams.put("preOrderIds", preOrderIds);

            final SqlFilters targetFilters = filterChanges ? sqlFilters : subQueryFilters;

            targetFilters.addCondition("bpc.bot_pre_order_id IN (:preOrderIds)");
            targetFilters.addJoin("JOIN bot_pre_order_change bpc ON bpc.quota_request_change_id = quota_request_change.id");
        }

        if (filterChanges) {
            addCampaignOrderFilter(filter, queryParams, sqlFilters);
        } else {
            addCampaignOrderFilter(filter, queryParams, subQueryFilters);
        }

        final String subQueryPart = conditionsToWhere(subQueryFilters);
        if (!subQueryPart.isEmpty()) {
            sqlFilters.addCondition("quota_request.id IN (SELECT quota_request_change.quota_request_id FROM quota_request_change" + subQueryPart + ")");
        }

        final Set<Long> changeRequestIds = filter.getChangeRequestIds();
        if (!changeRequestIds.isEmpty()) {
            sqlFilters.addCondition("quota_request.id IN (:quotaRequestIds)");
            queryParams.put("quotaRequestIds", changeRequestIds);
        }

        final Long owningCostGreaterOrEquals = filter.getOwningCostGreaterOrEquals();
        if (owningCostGreaterOrEquals != null) {
            sqlFilters.addCondition("quota_request.request_owning_cost >= :owningCostGreaterOrEquals");
            queryParams.put("owningCostGreaterOrEquals", owningCostGreaterOrEquals);
        }

        final Long owningCostLessOrEquals = filter.getOwningCostLessOrEquals();
        if (owningCostLessOrEquals != null) {
            sqlFilters.addCondition("quota_request.request_owning_cost <= :owningCostLessOrEquals");
            queryParams.put("owningCostLessOrEquals", owningCostLessOrEquals);
        }

        final DiQuotaChangeRequestImportantFilter importantFilter = filter.getImportantFilter();
        if (importantFilter != null) {
            switch (importantFilter) {
                case IMPORTANT:
                case NOT_IMPORTANT:
                    sqlFilters.addCondition("quota_request.important_request = :importantFilter");
                    queryParams.put("importantFilter", importantFilter == DiQuotaChangeRequestImportantFilter.IMPORTANT);
                    break;
                case BOTH:
                    break;
                default:
                    throw new IllegalStateException("Unsupported importance filter!");
            }
        }

        final DiQuotaChangeRequestUnbalancedFilter unbalancedFilter = filter.getUnbalancedFilter();
        if (unbalancedFilter != null) {
            switch (unbalancedFilter) {
                case BALANCED:
                case UNBALANCED:
                    sqlFilters.addCondition("quota_request.unbalanced = :unbalanced");
                    queryParams.put("unbalanced", unbalancedFilter == DiQuotaChangeRequestUnbalancedFilter.UNBALANCED);
                    break;
                case BOTH:
                    break;
                default:
                    throw new IllegalStateException("Unsupported unbalanced filter!");
            }
        }

        Set<Campaign.Type> campaignTypes = filter.getCampaignTypes();
        if (!campaignTypes.isEmpty()) {
            final Set<String> campaignTypeNames = campaignTypes.stream().map(Enum::name).collect(Collectors.toSet());
            int campaignTypeCounter = 0;
            final List<String> campaignTypeParams = new ArrayList<>(campaignTypeNames.size());
            for (final String campaignTypeName : campaignTypeNames) {
                final String campaignTypeKey = "campaignType" + campaignTypeCounter;
                queryParams.put(campaignTypeKey, campaignTypeName);
                campaignTypeParams.add("cast (:" + campaignTypeKey + " as campaign_type)");
                campaignTypeCounter += 1;
            }

            sqlFilters.addCondition("quota_request.campaign_type IN (" + String.join(", ", campaignTypeParams) + ")");
        }

        addSummaryFilter(filter, queryParams, sqlFilters);

        return sqlFilters;
    }

    private void addSummaryFilter(final QuotaChangeRequestFilter filter,
                                  final Map<String, Object> queryParams, final SqlFilters sqlFilters) {
        filter.getSummary().ifPresent(summary -> {
            final String trimmedQuery = summary.trim();
            if (trimmedQuery.isEmpty()) {
                return;
            }
            sqlFilters.addCondition("(quota_request.summary IS NOT NULL AND (quota_request.summary = :summary_full " +
                    "OR to_tsvector('english', quota_request.summary) @@ phraseto_tsquery('english', :summary_en) " +
                    "OR to_tsvector('russian', quota_request.summary) @@ phraseto_tsquery('russian', :summary_ru)))");
            queryParams.put("summary_full", summary);
            queryParams.put("summary_en", summary);
            queryParams.put("summary_ru", summary);
        });
    }

    private void addCampaignOrderFilter(final QuotaChangeRequestFilter filter,
                                        final Map<String, Object> queryParams, final SqlFilters sqlFilters) {
        final Set<Long> campaignOrderIds = filter.getCampaignOrderIds();
        if (!campaignOrderIds.isEmpty()) {
            final List<Campaign.CampaignOrder> campaignOrders = campaignDao.getCampaignOrders(campaignOrderIds);
            if (campaignOrders.size() != campaignOrderIds.size()) {
                final Set<Long> actualCampaignOrderIds = campaignOrders.stream().map(Campaign.CampaignOrder::getId).collect(Collectors.toSet());
                final Set<Long> missingCampaignOrderIds = Sets.difference(campaignOrderIds, actualCampaignOrderIds);
                throw new IllegalArgumentException("Invalid campaign order ids: " + missingCampaignOrderIds.stream()
                        .map(Object::toString).collect(Collectors.joining(", ")));
            }
            sqlFilters.addJoin("JOIN quota_request qr2 ON qr2.id = quota_request_change.quota_request_id");
            final StringBuilder conditionBuilder = new StringBuilder();
            conditionBuilder.append("(");
            for (int i = 0; i < campaignOrders.size(); i++) {
                queryParams.put("order_id_" + i, campaignOrders.get(i).getBigOrderId());
                queryParams.put("campaign_id_" + i, campaignOrders.get(i).getCampaignId());
                if (i > 0) {
                    conditionBuilder.append(" OR ");
                }
                conditionBuilder.append("(quota_request_change.order_id = :order_id_");
                conditionBuilder.append(i);
                conditionBuilder.append(" AND qr2.campaign_id = :campaign_id_");
                conditionBuilder.append(i);
                conditionBuilder.append(")");
            }
            conditionBuilder.append(")");
            sqlFilters.addCondition(conditionBuilder.toString());
        }
    }

    private void addGoalFilter(final QuotaChangeRequestFilter filter,
                               final Map<String, Object> queryParams, final SqlFilters filters) {
        final Set<Long> goalIds = filter.getGoalIds();
        if (goalIds.isEmpty()) {
            return;
        }
        filters.addCondition("quota_request.goal_id in (:goal_id)");
        queryParams.put("goal_id", goalIds);
    }

    private void addReasonTypeFilter(final QuotaChangeRequestFilter filter,
                                     final Map<String, Object> queryParams, final SqlFilters filters) {
        final Set<DiResourcePreorderReasonType> reasonTypes = filter.getReasonTypes();
        if (reasonTypes.isEmpty()) {
            return;
        }

        final Set<String> reasonTypeNames = reasonTypes.stream().map(Enum::name).collect(Collectors.toSet());
        int counter = 0;
        final List<String> reasonStatusParams = new ArrayList<>(reasonTypeNames.size());
        for (final String typeName : reasonTypeNames) {
            final String key = "reasonType" + counter;
            queryParams.put(key, typeName);
            reasonStatusParams.add("cast (:" + key + " as resource_preorder_reason_type)");
            counter += 1;
        }

        filters.addCondition("quota_request.resource_preorder_reason_type IN (" + String.join(", ", reasonStatusParams) + ")");
    }

    @Override
    @NotNull
    public List<RequestPreOrderAggregationEntry> readPreOrderAggregation(final QuotaChangeRequestFilter filter) {
        final HashMap<String, Object> queryParams = new HashMap<>();
        final String where = conditionsToWhere(makeFilters(filter, queryParams));

        final String idsSubQuery = GET_IDS + where;
        return jdbcTemplate.query(String.format(REPORT_AGGREGATION_QUERY, idsSubQuery), queryParams,
                (rs, i) -> new RequestPreOrderAggregationEntry(
                        rs.getLong("project_id"), rs.getLong("service_id"),
                        rs.getLong("server_id"), rs.getDouble("total_server_quantity"),
                        rs.getDouble("total_cost"))
        );
    }

    private static class SqlFilters {
        private final List<String> conditions = new ArrayList<>();
        private final List<String> joins = new ArrayList<>();

        public void addCondition(final String condition) {
            conditions.add(condition);
        }

        public void addJoin(final String join) {
            joins.add(join);
        }

        public List<String> getConditions() {
            return conditions;
        }

        public List<String> getJoins() {
            return joins;
        }

    }

    private String conditionsToWhere(final SqlFilters filters) {
        final StringBuilder whereBuilder = new StringBuilder();
        final List<String> joins = filters.getJoins();
        if (!joins.isEmpty()) {
            whereBuilder.append(" ")
                    .append(String.join(" ", joins));
        }
        final List<String> conditions = filters.getConditions();
        if (!conditions.isEmpty()) {
            whereBuilder.append(" WHERE ")
                    .append(String.join(" AND ", conditions));
        }

        return whereBuilder.toString();
    }
}
