package ru.yandex.qe.dispenser.domain.dao.quota.spec;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.api.v1.DiQuotaType;
import ru.yandex.qe.dispenser.domain.QuotaSpec;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;

public class SqlQuotaSpecDao extends SqlDaoBase implements QuotaSpecDao {
    private static final String SELECT_ALL_QUERY = "SELECT * FROM quota_spec";
    private static final String DELETE_QUERY = "DELETE FROM quota_spec WHERE id = :id";
    private static final String CLEAR_QUERY = "TRUNCATE quota_spec CASCADE";
    private static final String SELECT_BY_ID_QUERY = "SELECT * FROM quota_spec WHERE id = :id";
    private static final String SELECT_BY_KEY_QUERY = "SELECT * FROM quota_spec WHERE key = :key AND resource_id = (:resourceId)";
    private static final String CREATE_QUERY = "INSERT INTO quota_spec (key, description, resource_id, value_type) VALUES (:key, :description, :resourceId, CAST(:valueType AS quota_value_type))";
    private static final String UPDATE_QUERY = "UPDATE quota_spec SET key = :key, description = :description, resource_id = :resourceId, value_type = CAST(:valueType AS quota_value_type) WHERE id = :id";

    @Autowired
    @Qualifier("quotaDao")
    private QuotaDao quotaDao;

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public Set<QuotaSpec> getAll() {
        return new HashSet<>(jdbcTemplate.query(SELECT_ALL_QUERY, this::toQuota));
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public @NotNull QuotaSpec create(final @NotNull QuotaSpec newQuotaSpec) {
        jdbcTemplate.update(CREATE_QUERY, toParams(newQuotaSpec));
        final QuotaSpec quotaSpec = read(newQuotaSpec.getKey());
        quotaDao.createZeroQuotasFor(quotaSpec);
        return quotaSpec;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public @NotNull QuotaSpec read(final @NotNull Long id) throws EmptyResultDataAccessException {
        return jdbcTemplate.queryForOptional(SELECT_BY_ID_QUERY, ImmutableMap.of("id", id), this::toQuota)
                .orElseThrow(() -> new EmptyResultDataAccessException("No quota spec with id " + id, 1));
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public @NotNull QuotaSpec read(@NotNull final QuotaSpec.Key key) throws EmptyResultDataAccessException {
        final Map<String, ?> params = ImmutableMap.of("key", key.getPublicKey(), "resourceId", key.getResource().getId());
        return jdbcTemplate.queryForOptional(SELECT_BY_KEY_QUERY, params, this::toQuota)
                .orElseThrow(() -> new EmptyResultDataAccessException("No quota spec with key " + key, 1));
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean update(final @NotNull QuotaSpec trainsientObject) {
        return jdbcTemplate.update(UPDATE_QUERY, toParams(trainsientObject)) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean delete(final @NotNull QuotaSpec quotaSpec) {
        if (jdbcTemplate.update(DELETE_QUERY, toParams(quotaSpec)) > 0) {
            quotaDao.deleteAll(quotaDao.getQuotas(quotaSpec));
            return true;
        }
        return false;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean clear() {
        return jdbcTemplate.update(CLEAR_QUERY) > 0;
    }

    @Override
    public void lockForChanges() {
        acquireRowExclusiveLockOnTable("quota_spec");
    }

    @NotNull
    private QuotaSpec toQuota(@NotNull final ResultSet rs, final int i) throws SQLException {
        final Resource resource = Hierarchy.get().getResourceReader().read(rs.getLong("resource_id"));
        final QuotaSpec quotaSpec = new QuotaSpec.Builder(rs.getString("key"), resource)
                .description(rs.getString("description"))
                .type(DiQuotaType.valueOf(rs.getString("value_type")))
                .build();
        quotaSpec.setId(rs.getLong("id"));
        return quotaSpec;
    }

    @NotNull
    private Map<String, Object> toParams(@NotNull final QuotaSpec quotaSpec) {
        final Map<String, Object> parameters = new HashMap<>();
        parameters.put("key", quotaSpec.getKey().getPublicKey());
        parameters.put("id", quotaSpec.getId());
        parameters.put("description", quotaSpec.getDescription());
        parameters.put("valueType", quotaSpec.getType().toString());
        parameters.put("resourceId", quotaSpec.getKey().getResource().getId());
        return parameters;
    }
}
