package ru.yandex.qe.dispenser.domain.dao.resource.group;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.ResourceGroup;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.dao.service.SqlServiceDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.util.MapBuilder;

@ParametersAreNonnullByDefault
public class SqlResourceGroupDao extends SqlDaoBase implements ResourceGroupDao {
    private static final String GET_ALL_QUERY = "SELECT * FROM resource_group";
    private static final String CREATE_QUERY = "INSERT INTO resource_group (key, name, service_id, priority) VALUES (:key, :name, :serviceId, :priority)";
    private static final String UPDATE_QUERY = "UPDATE resource_group SET name = :name, priority = :priority WHERE id = :resourceGroupId";
    private static final String DELETE_QUERY = "DELETE FROM resource_group WHERE id = :resourceGroupId";
    private static final String READ_QUERY = "SELECT * FROM resource_group where id = :id";
    private static final String GET_BY_SERVICE_ID_QUERY = "SELECT * FROM resource_group WHERE service_id = :serviceId";

    @Override
    @NotNull
    public Set<ResourceGroup> getAll() {
        return jdbcTemplate.queryForSet(GET_ALL_QUERY, this::toResourceGroup);
    }


    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public ResourceGroup create(@NotNull final ResourceGroup group) {
        final long id = jdbcTemplate.insert(CREATE_QUERY, toParams(group));
        group.setId(id);
        return group;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean update(@NotNull final ResourceGroup group) {
        return jdbcTemplate.update(UPDATE_QUERY, toParams(group)) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean delete(@NotNull final ResourceGroup group) {
        return jdbcTemplate.update(DELETE_QUERY, toParams(group)) > 0;
    }

    @NotNull
    @Override
    public ResourceGroup read(@NotNull final Long id) throws EmptyResultDataAccessException {
        return jdbcTemplate.queryForOptional(READ_QUERY, Collections.singletonMap("id", id), this::toResourceGroup)
                .orElseThrow(() -> new EmptyResultDataAccessException("No resource group with id " + id, 1));
    }

    @Override
    public Set<ResourceGroup> read(final Service service) {
        return jdbcTemplate.queryForSet(GET_BY_SERVICE_ID_QUERY, SqlServiceDao.toParams(service), this::toResourceGroup);
    }

    private ResourceGroup toResourceGroup(final ResultSet resultSet, final int i) throws SQLException {
        final Service service = Hierarchy.get().getServiceReader().read(resultSet.getLong("service_id"));

        return new ResourceGroup.Builder(resultSet.getLong("id"), resultSet.getString("key"), service)
                .name(resultSet.getString("name"))
                .priority(getInteger(resultSet, "priority"))
                .build();
    }

    public static Map<String, Object> toParams(final ResourceGroup resourceGroup) {
        return MapBuilder.<String, Object>of("resourceGroupId", resourceGroup.getId())
                .put("key", resourceGroup.getKey().getPublicKey())
                .put("name", resourceGroup.getName())
                .put("serviceId", resourceGroup.getKey().getService().getId())
                .put("priority", resourceGroup.getPriority())
                .build();
    }
}
