package ru.yandex.qe.dispenser.domain.dao.resource.segmentation;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.ResourceSegmentation;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;

@ParametersAreNonnullByDefault
public class SqlResourceSegmentationDao extends SqlDaoBase implements ResourceSegmentationDao {
    private static final String TABLE_NAME = "resource_segmentation";
    private static final String GET_ALL_QUERY = "SELECT * FROM resource_segmentation";
    private static final String GET_RESOURCE_SEGMENTATION_QUERY = "SELECT * FROM resource_segmentation WHERE resource_id IN (:resourceId)";
    private static final String GET_QUERY = "SELECT * FROM resource_segmentation WHERE id = :resourceSegmentationId";
    private static final String CREATE_QUERY = "INSERT INTO resource_segmentation (resource_id, segmentation_id) VALUES (:resourceId, :segmentationId)";
    private static final String DELETE_RESOURCE_SEGMENTATION_QUERY = "DELETE FROM resource_segmentation WHERE resource_id IN (:resourceId)";
    private static final String DELETE_QUERY = "DELETE FROM resource_segmentation WHERE id IN (:resourceSegmentationId)";
    private static final String DELETE_ALL_QUERY = "DELETE FROM resource_segmentation";

    @Override
    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public ResourceSegmentation read(final Long id) throws EmptyResultDataAccessException {
        return jdbcTemplate.queryForOptional(GET_QUERY, Collections.singletonMap("resourceSegmentationId", id), this::toResourceSegmentation)
                .orElseThrow(() -> new EmptyResultDataAccessException("No resource segmentation with id " + id, 1));
    }

    @Override
    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public ResourceSegmentation create(final ResourceSegmentation resourceSegmentation) {
        final long id = jdbcTemplate.insert(CREATE_QUERY, toParams(resourceSegmentation));
        return read(id);
    }


    @Override
    public boolean update(@NotNull final ResourceSegmentation trainsientObject) {
        return false;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean delete(@NotNull final ResourceSegmentation resourceSegmentation) {
        return jdbcTemplate.update(DELETE_QUERY, toParams(resourceSegmentation)) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean clear() {
        return jdbcTemplate.update(DELETE_ALL_QUERY) > 0;
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public Set<ResourceSegmentation> getAll() {
        return jdbcTemplate.queryForSet(GET_ALL_QUERY, this::toResourceSegmentation);
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public void clearSegmentations(final Resource resource) {
        jdbcTemplate.update(DELETE_RESOURCE_SEGMENTATION_QUERY, toParams(resource));
    }

    @Override
    public void lockForChanges() {
        acquireRowExclusiveLockOnTable(TABLE_NAME);
    }

    @Override
    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public Set<ResourceSegmentation> getResourceSegmentations(final Resource resource) {
        return getResourceSegmentations(Collections.singleton(resource));
    }

    @Override
    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public Set<ResourceSegmentation> getResourceSegmentations(final Collection<Resource> resources) {
        if (resources.isEmpty()) {
            return Collections.emptySet();
        }
        final List<Long> ids = resources.stream()
                .map(Resource::getId)
                .collect(Collectors.toList());
        return jdbcTemplate.queryForSet(GET_RESOURCE_SEGMENTATION_QUERY, Collections.singletonMap("resourceId", ids), this::toResourceSegmentation);
    }

    private ResourceSegmentation toResourceSegmentation(final ResultSet rs, final int i) throws SQLException {
        final Resource resource = Hierarchy.get().getResourceReader().read(rs.getLong("resource_id"));
        final Segmentation segmentation = Hierarchy.get().getSegmentationReader().read(rs.getLong("segmentation_id"));
        final ResourceSegmentation resourceSegmentation = new ResourceSegmentation.Builder(resource, segmentation)
                .build();
        resourceSegmentation.setId(rs.getLong("id"));
        return resourceSegmentation;
    }

    private Map<String, Object> toParams(final ResourceSegmentation resourceSegmentation) {
        return ImmutableMap.<String, Object>builder()
                .put("resourceSegmentationId", resourceSegmentation.getId())
                .put("resourceId", resourceSegmentation.getResource().getId())
                .put("segmentationId", resourceSegmentation.getSegmentation().getId())
                .build();
    }
}
