package ru.yandex.qe.dispenser.domain.dao.resources_model;

import java.math.RoundingMode;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentReader;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.resources_model.ResourceModelMappingTarget;
import ru.yandex.qe.dispenser.domain.resources_model.ResourcesModelMapping;

import static ru.yandex.qe.dispenser.domain.util.CollectionUtils.ids;

public class SqlResourcesModelMappingDao extends SqlDaoBase implements ResourcesModelMappingDao {

    private static final String GET_BY_RESOURCE_QUERY = "SELECT * FROM resources_model_mapping" +
            " WHERE resource_id IN (:resourceId) AND campaign_id = :campaignId";

    private static final String GET_BY_PROVIDER_AND_CAMPAIGN_QUERY = "SELECT rmm.* FROM resources_model_mapping rmm" +
            " JOIN resource r ON rmm.resource_id = r.id WHERE r.service_id = :serviceId" +
            " AND rmm.campaign_id = :campaignId";

    private static final String GET_BY_CAMPAIGN_QUERY = "SELECT * FROM resources_model_mapping" +
            " WHERE campaign_id = :campaignId";

    private static final String GET_BY_CAMPAIGNS_QUERY = "SELECT * FROM resources_model_mapping" +
            " WHERE campaign_id IN (:campaignIds)";

    private static final String INSERT_QUERY = "INSERT INTO resources_model_mapping" +
            " (resource_id, segment_ids, campaign_id, numerator, denominator, " +
            "external_resource_id, external_resource_unit_key, skip, key, rounding)" +
            " VALUES (:resourceId, :segmentIds, :campaignId, :numerator, :denominator, " +
            ":externalResourceId, :externalResourceUnitKey, :skip, :key," +
            "cast(:rounding as resource_model_mapping_rounding)) RETURNING id";

    private static final String CLEAR_QUERY = "TRUNCATE resources_model_mapping";

    @Autowired
    private HierarchySupplier hierarchySupplier;

    @Override
    public List<ResourcesModelMapping> getResourcesMappings(Set<Resource> resources, long campaignId) {
        if (resources.isEmpty()) {
            return List.of();
        }

        Hierarchy hierarchy = hierarchySupplier.get();

        return jdbcTemplate.query(GET_BY_RESOURCE_QUERY, Map.of("resourceId", ids(resources), "campaignId", campaignId),
                (rs, i) -> toRow(rs, hierarchy));
    }

    @Override
    public List<ResourcesModelMapping> getResourcesMappingsForProvider(Service service, long campaignId) {
        final Hierarchy hierarchy = hierarchySupplier.get();
        return jdbcTemplate.query(GET_BY_PROVIDER_AND_CAMPAIGN_QUERY,
                Map.of("serviceId", service.getId(), "campaignId", campaignId),
                (rs, i) -> toRow(rs, hierarchy));
    }

    @Override
    public List<ResourcesModelMapping> getResourcesMappingsForCampaign(long campaignId) {
        final Hierarchy hierarchy = hierarchySupplier.get();
        return jdbcTemplate.query(GET_BY_CAMPAIGN_QUERY, Map.of("campaignId", campaignId), (rs, i) -> toRow(rs, hierarchy));
    }

    @Override
    public List<ResourcesModelMapping> getResourcesMappingsForCampaigns(Set<Long> campaigns) {
        if (campaigns.isEmpty()) {
            return Collections.emptyList();
        }
        final Hierarchy hierarchy = hierarchySupplier.get();
        return jdbcTemplate.query(GET_BY_CAMPAIGNS_QUERY, Map.of("campaignIds", campaigns), (rs, i) -> toRow(rs, hierarchy));
    }

    @Override
    public ResourcesModelMapping createResourceMapping(ResourcesModelMapping.Builder mapping) {
        long id = jdbcTemplate.queryForObject(INSERT_QUERY, toParams(mapping), Long.class);
        return mapping.build(id);
    }

    @Override
    public void clear() {
        jdbcTemplate.update(CLEAR_QUERY);
    }

    private ResourcesModelMapping toRow(ResultSet rs, Hierarchy hierarchy) throws SQLException {
        Resource resource = hierarchy.getResourceReader().read(rs.getLong("resource_id"));
        SegmentReader segmentReader = hierarchy.getSegmentReader();
        Set<Segment> segments = Arrays.stream((Long[]) rs.getArray("segment_ids").getArray())
                .map(segmentReader::read)
                .collect(Collectors.toSet());
        boolean skip = rs.getBoolean("skip");
        if (skip) {
            return ResourcesModelMapping.builder()
                    .id(rs.getLong("id"))
                    .resource(resource)
                    .addSegments(segments)
                    .campaignId(rs.getLong("campaign_id"))
                    .rounding(RoundingMode.valueOf(rs.getString("rounding")))
                    .build();
        } else {
            return ResourcesModelMapping.builder()
                    .id(rs.getLong("id"))
                    .resource(resource)
                    .addSegments(segments)
                    .campaignId(rs.getLong("campaign_id"))
                    .target(ResourceModelMappingTarget.builder()
                            .numerator(rs.getLong("numerator"))
                            .denominator(rs.getLong("denominator"))
                            .externalResourceId(UUID.fromString(rs.getString("external_resource_id")))
                            .externalResourceBaseUnitKey(rs.getString("external_resource_unit_key"))
                            .key(rs.getString("key"))
                            .build())
                    .rounding(RoundingMode.valueOf(rs.getString("rounding")))
                    .build();
        }
    }

    private Map<String, ?> toParams(ResourcesModelMapping.Builder mapping) {
        long[] segmentIds = mapping.getSegments()
                .stream()
                .mapToLong(LongIndexBase::getId)
                .toArray();
        HashMap<String, Object> params = new HashMap<>();
        params.put("resourceId", mapping.getResource()
                .orElseThrow(() -> new NoSuchElementException("ResourceId is required")).getId());
        params.put("segmentIds", segmentIds);
        params.put("campaignId", mapping.getCampaignId()
                .orElseThrow(() -> new NoSuchElementException("CampaignId is required")));
        params.put("rounding", mapping.getRounding() != null ?
                mapping.getRounding().name() : RoundingMode.HALF_UP.name());
        if (mapping.getTarget().isPresent()) {
            params.put("numerator", mapping.getTarget().get().getNumerator());
            params.put("denominator", mapping.getTarget().get().getDenominator());
            params.put("externalResourceId", mapping.getTarget().get().getExternalResourceId());
            params.put("externalResourceUnitKey", mapping.getTarget().get().getExternalResourceBaseUnitKey());
            params.put("key", mapping.getTarget().get().getKey());
            params.put("skip", false);
        } else {
            params.put("numerator", null);
            params.put("denominator", null);
            params.put("externalResourceId", null);
            params.put("externalResourceUnitKey", null);
            params.put("key", null);
            params.put("skip", true);
        }
        return params;
    }

}
