package ru.yandex.qe.dispenser.domain.dao.segment;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.dao.InMemoryLongKeyDaoImpl;

@ParametersAreNonnullByDefault
public class SegmentDaoImpl extends InMemoryLongKeyDaoImpl<Segment> implements SegmentDao {

    @NotNull
    @Override
    public Set<Segment> get(final Segmentation segmentation) {
        return filter(segment -> segment.getKey().getSegmentation().equals(segmentation))
                .collect(Collectors.toSet());
    }

    @NotNull
    @Override
    public Map<Long, Set<Segment>> getBySegmentationIds(@NotNull final Set<Long> segmentationIds) {
        if (segmentationIds.isEmpty()) {
            return Collections.emptyMap();
        }
        final Map<Long, Set<Segment>> result = filter(segment -> segmentationIds.contains(segment.getSegmentation().getId()))
                .collect(Collectors.groupingBy(s -> s.getSegmentation().getId(), HashMap::new, Collectors.toSet()));
        segmentationIds.forEach(id -> result.computeIfAbsent(id, k -> new HashSet<>()));
        return result;
    }

    @Override
    @NotNull
    public Segment read(final String publicKey) {
        return filter(segment -> segment.getKey().getPublicKey().equals(publicKey))
                .findFirst()
                .orElseThrow(() -> new EmptyResultDataAccessException("No segment with key '" + publicKey + "'", 1));
    }

    @NotNull
    @Override
    public Segment readForUpdate(final Long id) {
        try {
            return this.read(id);
        } catch (EmptyResultDataAccessException ignore) {
            throw new EmptyResultDataAccessException("No segment with id " + id, 1);
        }
    }

    @NotNull
    @Override
    public Segment create(@NotNull final Segment newInstance) {
        if (newInstance.getPriority() == null) {
            throw new NullPointerException("null value in entry: priority=null");
        }
        return super.create(newInstance);
    }

    @NotNull
    @Override
    public Short getMaxPriority(final Segmentation segmentation) {
        return (short) filter(segment -> segment.getKey().getSegmentation().equals(segmentation))
                .mapToInt(Segment::getPriority)
                .max()
                .orElse(0);
    }
}
