package ru.yandex.qe.dispenser.domain.dao.segment;

import java.util.Collection;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.exception.SingleMessageException;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.util.MoreCollectors;

public final class SegmentUtils {
    private SegmentUtils() {
    }

    private static void checkSegmentInSegmentations(@NotNull final Set<Segmentation> segmentations,
                                                    @NotNull final Segment segment) {
        if (!segmentations.contains(segment.getSegmentation())) {
            throw SingleMessageException.illegalArgument("invalid.segmentation.with.available",
                    segment.getSegmentation().getKey().getPublicKey(),
                    segment.getKey().getPublicKey(),
                    segmentations.stream()
                            .map(segmentation -> segmentation.getKey().getPublicKey())
                            .collect(Collectors.joining("\", \"")));
        }
    }

    @NotNull
    public static Set<Segment> getCompleteSegmentSet(@NotNull final Resource resource, @NotNull final Set<String> segmentKeys) {
        final Set<Segmentation> allSegmentations = Hierarchy.get()
                .getResourceSegmentationReader()
                .getSegmentations(resource);

        if (!segmentKeys.isEmpty() && allSegmentations.isEmpty()) {
            throw SingleMessageException.illegalArgument("no.segmentation.available.for.resource", resource.getKey().getPublicKey());
        }

        final Set<Segment> segments = segmentKeys.stream()
                .map(key -> Hierarchy.get().getSegmentReader().read(key))
                .peek(segment -> checkSegmentInSegmentations(allSegmentations, segment))
                .collect(Collectors.toSet());

        final Set<Segmentation> specifiedSegmentations = segments.stream()
                .map(Segment::getSegmentation)
                .collect(Collectors.toSet());

        Sets.difference(allSegmentations, specifiedSegmentations)
                .stream()
                .map(Segment::totalOf)
                .forEach(segments::add);

        final Multimap<Segmentation, Segment> segmentsBySegmentation = segments.stream()
                .collect(MoreCollectors.toLinkedMultimap(Segment::getSegmentation, Function.identity()));
        segmentsBySegmentation.keySet().forEach(segmentation -> {
            final Collection<Segment> segmentationSegments = segmentsBySegmentation.get(segmentation);
            if (segmentationSegments.size() > 1) {
                throw SingleMessageException.illegalArgument("invalid.segment.combination", segmentKeys, segmentation.getKey().getPublicKey());
            }
        });

        return segments;
    }

    @NotNull
    public static Set<Segment> getNonAggregationCompleteSegmentSet(@NotNull final Resource resource, @NotNull final Set<String> segmentKeys) {
        final Set<Segment> segments = getCompleteSegmentSet(resource, segmentKeys);

        final boolean hasAggregationSegments = segments.stream()
                .anyMatch(Segment::isAggregationSegment);

        if (hasAggregationSegments) {
            throw SingleMessageException.illegalArgument("invalid.segment.set");
        }

        return segments;
    }


    @NotNull
    public static Set<String> getNonAggregationSegmentKeys(@NotNull final Collection<Segment> segments) {
        return segments.stream()
                .filter(segment -> !segment.isAggregationSegment())
                .map(segment -> segment.getKey().getPublicKey())
                .collect(Collectors.toSet());
    }

    public static Optional<Segment> getSegmentBySegmentationKey(final Collection<Segment> segments, final String segmentationKey) {
        return segments.stream()
                .filter(s -> s.getSegmentation().getKey().getPublicKey().equals(segmentationKey))
                .findFirst();
    }
}
