package ru.yandex.qe.dispenser.domain.dao.service;

import java.util.Collection;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.InMemoryLongKeyDaoImpl;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;

public final class ServiceDaoImpl extends InMemoryLongKeyDaoImpl<Service> implements ServiceDao {
    @NotNull
    private final Multimap<Service, Person> admins = HashMultimap.create();
    @NotNull
    private final Multimap<Service, Person> trustees = HashMultimap.create();

    @Override
    public boolean delete(@NotNull final Service service) {
        admins.removeAll(service);
        trustees.removeAll(service);
        return super.delete(service);
    }

    @NotNull
    @Override
    public Multimap<Service, Person> getAdmins(@NotNull final Collection<Service> services) {
        return CollectionUtils.filter(admins, (s, a) -> services.contains(s));
    }

    @Override
    public boolean attachAdmins(@NotNull final Service service, @NotNull final Collection<Person> admins) {
        return this.admins.putAll(service, admins);
    }

    @Override
    public boolean detachAdmins(@NotNull final Service service, @NotNull final Collection<Person> admins) {
        return this.admins.get(service).removeAll(admins);
    }

    @Override
    public void detachAllAdmins(@NotNull final Service service) {
        this.admins.get(service).clear();
    }

    @NotNull
    @Override
    public Multimap<Service, Person> getTrustees(@NotNull final Collection<Service> services) {
        return CollectionUtils.filter(trustees, (s, t) -> services.contains(s));
    }

    @Override
    public boolean attachTrustees(@NotNull final Service service, @NotNull final Collection<Person> trustees) {
        return this.trustees.putAll(service, trustees);
    }

    @Override
    public boolean detachTrustees(@NotNull final Service service, @NotNull final Collection<Person> trustees) {
        return this.trustees.get(service).removeAll(trustees);
    }

    @Override
    public void detachAllTrustees(@NotNull final Service service) {
        this.trustees.get(service).clear();
    }

    @NotNull
    @Override
    public Set<Service> getAdminServices(final @NotNull Person admin) {
        return getAll().stream().filter(s -> admins.get(s).contains(admin)).collect(Collectors.toSet());
    }

}
