package ru.yandex.qe.dispenser.domain.dictionaries.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimaps;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import ru.yandex.qe.dispenser.api.v1.DiCampaign;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeRequestSortField;
import ru.yandex.qe.dispenser.api.v1.DiResourcePreorderReasonType;
import ru.yandex.qe.dispenser.api.v1.DiResourceType;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignResourceDao;
import ru.yandex.qe.dispenser.domain.dao.resource.unit.ResourceUnitDao;
import ru.yandex.qe.dispenser.domain.dictionaries.model.CampaignProvidersSettingsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.CampaignsWithBigOrdersDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.ProvidersWithCampaignsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.RequestReasonsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.RequestStatusesDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.ResourcesWithCampaignsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.ResourcesWithUnitAliasesDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.SegmentationsWithCampaignsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.SortFieldsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.UnitsDictionary;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;


@Component
public class FrontDictionariesLoader {

    @NotNull
    private final CampaignDao campaignDao;
    @NotNull
    private final CampaignResourceDao campaignResourceDao;
    @NotNull
    private final HierarchySupplier hierarchySupplier;
    @NotNull
    private final PerCampaignFrontDictionariesManager perCampaignFrontDictionariesManager;
    private final @NotNull ResourceUnitDao resourceUnitDao;

    public FrontDictionariesLoader(@NotNull final CampaignDao campaignDao,
                                   @NotNull final CampaignResourceDao campaignResourceDao,
                                   @NotNull final HierarchySupplier hierarchySupplier,
                                   @NotNull final PerCampaignFrontDictionariesManager perCampaignFrontDictionariesManager,
                                   @NotNull final ResourceUnitDao resourceUnitDao) {
        this.campaignDao = campaignDao;
        this.campaignResourceDao = campaignResourceDao;
        this.hierarchySupplier = hierarchySupplier;
        this.perCampaignFrontDictionariesManager = perCampaignFrontDictionariesManager;
        this.resourceUnitDao = resourceUnitDao;
    }

    private static boolean statusFilter(final QuotaChangeRequest.Status status) {
        return status != QuotaChangeRequest.Status.APPLIED;
    }

    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public Optional<CampaignProvidersSettingsDictionary> getLastActiveCampaignProvidersSettings() {
        // TODO Consistent with default campaign for quota request creation but still pretty confusing, may be disallow default completely later
        final Optional<Campaign> activeCampaignO = campaignDao.getLastActiveDraft();
        return activeCampaignO.map(perCampaignFrontDictionariesManager::getPerCampaignProviderSettings);
    }

    @NotNull
    public RequestStatusesDictionary getRequestStatusesDictionary() {
        final List<RequestStatusesDictionary.RequestStatus> statuses = Stream.of(QuotaChangeRequest.Status.values())
                .filter(FrontDictionariesLoader::statusFilter)
                .map(s -> new RequestStatusesDictionary.RequestStatus(s.toView(), s.getPermissionView()))
                .collect(Collectors.toList());
        return new RequestStatusesDictionary(statuses);
    }

    @NotNull
    public RequestReasonsDictionary getRequestReasonDictionary() {
        final List<RequestReasonsDictionary.RequestReason> reasons = Stream.of(DiResourcePreorderReasonType.values())
                .map(RequestReasonsDictionary.RequestReason::new).collect(Collectors.toList());
        return new RequestReasonsDictionary(reasons);
    }

    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public ProvidersWithCampaignsDictionary getProvidersWithCampaigns() {
        final Set<Service> services = hierarchySupplier.get().getServiceReader().getAll();
        final Map<Long, Set<Long>> campaignIdsByServiceId = campaignResourceDao
                .getAvailableCampaignsForServices(services);
        final List<ProvidersWithCampaignsDictionary.Provider> providers = services.stream()
                .sorted(Comparator.comparing(Service::getPriority, Comparator.nullsLast(Comparator.naturalOrder()))
                        .thenComparing(LongIndexBase::getId))
                .map(s -> new ProvidersWithCampaignsDictionary.Provider(s.getId(), s.getKey(), s.getName(), s.getPriority(),
                        campaignIdsByServiceId.get(s.getId()).stream().sorted().collect(Collectors.toList())))
                .collect(Collectors.toList());
        return new ProvidersWithCampaignsDictionary(providers);
    }

    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public CampaignsWithBigOrdersDictionary getCampaignsWithBigOrders() {
        final List<DiCampaign> campaigns = campaignDao.getAllSorted(Collections.emptySet())
                .stream().map(Campaign::toView).collect(Collectors.toList());
        return new CampaignsWithBigOrdersDictionary(campaigns);
    }

    @NotNull
    public SortFieldsDictionary getSortFields() {
        final List<SortFieldsDictionary.SortField> fields = Stream.of(DiQuotaChangeRequestSortField.values())
                .filter(f -> f != DiQuotaChangeRequestSortField.COST) //temporary disabled in DISPENSER-2754
                .map(SortFieldsDictionary.SortField::new).collect(Collectors.toList());
        return new SortFieldsDictionary(fields);
    }

    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public ResourcesWithCampaignsDictionary getResourcesWithCampaigns() {
        final Set<Resource> resources = hierarchySupplier.get().getResourceReader().getAll();
        final Map<Long, Set<Long>> campaignIdsByResourceId = campaignResourceDao
                .getAvailableCampaignsForResources(resources);
        final List<ResourcesWithCampaignsDictionary.Resource> resourcesDictionary = resources.stream()
                .sorted(Comparator.comparing(Resource::getPriority, Comparator.nullsLast(Comparator.naturalOrder()))
                        .thenComparing(LongIndexBase::getId))
                .map(r -> new ResourcesWithCampaignsDictionary.Resource(r.getId(), r.getPublicKey(),
                        r.getName(), r.getType(), r.getService().getId(), r.getPriority(),
                        campaignIdsByResourceId.get(r.getId()).stream().sorted().collect(Collectors.toList())))
                .collect(Collectors.toList());
        return new ResourcesWithCampaignsDictionary(resourcesDictionary);
    }

    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public SegmentationsWithCampaignsDictionary getSegmentationsWithCampaigns() {
        final Set<Segmentation> segmentations = hierarchySupplier.get().getSegmentationReader().getAll();
        final Set<Segment> segments = hierarchySupplier.get().getSegmentReader().getAll();
        final Map<Long, Set<Segment>> segmentsBySegmentation = segments.stream()
                .collect(Collectors.groupingBy(s -> s.getSegmentation().getId(), Collectors.toSet()));
        final Map<Long, Set<Long>> campaignsBySegmentation = campaignResourceDao
                .getAvailableCampaignsForSegmentations(segmentations);
        final List<SegmentationsWithCampaignsDictionary.Segmentation> segmentationsDictionary = new ArrayList<>();
        segmentations.stream()
                .sorted(Comparator.comparing(Segmentation::getPriority))
                .forEach(segmentation -> {
                    final List<SegmentationsWithCampaignsDictionary.Segment> segmentsDictionary = segmentsBySegmentation
                            .getOrDefault(segmentation.getId(), Collections.emptySet()).stream()
                            .sorted(Comparator.comparing(Segment::getPriority,
                                    Comparator.nullsLast(Comparator.naturalOrder())).thenComparing(LongIndexBase::getId))
                            .map(s -> new SegmentationsWithCampaignsDictionary.Segment(s.getId(), s.getPublicKey(),
                                    s.getName(), s.getPriority())).collect(Collectors.toList());
                    segmentationsDictionary.add(new SegmentationsWithCampaignsDictionary.Segmentation(segmentation.getId(),
                            segmentation.getKey().getPublicKey(), segmentation.getName(), segmentsDictionary,
                            campaignsBySegmentation.get(segmentation.getId()).stream().sorted().collect(Collectors.toList()), segmentation.getPriority()));
                });
        return new SegmentationsWithCampaignsDictionary(segmentationsDictionary);
    }

    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public UnitsDictionary getUnits() {

        final DiUnit.Ensemble[] ensembles = DiUnit.Ensemble.values();
        final ArrayList<UnitsDictionary.Ensemble> resultEnsembles = new ArrayList<>(ensembles.length);
        final HashMultimap<DiUnit.Ensemble, DiUnit> unitByEnsembleKey = HashMultimap.create();

        for (final DiUnit.Ensemble ensemble : ensembles) {
            resultEnsembles.add(new UnitsDictionary.Ensemble(ensemble.getKey(), ensemble.getBase()));
            unitByEnsembleKey.putAll(ensemble, ensemble.getUnit2deg().keySet());
        }

        final HashMultimap<DiUnit, DiUnit.Ensemble> ensembleByUnit = Multimaps.invertFrom(unitByEnsembleKey, HashMultimap.create());

        final List<UnitsDictionary.Unit> units = Arrays.stream(DiUnit.values()).map(unit -> {
            final DiUnit.Ensemble ensemble = ensembleByUnit.get(unit).iterator().next();
            return new UnitsDictionary.Unit(
                    unit.name(),
                    unit.getAbbreviation(),
                    ensemble.getKey(),
                    new UnitsDictionary.UnitMultiplier(ensemble.getUnit2deg().get(unit))
            );
        }).collect(Collectors.toList());

        final List<UnitsDictionary.ResourceType> resourceTypes = Arrays.stream(DiResourceType.values())
                .map(rt -> new UnitsDictionary.ResourceType(rt.name(), rt.getBaseUnit().name()))
                .collect(Collectors.toList());

        return new UnitsDictionary(resultEnsembles, units, resourceTypes);
    }

    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public ResourcesWithUnitAliasesDictionary getResourcesWithUnitAliases() {

        final List<ResourcesWithUnitAliasesDictionary.Resource> resources = resourceUnitDao.getAll().stream()
                .map(ru -> new ResourcesWithUnitAliasesDictionary.Resource(ru.getResourceId(), ru.getDefaultUnit(), ru.getUnitsSettings()))
                .collect(Collectors.toList());

        return new ResourcesWithUnitAliasesDictionary(resources);
    }
}
