package ru.yandex.qe.dispenser.domain.dictionaries.impl;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.domain.dictionaries.model.CampaignProvidersSettingsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.CampaignsWithBigOrdersDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.FrontDictionaries;
import ru.yandex.qe.dispenser.domain.dictionaries.model.ProvidersWithCampaignsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.RequestReasonsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.RequestStatusesDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.ResourcesWithCampaignsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.ResourcesWithUnitAliasesDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.SegmentationsWithCampaignsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.SortFieldsDictionary;
import ru.yandex.qe.dispenser.domain.dictionaries.model.UnitsDictionary;

import javax.annotation.PreDestroy;
import javax.inject.Inject;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

@Component
public class FrontDictionariesManager {

    private static final Logger LOGGER = LoggerFactory.getLogger(FrontDictionariesManager.class);

    @NotNull
    private final ExecutorService executor;
    @NotNull
    private final LoadingCache<String, Object> dictionaryCache;
    @NotNull
    private final LoadingCache<Long, Optional<CampaignProvidersSettingsDictionary>> perCampaignCache;

    @Inject
    public FrontDictionariesManager(@NotNull final PerCampaignFrontDictionariesManager perCampaignFrontDictionariesManager,
                                    @NotNull final FrontDictionariesLoader frontDictionariesLoader) {
        final ThreadFactory threadFactory = new ThreadFactoryBuilder()
                .setDaemon(true)
                .setNameFormat("dictionary-cache-refresh-pool-%d")
                .setUncaughtExceptionHandler((t, e) -> LOGGER.error("Uncaught exception in thread " + t, e))
                .build();
        this.executor = new ThreadPoolExecutor(3, 3, 0L, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(100), threadFactory);
        this.dictionaryCache = CacheBuilder.newBuilder()
                .expireAfterWrite(5, TimeUnit.MINUTES)
                .refreshAfterWrite(1, TimeUnit.MINUTES)
                .build(new DictionaryCacheLoader(frontDictionariesLoader, executor));
        this.perCampaignCache = CacheBuilder.newBuilder()
                .expireAfterWrite(5, TimeUnit.MINUTES)
                .refreshAfterWrite(1, TimeUnit.MINUTES)
                .maximumSize(1000)
                .build(new PerCampaignCacheLoader(perCampaignFrontDictionariesManager, executor));
    }

    @PreDestroy
    public void preDestroy() {
        executor.shutdown();
        try {
            executor.awaitTermination(500, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        executor.shutdownNow();
    }

    @SuppressWarnings("unchecked")
    public CampaignProvidersSettingsDictionary getCampaignProviderSettings(@Nullable final Long campaignId) {
        if (campaignId != null) {
            final Optional<CampaignProvidersSettingsDictionary> dictionaryO = perCampaignCache.getUnchecked(campaignId);
            if (dictionaryO.isPresent()) {
                return dictionaryO.get();
            }
        } else {
            final Optional<CampaignProvidersSettingsDictionary> dictionaryO = (Optional<CampaignProvidersSettingsDictionary>) dictionaryCache
                    .getUnchecked(FrontDictionaries.CAMPAIGN_PROVIDERS_SETTINGS);
            if (dictionaryO.isPresent()) {
                return dictionaryO.get();
            }
        }
        return null;
    }

    public void prepareRequestStatusesDictionary(@NotNull final FrontDictionaries.Builder dictionariesBuilder) {
        final RequestStatusesDictionary dictionary = (RequestStatusesDictionary) dictionaryCache
                .getUnchecked(FrontDictionaries.REQUEST_STATUS);
        dictionariesBuilder.requestStatusesDictionary(dictionary);
    }

    public void prepareRequestReasonDictionary(@NotNull final FrontDictionaries.Builder dictionariesBuilder) {
        final RequestReasonsDictionary dictionary = (RequestReasonsDictionary) dictionaryCache
                .getUnchecked(FrontDictionaries.REQUEST_REASON);
        dictionariesBuilder.requestReasonsDictionary(dictionary);
    }

    public void prepareProvidersWithCampaigns(@NotNull final FrontDictionaries.Builder dictionariesBuilder) {
        final ProvidersWithCampaignsDictionary dictionary = (ProvidersWithCampaignsDictionary) dictionaryCache
                .getUnchecked(FrontDictionaries.PROVIDERS_WITH_CAMPAIGNS);
        dictionariesBuilder.providersWithCampaignsDictionary(dictionary);
    }

    public void prepareCampaignsWithBigOrders(@NotNull final FrontDictionaries.Builder dictionariesBuilder) {
        final CampaignsWithBigOrdersDictionary dictionary = (CampaignsWithBigOrdersDictionary) dictionaryCache
                .getUnchecked(FrontDictionaries.CAMPAIGNS_WITH_BIG_ORDERS);
        dictionariesBuilder.campaignsWithBigOrdersDictionary(dictionary);
    }

    public void prepareSortFields(@NotNull final FrontDictionaries.Builder dictionariesBuilder) {
        final SortFieldsDictionary dictionary = (SortFieldsDictionary) dictionaryCache
                .getUnchecked(FrontDictionaries.SORT_FIELDS);
        dictionariesBuilder.sortFieldsDictionary(dictionary);
    }

    public void prepareResourcesWithCampaigns(@NotNull final FrontDictionaries.Builder dictionariesBuilder) {
        final ResourcesWithCampaignsDictionary dictionary = (ResourcesWithCampaignsDictionary) dictionaryCache
                .getUnchecked(FrontDictionaries.RESOURCES_WITH_CAMPAIGNS);
        dictionariesBuilder.resourcesWithCampaignsDictionary(dictionary);
    }

    public void prepareSegmentationsWithCampaigns(@NotNull final FrontDictionaries.Builder dictionariesBuilder) {
        final SegmentationsWithCampaignsDictionary dictionary = (SegmentationsWithCampaignsDictionary) dictionaryCache
                .getUnchecked(FrontDictionaries.SEGMENTATIONS_WITH_CAMPAIGNS);
        dictionariesBuilder.segmentationsWithCampaignsDictionary(dictionary);
    }

    public void prepareUnits(@NotNull final FrontDictionaries.Builder dictionariesBuilder) {
        final UnitsDictionary dictionary = (UnitsDictionary) dictionaryCache
                .getUnchecked(FrontDictionaries.UNITS);
        dictionariesBuilder.unitsDictionary(dictionary);
    }

    public void prepareResourcesWithUnitAliases(@NotNull final FrontDictionaries.Builder dictionariesBuilder) {
        final ResourcesWithUnitAliasesDictionary dictionary = (ResourcesWithUnitAliasesDictionary) dictionaryCache
                .getUnchecked(FrontDictionaries.RESOURCES_WITH_UNIT_ALIASES);
        dictionariesBuilder.resourcesWithUnitAliasesDictionary(dictionary);
    }

    private static final class DictionaryCacheLoader extends CacheLoader<String, Object> {

        @NotNull
        private final FrontDictionariesLoader frontDictionariesLoader;
        @NotNull
        private final ExecutorService executor;

        private DictionaryCacheLoader(@NotNull final FrontDictionariesLoader frontDictionariesLoader,
                                      @NotNull final ExecutorService executor) {
            this.frontDictionariesLoader = frontDictionariesLoader;
            this.executor = executor;
        }

        @Override
        public Object load(@NotNull final String key) {
            if (FrontDictionaries.REQUEST_STATUS.equals(key)) {
                return frontDictionariesLoader.getRequestStatusesDictionary();
            }
            if (FrontDictionaries.REQUEST_REASON.equals(key)) {
                return frontDictionariesLoader.getRequestReasonDictionary();
            }
            if (FrontDictionaries.PROVIDERS_WITH_CAMPAIGNS.equals(key)) {
                return frontDictionariesLoader.getProvidersWithCampaigns();
            }
            if (FrontDictionaries.CAMPAIGNS_WITH_BIG_ORDERS.equals(key)) {
                return frontDictionariesLoader.getCampaignsWithBigOrders();
            }
            if (FrontDictionaries.SORT_FIELDS.equals(key)) {
                return frontDictionariesLoader.getSortFields();
            }
            if (FrontDictionaries.CAMPAIGN_PROVIDERS_SETTINGS.equals(key)) {
                return frontDictionariesLoader.getLastActiveCampaignProvidersSettings();
            }
            if (FrontDictionaries.RESOURCES_WITH_CAMPAIGNS.equals(key)) {
                return frontDictionariesLoader.getResourcesWithCampaigns();
            }
            if (FrontDictionaries.SEGMENTATIONS_WITH_CAMPAIGNS.equals(key)) {
                return frontDictionariesLoader.getSegmentationsWithCampaigns();
            }
            if (FrontDictionaries.UNITS.equals(key)) {
                return frontDictionariesLoader.getUnits();
            }
            if (FrontDictionaries.RESOURCES_WITH_UNIT_ALIASES.equals(key)) {
                return frontDictionariesLoader.getResourcesWithUnitAliases();
            }
            throw new IllegalArgumentException("Unsupported dictionary cache key: " + key);
        }

        @Override
        public ListenableFuture<Object> reload(@NotNull final String key, @NotNull final Object oldValue) {
            try {
                final ListenableFutureTask<Object> task = ListenableFutureTask.create(() -> load(key));
                executor.execute(task);
                return task;
            } catch (final RuntimeException e) {
                LOGGER.error("Dictionary cache reload failure", e);
                throw e;
            }
        }

    }

    private static final class PerCampaignCacheLoader extends CacheLoader<Long, Optional<CampaignProvidersSettingsDictionary>> {

        @NotNull
        private final PerCampaignFrontDictionariesManager perCampaignFrontDictionariesManager;
        @NotNull
        private final ExecutorService executor;

        private PerCampaignCacheLoader(@NotNull final PerCampaignFrontDictionariesManager perCampaignFrontDictionariesManager,
                                       @NotNull final ExecutorService executor) {
            this.perCampaignFrontDictionariesManager = perCampaignFrontDictionariesManager;
            this.executor = executor;
        }

        @Override
        public Optional<CampaignProvidersSettingsDictionary> load(@NotNull final Long key) {
            return perCampaignFrontDictionariesManager.getPerCampaignProviderSettings(key);
        }

        @Override
        public ListenableFuture<Optional<CampaignProvidersSettingsDictionary>> reload(
                @NotNull final Long key, @NotNull final Optional<CampaignProvidersSettingsDictionary> oldValue) {
            try {
                final ListenableFutureTask<Optional<CampaignProvidersSettingsDictionary>> task
                        = ListenableFutureTask.create(() -> load(key));
                executor.execute(task);
                return task;
            } catch (final RuntimeException e) {
                LOGGER.error("Per campaign dictionary cache reload failure", e);
                throw e;
            }
        }

    }

    @TestOnly
    public void clearCache(){
        dictionaryCache.invalidateAll();
        perCampaignCache.invalidateAll();
    }
}
