package ru.yandex.qe.dispenser.domain.distributed;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.stream.Collectors;

import javax.annotation.PreDestroy;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.api.v1.DiMetaValueSet;
import ru.yandex.qe.dispenser.api.v1.DiProject;
import ru.yandex.qe.dispenser.api.v1.request.DiEntity;
import ru.yandex.qe.dispenser.api.v1.request.DiProcessingMode;
import ru.yandex.qe.dispenser.api.v1.response.DiEntityOwnershipResponse;
import ru.yandex.qe.dispenser.client.v1.DiOAuthToken;
import ru.yandex.qe.dispenser.client.v1.Dispenser;
import ru.yandex.qe.dispenser.client.v1.DispenserFactory;
import ru.yandex.qe.dispenser.client.v1.builder.GetEntityOwnershipsRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.impl.DispenserConfig;
import ru.yandex.qe.dispenser.client.v1.impl.GetEntityOwnershipsRequestBuilderImpl;
import ru.yandex.qe.dispenser.client.v1.impl.RemoteDispenserFactory;
import ru.yandex.qe.dispenser.domain.Entity;
import ru.yandex.qe.dispenser.domain.EntitySpec;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.DiJdbcTemplate;
import ru.yandex.qe.dispenser.domain.dao.entity.InMemoryOnlyEntityDao;
import ru.yandex.qe.dispenser.domain.dao.entity.spec.EntitySpecDao;
import ru.yandex.qe.dispenser.domain.dao.quota.MixedQuotaDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.support.EntityOperation;
import ru.yandex.qe.dispenser.domain.support.EntityUsageDiff;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;

public class DistributedManager {
    private static final Logger LOG = LoggerFactory.getLogger(DistributedManager.class);
    private volatile boolean isStopped = false;


    @Autowired
    private DispenserConfig.Environment env;

    @Autowired
    private DiJdbcTemplate jdbcTemplate;

    @Autowired
    private MixedQuotaDao mixedQuotaDao;

    @Autowired
    private EntitySpecDao entitySpecDao;

    @Autowired
    private InMemoryOnlyEntityDao entityDao;

    private final Identifier myIdentifier;

    @NotNull
    private final String authToken;

    private volatile MasterPair remote;

    public DistributedManager(@NotNull final Identifier myIdentifier, @NotNull final String authToken) {
        LOG.info("initializing DistributedManager with {}", myIdentifier);
        this.myIdentifier = myIdentifier;
        this.authToken = authToken;

        Runtime.getRuntime().addShutdownHook(new Thread(this::destroy));
    }

    @PreDestroy
    private synchronized void destroy() {
        LOG.info("start destroying DistributedManager");
        checkStopped();
        remove(myIdentifier);
        isStopped = true;
        LOG.info("done!");
    }

    private void checkStopped() {
        if (isStopped) {
            throw new IllegalStateException("manager is not running");
        }
    }

    @Transactional
    public void dump() {
        LOG.info("start dumping");
        if (myPriority() != 0) {
            LOG.info("it is not my tern, cancel");
            return;
        }

        final List<EntitySpec> specs = entitySpecDao.getAll().stream().filter(s -> s.getTag().equals(myIdentifier.tag())).collect(Collectors.toList());
        LOG.info("dumping {}", specs);
        mixedQuotaDao.dump(specs);
        LOG.info("OK!");

    }

    @Transactional
    public synchronized void update() {
        LOG.info("start updating {}", readableName());

        checkStopped();

        List<Identifier> identifierList = getIdentifiers();

        {
            final StringJoiner sj = new StringJoiner(", ");
            identifierList.stream().map(Identifier::toString).forEach(sj::add);
            LOG.info("got identifiers: {}", sj);
        }

        //remove old instances
        identifierList.stream().filter(id -> id.host().equals(myIdentifier.host()) && id.ts() != myIdentifier.ts()).forEach(id -> {
                    LOG.info("remove {} (host was reloaded)", id);
                    remove(id);
                }
        );

        //id host is not registered yet
        if (!identifierList.contains(myIdentifier)) {
            LOG.info("insert myself");
            jdbcTemplate.update("insert into cluster_state (tag, ts, host) values (:tag, :ts, :host)", toParams(myIdentifier));
        } else {
            LOG.info("I am already in list");
        }

        identifierList = getIdentifiers();

        final int myPosition = identifierList.indexOf(myIdentifier);
        if (myPosition < 0) {
            throw new IllegalStateException("I am not in identifier list");
        }

        final int masterPosition = (myPosition == 0 ? identifierList.size() : myPosition) - 1;
        final Identifier newMaster = identifierList.get(masterPosition);


        LOG.info("new master {}/{} (was {})", newMaster, masterPosition, remote);

        if (myIdentifier.equals(newMaster)) {
            LOG.warn("no master found!");
            remote = null;
        } else {
            // master is not specified or marser has been changed
            if ((remote == null || !remote.master.equals(newMaster))) {
                LOG.info("change master: {}", newMaster);
                remote = new MasterPair(newMaster);
            }
        }

    }

    @Transactional(propagation = Propagation.MANDATORY)
    private int remove(@NotNull final Identifier id) {
        return jdbcTemplate.update("delete from cluster_state where host = :host and ts = :ts", toParams(id));
    }

    public synchronized void sync() {
        LOG.info("start entity sync");
        if (remote == null) {
            LOG.error("no master found, exit");
            return;
        }

        final Dispenser dispenser = remote.client;
        int attemp = 0;
        int gotEntities = 0;

        do {

            final int[] counter = {0};

            final Map<Identifier, Long> updateMap = entityDao.getUpdateTimes();

            final GetEntityOwnershipsRequestBuilder<?> builder = dispenser.getEntityOwnerships()
                    //dont remove it, need for correct client work
                    .forHost(myIdentifier.toString(), 1000000L + System.currentTimeMillis());
            updateMap.forEach((identifier, ts) -> builder.forHost(identifier.toString(), ts));


            final DiEntityOwnershipResponse resp;

            try {
                resp = builder.perform();
            } catch (Exception e) {
                LOG.error("client error, remove master " + remote.master + " from list!", e);
                remove(remote.master);
                remote = null;
                return;
            }

            final List<EntityOperation> operations = resp.stream().map(ownership -> {

                final DiEntity e = ownership.getEntity();

                final Service service = Hierarchy.get().getServiceReader().read(
                        e.tryGetServiceKey().orElseThrow(() -> new RuntimeException("serviceKey expected")));

                if (Hierarchy.get().isSql()) {
                    LOG.error("sql dao is forbidden fo inmemory sync!");
                    throw new RuntimeException("sql dao is forbidden fo inmemory sync!");
                }
                final EntitySpec spec = Hierarchy.get().getEntitySpecReader().read(new EntitySpec.Key(e.getSpecificationKey(), service));
                final Map<String, Resource> resources = CollectionUtils.toMap(spec.getResources(), r -> r.getKey().getPublicKey());

                final Entity.Builder entityBuilder = Entity.builder(e.getKey()).spec(spec);

                e.getDimensions().forEach(d -> {
                    final Resource resource = resources.get(d.getResourceKey());
                    if (resource == null) {
                        final String message =
                                "No resource '" + d.getResourceKey() + "' in entity specification '" + spec.getKey().getPublicKey() + "'!";
                        throw new IllegalArgumentException(message);
                    }
                    entityBuilder.dimension(resource, d.getAmount());
                });

                final DiMetaValueSet metaValues = e.getMetaValues();

                final Identifier identifier = Identifier.fromString(
                        e.tryGetHostIdentifier()
                                .orElseThrow(() -> new RuntimeException("owner identifier must present")));
                entityBuilder.identifier(identifier);

                final Long creationTime = e.tryGetCreationTime().orElseThrow(() -> new RuntimeException("creatin time is required for sync"));
                entityBuilder.creationTime(creationTime);

                final DiProject diProject = ownership.getProject();
                final String login = Optional.ofNullable(diProject.getPerson()).orElseThrow(() -> new RuntimeException("person expected"));

                final Collection<Project> projects = Hierarchy.get().getProjectReader().readPresent(Collections.singleton(Project.Key.of(
                        ownership.getProject().getKey(),
                        Hierarchy.get().getPersonReader().read(login)
                ))).values();

                if (projects.size() != 1) {
                    throw new RuntimeException("Personal project is not present, can't handle");
                }

                if (ownership.getUsagesCount() != 1) {
                    throw new RuntimeException("Usages count != 1, sharing is not supported (" + ownership.getUsagesCount() + ")");
                }

                final Project personal = projects.iterator().next();

                final Entity entity = entityBuilder.build();
                counter[0]++;
                if (!entityDao.hasEntity(entity)) {
                    entityDao.createIfAbsent(entity);
                    return Optional.of(new EntityUsageDiff(DiProcessingMode.ROLLBACK_ON_ERROR, entity, personal, metaValues, ownership.getUsagesCount(), null));
                } else {
                    return Optional.<EntityUsageDiff>empty();
                }
            }).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());

            gotEntities = counter[0];
            LOG.info("sync with {}\tgot {} entities ( attempt: {}, total: {})", remote.master, operations.size(), attemp, (attemp * GetEntityOwnershipsRequestBuilderImpl.LIMIT + gotEntities));

            entityDao.doChanges(operations);
            attemp++;

        } while (gotEntities >= GetEntityOwnershipsRequestBuilderImpl.LIMIT);

        LOG.info("sync with {}", remote.master);
        return;
    }

    private String readableName() {
        return String.format("DistributedManager[%s]", myIdentifier.toString());
    }


    public List<Identifier> getIdentifiers() {
        return jdbcTemplate.query("select * from cluster_state where tag = :tag order by ts", toParams(myIdentifier), this::toIdentifier);
    }

    public int myPriority() {
        return getIdentifiers().indexOf(myIdentifier);
    }


    private Identifier toIdentifier(final ResultSet rs, final int i) throws SQLException {
        return new Identifier(rs.getString("host"), rs.getString("tag"), rs.getLong("ts"));
    }

    @NotNull
    private Map<String, Object> toParams(@NotNull final Identifier id) {
        return ImmutableMap.of("tag", id.tag(), "host", id.host(), "ts", id.ts());
    }

    private class MasterPair {
        private final Identifier master;

        private final Dispenser client;

        public MasterPair(final Identifier master) {
            this.master = master;

            final DispenserConfig dispenserConfig = new DispenserConfig()
                    .setClientId(("dispenser-distributed-" + myIdentifier.name()).replaceAll(":", "-").toUpperCase())
                    .setDispenserHost("http://" + master.host())
                    .setReceiveTimeout(30000L)
                    .setEnvironment(env)
                    .setServiceZombieOAuthToken(DiOAuthToken.of(authToken));

            final DispenserFactory factory = new RemoteDispenserFactory(dispenserConfig);
            this.client = factory.get();

        }

        public Identifier master() {
            return master;
        }

        public Dispenser client() {
            return client;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }

            if (o == null) {
                return false;
            }

            if (o instanceof Identifier) {
                return Objects.equals(master, o);
            }

            if (getClass() != o.getClass()) {
                return false;
            }

            final MasterPair that = (MasterPair) o;
            return Objects.equals(master, that.master);
        }

        @Override
        public int hashCode() {
            return Objects.hash(master);
        }

        @Override
        public String toString() {
            return master.toString();
        }
    }
}
