package ru.yandex.qe.dispenser.domain.distributed;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class Identifier {
    private static final List<Identifier> IDENTIFIERS = new CopyOnWriteArrayList<>();
    private final String host;
    private final String tag;
    private final long ts;

    public Identifier(final String host, final String tag) {
        this(host, tag, System.currentTimeMillis());
    }

    public Identifier(
            @NotNull @JsonProperty(value = "host") final String host,
            @NotNull @JsonProperty(value = "tag") final String tag,
            @JsonProperty(value = "ts") final long ts) {
        this.host = host;
        this.tag = tag;
        this.ts = ts;
    }

    @JsonProperty(value = "host")
    public String host() {
        return host;
    }

    @JsonProperty(value = "tag")
    public String tag() {
        return tag;
    }

    @JsonProperty(value = "ts")
    public long ts() {
        return ts;
    }

    public String name() {
        return host + "-" + ts;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final Identifier that = (Identifier) o;
        return ts == that.ts &&
                Objects.equals(host, that.host) &&
                Objects.equals(tag, that.tag);
    }

    @Override
    public int hashCode() {
        return Objects.hash(host, tag, ts);
    }

    @Override
    public String toString() {
        return String.format("%s@%d@%s", host, ts, tag);
    }

    public static Identifier cached(@Nullable final Identifier identifier) {
        if (identifier == null) {
            return null;
        }
        //TODO memory issue
        final int index = IDENTIFIERS.indexOf(identifier);
        if (index < 0) {
            synchronized (IDENTIFIERS) {
                if (IDENTIFIERS.indexOf(identifier) < 0) {
                    IDENTIFIERS.add(identifier);
                } else {
                    return cached(identifier);
                }
            }
            return identifier;
        } else {
            return IDENTIFIERS.get(index);
        }
    }

    public static Identifier fromString(final String value) {
        final String[] parts = value.split("@");
        return cached(new Identifier(parts[0], parts[2], Long.valueOf(parts[1])));
    }
}
