package ru.yandex.qe.dispenser.domain.hierarchy;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.common.collect.Table;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaSpec;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.ResourceSegmentation;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.YaGroup;
import ru.yandex.qe.dispenser.domain.dao.dispenser_admins.DispenserAdminsDao;
import ru.yandex.qe.dispenser.domain.dao.dispenser_admins.DispenserAdminsDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.entity.spec.EntitySpecDao;
import ru.yandex.qe.dispenser.domain.dao.entity.spec.EntitySpecDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.person.CachingPersonReader;
import ru.yandex.qe.dispenser.domain.dao.person.PersonProjectRelations;
import ru.yandex.qe.dispenser.domain.dao.person.StaffCache;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectUtils;
import ru.yandex.qe.dispenser.domain.dao.project.SqlProjectDao;
import ru.yandex.qe.dispenser.domain.dao.project.role.ProjectRoleCacheImpl;
import ru.yandex.qe.dispenser.domain.dao.property.CachingPropertyReader;
import ru.yandex.qe.dispenser.domain.dao.property.PropertyDao;
import ru.yandex.qe.dispenser.domain.dao.quota.MixedQuotaDao;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaCacheImpl;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaDao;
import ru.yandex.qe.dispenser.domain.dao.quota.spec.QuotaSpecDao;
import ru.yandex.qe.dispenser.domain.dao.quota.spec.QuotaSpecDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.resource.ResourceDao;
import ru.yandex.qe.dispenser.domain.dao.resource.ResourceDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.resource.group.ResourceGroupDao;
import ru.yandex.qe.dispenser.domain.dao.resource.group.ResourceGroupDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.resource.segmentation.ResourceSegmentationDao;
import ru.yandex.qe.dispenser.domain.dao.resource.segmentation.ResourceSegmentationDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentDao;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.segmentation.SegmentationDao;
import ru.yandex.qe.dispenser.domain.dao.segmentation.SegmentationDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.service.ServiceDao;
import ru.yandex.qe.dispenser.domain.dao.service.ServiceDaoImpl;
import ru.yandex.qe.dispenser.domain.health.HealthCheck;
import ru.yandex.qe.dispenser.domain.health.HealthChecks;
import ru.yandex.qe.dispenser.domain.hierarchy.dao.CachingProjectDao;
import ru.yandex.qe.dispenser.domain.hierarchy.dao.CachingQuotaDao;
import ru.yandex.qe.dispenser.domain.util.MoreCollectors;

@ThreadSafe
public class CachingHierarchySupplier implements HierarchySupplier {
    private static final Logger LOG = LoggerFactory.getLogger(CachingHierarchySupplier.class);
    private static final Long SHORT_UPDATE_THRESHOLD_MS = 100L;
    private static final Long LONG_UPDATE_THRESHOLD_MS = 1000L;
    private static final Long HUGE_UPDATE_THRESHOLD_MS = 10000L;

    @Nullable
    private volatile Hierarchy hierarchy;
    @GuardedBy("updateImpl()")
    private long updateTs;

    @Autowired
    private DispenserAdminsDao dispenserAdminsDao;
    @Autowired
    private SqlProjectDao projectDao;
    @Autowired
    private ServiceDao serviceDao;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private QuotaDao quotaDao;
    @Autowired
    private SegmentationDao segmentationDao;
    @Autowired
    private SegmentDao segmentDao;
    @Autowired
    private ResourceSegmentationDao resourceSegmentationDao;
    @Autowired
    private ResourceGroupDao resourceGroupDao;
    @Autowired
    private PropertyDao propertyDao;
    @Autowired
    private EntitySpecDao entitySpecDao;
    @Autowired
    private QuotaSpecDao quotaSpecDao;
    @Autowired
    private ProjectRoleCacheImpl projectRoleCache;

    @Autowired
    private Hierarchy firstHierarchy;
    @Autowired
    private StaffCache staffCache;
    @Autowired
    private HealthChecks healthChecks;
    @Value("${hierarchy.relevance.time}")
    private long relevanceTime;
    @Autowired
    @Lazy
    private PermissionsCache permissionsCache;

    @PostConstruct
    private void initHealthCheck() {
        healthChecks.register("hierarchy.update", () -> {
            final long timeAfterUpdate = System.currentTimeMillis() - updateTs;
            if (timeAfterUpdate > relevanceTime) {
                return HealthCheck.Result.unhealthy("Hierarchy was not updated during " + timeAfterUpdate + " ms");
            }
            return HealthCheck.Result.healthy();
        });
    }

    @NotNull
    @Override
    public Hierarchy get() {
        final Hierarchy hierarchySnapshot = hierarchy;
        if (hierarchySnapshot != null) {
            return hierarchySnapshot;
        }
        return Objects.requireNonNull(firstHierarchy, "No fallback hierarchy!");
    }

    @Override
    @Transactional
    public void update() {
        Session.IS_READONLY.set(true);
        LOG.info("Hierarchy updating started");
        try {
            updateImpl(System.currentTimeMillis());
        } catch (RuntimeException e) {
            Session.HIERARCHY.remove();
            LOG.error("Hierarchy update error!", e);
        } finally {
            Session.IS_READONLY.set(false);
        }
    }

    @TestOnly
    @Override
    public void reset() {
        hierarchy = null;
    }

    private synchronized void updateImpl(final long startTs) {
        if (updateTs > startTs) {
            return;
        }
        final Stopwatch stopwatch = Stopwatch.createStarted();

        final ProxyingHierarchy nextHierarchy = new ProxyingHierarchy();

        // caching getAll() results
        Session.HIERARCHY.set(nextHierarchy);
        // order of next getAll() is very impotant

        setStubPropetiesDao(nextHierarchy);

        projectRoleCache.invalidate();

        final PersonProjectRelations relations = new PersonProjectRelations();
        final QuotaDao fakeQuotaDao = new CachingQuotaDao(quotaDao, relevanceTime);

        setStubPersonDao(nextHierarchy, relations);

        setStubProjectDao(nextHierarchy, relations, fakeQuotaDao);

        setStubServiceDao(nextHierarchy);

        final ResourceGroupDaoImpl stubResourceGroupoDao = new ResourceGroupDaoImpl();
        stubResourceGroupoDao.createAll(resourceGroupDao.getAll());
        nextHierarchy.setResourceGroupDao(stubResourceGroupoDao);

        final Stopwatch resourceDaoStopwatch = Stopwatch.createStarted();
        //todo
        final ResourceDao stubResourceDao = new ResourceDaoImpl();
        stubResourceDao.createAll(resourceDao.getAll());
        nextHierarchy.setResourceDao(stubResourceDao);
        logAction("ResourceDao updated", resourceDaoStopwatch.elapsed(TimeUnit.MILLISECONDS), SHORT_UPDATE_THRESHOLD_MS);

        final Stopwatch quotaSpecDaoStopwatch = Stopwatch.createStarted();
        final QuotaSpecDaoImpl stubQuotaSpecDao = new QuotaSpecDaoImpl();
        stubQuotaSpecDao.setQuotaDao(fakeQuotaDao);
        stubQuotaSpecDao.createAll(quotaSpecDao.getAll());
        nextHierarchy.setQuotaSpecDao(stubQuotaSpecDao);
        logAction("QuotaSpecDao updated", quotaSpecDaoStopwatch.elapsed(TimeUnit.MILLISECONDS), SHORT_UPDATE_THRESHOLD_MS);

        final Stopwatch dispenserAdminsStopwatch = Stopwatch.createStarted();
        final DispenserAdminsDaoImpl stubDispenserAdminsDao = new DispenserAdminsDaoImpl();
        stubDispenserAdminsDao.setDispenserAdmins(dispenserAdminsDao.getDispenserAdmins());
        nextHierarchy.setDispenserAdminsDao(stubDispenserAdminsDao);
        logAction("DispenserAdminsDao updated", dispenserAdminsStopwatch.elapsed(TimeUnit.MILLISECONDS), SHORT_UPDATE_THRESHOLD_MS);

        setStubEntitySpecDao(nextHierarchy);

        final Stopwatch segmentationDaoStopwatch = Stopwatch.createStarted();
        final SegmentationDaoImpl stubSegmentationDao = new SegmentationDaoImpl();
        stubSegmentationDao.createAll(segmentationDao.getAll());
        nextHierarchy.setSegmentationDao(stubSegmentationDao);
        final SegmentDaoImpl stubSegmentDao = new SegmentDaoImpl();
        stubSegmentDao.createAll(segmentDao.getAll());
        nextHierarchy.setSegmentDao(stubSegmentDao);
        logAction("SegmentationDao updated", segmentationDaoStopwatch.elapsed(TimeUnit.MILLISECONDS), SHORT_UPDATE_THRESHOLD_MS);

        final ResourceSegmentationDaoImpl stubResourceSegmentationDao = new ResourceSegmentationDaoImpl();
        stubResourceSegmentationDao.createAll(resourceSegmentationDao.getAll());
        nextHierarchy.setResourceSegmentationDao(stubResourceSegmentationDao);

        if (quotaDao instanceof MixedQuotaDao) {
            final Stopwatch mixedQuotaDaoStopwatch = Stopwatch.createStarted();
            ((MixedQuotaDao) quotaDao).sync();
            logAction("MixedQuotaDao synced", mixedQuotaDaoStopwatch.elapsed(TimeUnit.MILLISECONDS), HUGE_UPDATE_THRESHOLD_MS);
        }
        setStubQuotaDao(nextHierarchy, fakeQuotaDao);
        nextHierarchy.setPermissionsCache(permissionsCache);

        hierarchy = nextHierarchy;
        updateTs = System.currentTimeMillis();

        LOG.info("Hierarchy successfully updated in {} ms", stopwatch.elapsed(TimeUnit.MILLISECONDS));
    }

    private void setStubPropetiesDao(@NotNull final ProxyingHierarchy hierarchy) {
        final Stopwatch stopwatch = Stopwatch.createStarted();
        final CachingPropertyReader reader = new CachingPropertyReader(propertyDao);
        hierarchy.setPropertyReader(reader);
        logAction("PropertyDao updated", stopwatch.elapsed(TimeUnit.MILLISECONDS), SHORT_UPDATE_THRESHOLD_MS);
    }

    private void setStubPersonDao(@NotNull final ProxyingHierarchy hierarchy, @NotNull final PersonProjectRelations relations) {
        final Stopwatch stopwatch = Stopwatch.createStarted();
        final CachingPersonReader cachingPersonReader = new CachingPersonReader(staffCache, relations);
        hierarchy.setPersonReader(cachingPersonReader);
        logAction("CachingPersonReader updated", stopwatch.elapsed(TimeUnit.MILLISECONDS), SHORT_UPDATE_THRESHOLD_MS);
    }

    private void setStubProjectDao(@NotNull final ProxyingHierarchy hierarchy,
                                   @NotNull final PersonProjectRelations relations,
                                   @NotNull final QuotaDao fakeQuotaDao) {
        final Stopwatch stopwatch = Stopwatch.createStarted();
        final ProjectDaoImpl stubProjectDao = new CachingProjectDao(projectDao);
        stubProjectDao.setUserProjects(relations);
        stubProjectDao.setQuotaDao(fakeQuotaDao);
        stubProjectDao.setStaffCache(staffCache);
        final List<Project> allProjects = ProjectUtils.topologicalOrder(projectDao.getRoot());
        logAction("Getting all projects from SqlProjectDao finished", stopwatch.elapsed(TimeUnit.MILLISECONDS), LONG_UPDATE_THRESHOLD_MS);

        final Table<Project, String, Set<Person>> linkedPersons = projectDao.getLinkedPersons(allProjects);
        final Table<Project, String, Set<YaGroup>> linkedGroups = projectDao.getLinkedGroups(allProjects);

        relations.attachAll(PersonProjectRelations.EntityType.PERSON, linkedPersons);
        relations.attachAll(PersonProjectRelations.EntityType.GROUP, linkedGroups);

        logAction("Getting all data from SqlProjectDao finished", stopwatch.elapsed(TimeUnit.MILLISECONDS), LONG_UPDATE_THRESHOLD_MS);
        allProjects.forEach(stubProjectDao::create);
        hierarchy.setProjectDao(stubProjectDao);
        logAction("ProjectDao updated", stopwatch.elapsed(TimeUnit.MILLISECONDS), LONG_UPDATE_THRESHOLD_MS);
    }

    private void setStubServiceDao(@NotNull final ProxyingHierarchy hierarchy) {
        final Stopwatch stopwatch = Stopwatch.createStarted();
        final ServiceDao stubServiceDao = new ServiceDaoImpl();
        final Set<Service> allServices = serviceDao.getAll();
        stubServiceDao.createAll(allServices);
        serviceDao.getAdmins(allServices).asMap().forEach(stubServiceDao::attachAdmins);
        serviceDao.getTrustees(allServices).asMap().forEach(stubServiceDao::attachTrustees);
        hierarchy.setServiceDao(stubServiceDao);
        logAction("ServiceDao updated", stopwatch.elapsed(TimeUnit.MILLISECONDS), SHORT_UPDATE_THRESHOLD_MS);
    }

    private void setStubEntitySpecDao(@NotNull final ProxyingHierarchy hierarchy) {
        final Stopwatch stopwatch = Stopwatch.createStarted();
        final EntitySpecDao stubEntitySpecDao = new EntitySpecDaoImpl();
        stubEntitySpecDao.createAll(entitySpecDao.getAll());
        hierarchy.setEntitySpecDao(stubEntitySpecDao);
        logAction("EntitySpecDao updated", stopwatch.elapsed(TimeUnit.MILLISECONDS), SHORT_UPDATE_THRESHOLD_MS);
    }

    private void setStubQuotaDao(@NotNull final ProxyingHierarchy hierarchy, @NotNull final QuotaDao fakeQuotaDao) {
        final Stopwatch updateStopwatch = Stopwatch.createStarted();
        final Set<Project> allKnownProjects = hierarchy.getProjectReader().getAll();
        final Collection<Quota> quotas = getAllQuotas(hierarchy, allKnownProjects);

        final Stopwatch cacheStopwatch = Stopwatch.createStarted();
        hierarchy.setQuotaCache(new QuotaCacheImpl(quotas));
        logAction("Cache initialized", cacheStopwatch.elapsed(TimeUnit.MILLISECONDS), HUGE_UPDATE_THRESHOLD_MS);

        logAction("QuotaDao updated", updateStopwatch.elapsed(TimeUnit.MILLISECONDS), HUGE_UPDATE_THRESHOLD_MS);
    }

    private Collection<Quota> getAllQuotas(@NotNull final ProxyingHierarchy hierarchy, @NotNull final Set<Project> projects) {
        final Set<QuotaSpec> quotaSpecs = hierarchy.getQuotaSpecReader().getAll();
        final Map<Resource, List<Set<Segment>>> allSegmentsCombinations = getAllSegmentsCombinations(hierarchy);

        final Map<Quota.Key, Quota> quotas = new HashMap<>();

        final Set<Quota> nonEmptyQuotas = quotaDao.getNonEmptyQuotas(projects);
        for (final Quota quota : nonEmptyQuotas) {
            quotas.put(quota.getKey(), quota);
        }

        for (final Project project : projects) {
            for (final QuotaSpec quotaSpec : quotaSpecs) {
                final List<Set<Segment>> segmentCombinations = allSegmentsCombinations.get(quotaSpec.getResource());
                for (final Set<Segment> segmentCombination : segmentCombinations) {
                    final Quota.Key key = new Quota.Key(quotaSpec, project, segmentCombination);
                    if (!quotas.containsKey(key)) {
                        quotas.put(key, Quota.noQuota(key));
                    }
                }
            }
        }

        return quotas.values();
    }

    private Map<Resource, List<Set<Segment>>> getAllSegmentsCombinations(@NotNull final ProxyingHierarchy hierarchy) {

        final Multimap<Segmentation, Segment> segmentationSegments = hierarchy.getSegmentReader().getAll().stream()
                .collect(MoreCollectors.toLinkedMultimap(Segment::getSegmentation, Function.identity()));

        final Map<Resource, List<Set<Segment>>> segmentationsSegmentsByResource = hierarchy.getResourceSegmentationReader().getAll().stream()
                .collect(MoreCollectors.toMultiValueMap(ResourceSegmentation::getResource, rs -> Sets.union(Sets.newHashSet(segmentationSegments.get(rs.getSegmentation())), Collections.singleton(Segment.totalOf(rs.getSegmentation())))));


        final Map<Resource, List<Set<Segment>>> result = segmentationsSegmentsByResource.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e ->
                        Sets.cartesianProduct(e.getValue())
                                .stream()
                                .map(HashSet::new)
                                .collect(Collectors.toList())
                ));


        hierarchy.getResourceReader().getAll().stream()
                .filter(r -> !result.containsKey(r))
                .forEach(r -> result.put(r, Collections.singletonList(Collections.emptySet())));

        return result;
    }

    private void logAction(final String action, final long updateTimeMs, final long thresholdMs) {
        if (updateTimeMs < thresholdMs) {
            LOG.debug("{} in {} ms", action, updateTimeMs);
        } else {
            LOG.warn("{} in {} ms", action, updateTimeMs);
        }
    }

}
