package ru.yandex.qe.dispenser.domain.hierarchy;

import java.time.Duration;

import javax.inject.Inject;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectDao;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectReader;

@Component
public class PermissionsCacheImpl implements PermissionsCache {

    private final LoadingCache<String, Project> owningCostAccessProjectCache;
    private final LoadingCache<Person, Boolean> canUserViewMoneyCache;
    private final LoadingCache<Person, Boolean> canUserViewBotMoneyCache;

    @Inject
    public PermissionsCacheImpl(
            @Value("${dispenser.owning.cost.access.project.key}") String owningCostAccessProjectKey,
            ProjectDao projectDao,
            HierarchySupplier hierarchySupplier) {
        this.owningCostAccessProjectCache = CacheBuilder.newBuilder()
                .expireAfterWrite(Duration.ofMinutes(30))
                .build(new CacheLoader<>() {
                    @Override
                    public Project load(String key) {
                        return projectDao.read(key);
                    }
                });
        this.canUserViewMoneyCache = CacheBuilder.newBuilder()
                .expireAfterWrite(Duration.ofMinutes(1))
                .maximumSize(10000)
                .build(new CacheLoader<>() {
                    @Override
                    public Boolean load(Person key) {
                        Hierarchy hierarchy = hierarchySupplier.get();
                        ProjectReader projectReader = hierarchy.getProjectReader();
                        return isInOwningCostAccessProject(projectReader, owningCostAccessProjectCache,
                                        owningCostAccessProjectKey, key);
                    }
                });
        this.canUserViewBotMoneyCache = CacheBuilder.newBuilder()
                .expireAfterWrite(Duration.ofMinutes(1))
                .maximumSize(10000)
                .build(new CacheLoader<>() {
                    @Override
                    public Boolean load(Person key) {
                        Hierarchy hierarchy = hierarchySupplier.get();
                        ProjectReader projectReader = hierarchy.getProjectReader();
                        return isInOwningCostAccessProject(projectReader, owningCostAccessProjectCache,
                                owningCostAccessProjectKey, key);
                    }
                });
    }

    @Override
    public boolean canUserViewMoney(Person person) {
        return canUserViewMoneyCache.getUnchecked(person);
    }

    @Override
    public boolean canUserViewBotMoney(Person person) {
        return canUserViewBotMoneyCache.getUnchecked(person);
    }

    @Override
    public void invalidateCaches() {
        owningCostAccessProjectCache.invalidateAll();
        canUserViewMoneyCache.invalidateAll();
        canUserViewBotMoneyCache.invalidateAll();
    }

    private static boolean isInOwningCostAccessProject(ProjectReader projectReader,
                                                       LoadingCache<String, Project> owningCostAccessProjectCache,
                                                       String owningCostAccessProjectKey,
                                                       Person person) {
        Project owningCostAccessProject = owningCostAccessProjectCache.getUnchecked(owningCostAccessProjectKey);
        return projectReader.hasRoleNoInheritance(person, owningCostAccessProject, Role.MEMBER)
                || projectReader.hasRoleNoInheritance(person, owningCostAccessProject, Role.STEWARD)
                || projectReader.hasRoleNoInheritance(person, owningCostAccessProject, Role.VS_LEADER);
    }

}
