package ru.yandex.qe.dispenser.domain.hierarchy.dao;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.jetbrains.annotations.NotNull;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectDao;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectDaoImpl;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;

public final class CachingProjectDao extends ProjectDaoImpl {
    @NotNull
    private final ProjectDao directProjectDao;

    public CachingProjectDao(@NotNull final ProjectDao directProjectDao) {
        this.directProjectDao = directProjectDao;
    }

    @NotNull
    @Override
    public Project createIfAbsent(@NotNull final Project project) {
        try {
            Project result = read(project.getKey());
            if (result.isRoot()) {
                root = result;
            }
            return result;
        } catch (EmptyResultDataAccessException ignored) {
            return directProjectDao.createIfAbsent(project);
        }
    }

    @NotNull
    @Override
    public Map<Project.Key, Project> createAllIfAbsent(@NotNull final Collection<Project> projects) {
        final Map<Project.Key, Project> cachedProjects = readPresent(CollectionUtils.keys(projects));
        final Set<Project> absentProjects = CollectionUtils.filter(projects, p -> !cachedProjects.containsKey(p.getKey()));
        final Map<Project.Key, Project> result = new HashMap<>(directProjectDao.createAllIfAbsent(absentProjects));
        result.putAll(cachedProjects);
        projects.forEach(project -> {
            if (project.isRoot()) {
                root = project;
            }
        });
        return result;
    }

    @NotNull
    @Override
    public Project read(@NotNull final Long id) {
        try {
            return super.read(id);
        } catch (EmptyResultDataAccessException ignored) {
            return directProjectDao.read(id);
        }
    }
}
