package ru.yandex.qe.dispenser.domain.lots;

import java.util.Collection;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaView;
import ru.yandex.qe.dispenser.domain.aspect.HierarchyRequired;
import ru.yandex.qe.dispenser.domain.aspect.SecondaryOperation;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaDao;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaUtils;

public class LotsManager {
    private static final Logger LOG = LoggerFactory.getLogger(LotsManager.class);

    @Autowired
    @Qualifier("quotaDao")
    private QuotaDao quotaDao;

    @SecondaryOperation
    @HierarchyRequired(canReject = true)
    public void update() {
        LOG.info("LOTS updating started");
        try {
            updateImpl();
        } catch (RuntimeException e) {
            LOG.warn("LOTS update error!", e);
        }
    }

    private void updateImpl() {
        final Stopwatch stopwatch = Stopwatch.createStarted();
        final Map<Quota.Key, Quota> quotaMap = QuotaUtils.indexByKey(quotaDao.getAll());
        final Collection<QuotaView> quotas = QuotaUtils.aggregateWithActualValuesInternal(quotaMap).values();
        // update quota in new transaction to avoid deadlocks
        quotas.stream()
                .filter(q -> q.getProject().isReal())
                .forEach(q -> {
                    if (q.getTotalActual() > 0 && q.getTotalActual() >= q.getMax()) {
                        if (q.getLastOverquotingTs() == null) {
                            quotaDao.setLastOverquotingTs(quotaMap.get(q.getKey()), System.currentTimeMillis());
                        }
                    } else if (q.getLastOverquotingTs() != null) {
                        quotaDao.setLastOverquotingTs(quotaMap.get(q.getKey()), null);
                    }
                });
        LOG.info("LOTS updating finished in {}", stopwatch.elapsed(TimeUnit.MILLISECONDS));
    }
}
