package ru.yandex.qe.dispenser.domain.notifications;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Stopwatch;
import com.google.common.primitives.Ints;
import org.apache.commons.io.Charsets;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.qe.dispenser.client.v1.impl.DispenserConfig;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.QuotaView;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.aspect.HierarchyRequired;
import ru.yandex.qe.dispenser.domain.aspect.SecondaryOperation;
import ru.yandex.qe.dispenser.domain.dao.notifications.EmailSender;
import ru.yandex.qe.dispenser.domain.dao.notifications.NotificationsDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.util.DateTimeUtils;
import ru.yandex.qe.dispenser.domain.util.PropertyUtils;
import ru.yandex.qe.dispenser.solomon.Solomon;


public class NotificationManager {
    private static final Logger LOG = LoggerFactory.getLogger(NotificationManager.class);
    private static final Predicate<NotificationEntry> ALWAYS_FALSE_PREDICATE = x -> false;
    private static final int ONE_HUNDRED_PERCENT = 100;
    private static final BigDecimal ONE_HUNDRED_PERCENT_BD = BigDecimal.valueOf(ONE_HUNDRED_PERCENT);
    public static final String QUOTA_LIMITS_PROPERTY = "quota_limits";
    private static final Set<String> MDB_KEYS = Set.of("mdb");
    private static final Set<String> MDS_KEYS = Set.of("s3", "mds", "avatars");

    @Autowired
    private EmailSender emailSender;

    @Autowired
    private NotificationsDao notificationsDao;

    @Autowired
    private HierarchySupplier hierarchySupplier;

    @Autowired
    private Solomon solomon;

    @Value("${qe.app.environment}")
    private DispenserConfig.Environment environment;

    @Value("${r.y.q.d.d.n.NotificationManager.enabled}")
    private Boolean isEnabled;

    @Value("${dispenser.cluster.prefix}")
    private String clusterPrefix;

    @Value("#{T(ru.yandex.qe.dispenser.domain.notifications.NotificationManager).readNotificationTemplate('${r.y.q.d.d.n.NotificationManager.notificationTemplatePath}')}")
    private String nirvanaNotificationTemplate;

    @Value("#{T(ru.yandex.qe.dispenser.domain.notifications.NotificationManager).readNotificationTemplate('${r.y.q.d.d.n.NotificationManager.commonNotificationTemplatePath}')}")
    private String commonNotificationTemplate;

    @Value("#{T(ru.yandex.qe.dispenser.domain.notifications.NotificationManager).parseNotificationFilteringPredicate('${r.y.q.d.d.n.NotificationManager.notificationFilteringPredicate}')}")
    private Predicate<NotificationEntry> notificationFilteringPredicate;

    @Value("#{T(ru.yandex.qe.dispenser.domain.notifications.NotificationManager).getServicesWithDisabledZeroMaxNotifications('${r.y.q.d.d.n.NotificationManager.servicesWithDisabledZeroMaxNotifications}')}")
    private Set<String> servicesWithDisabledZeroMaxNotifications;

    public static String readNotificationTemplate(@NotNull final String resourcePath) throws IOException {
        return IOUtils.toString(NotificationManager.class.getClassLoader().getResourceAsStream(resourcePath), Charsets.UTF_8);
    }

    public static Predicate<NotificationEntry> parseNotificationFilteringPredicate(@NotNull final String string) {
        LOG.info("Setting up NotificationManager using filtering predicate: {}", string);
        return Arrays.stream(string.split(","))
                .map(String::trim)
                .filter(s -> !s.isEmpty())
                .map(NotificationQuotaKeysPredicate::fromString)
                .reduce(Predicate::or)
                .orElse(ALWAYS_FALSE_PREDICATE);
    }

    @TestOnly
    public void setEmailSender(@NotNull final EmailSender emailSender) {
        this.emailSender = emailSender;
    }

    @TestOnly
    public void setIsEnabled(final boolean isEnabled) {
        this.isEnabled = isEnabled;
    }

    @TestOnly
    public void setNotificationFilteringPredicate(@NotNull final Predicate<NotificationEntry> notificationFilteringPredicate) {
        this.notificationFilteringPredicate = notificationFilteringPredicate;
    }

    @SecondaryOperation
    @HierarchyRequired(canReject = true)
    public void sendNotifications() {
        if (!isEnabled) {
            LOG.info("NotificationManager is disabled");
            return;
        }
        LOG.info("NotificationManager is enabled, start working...");
        final Stopwatch stopwatch = Stopwatch.createStarted();
        final Set<NotificationEntry> alreadyNotified = notificationsDao.getActualNotifications();
        final Collection<QuotaView> quotas = hierarchySupplier.get().getQuotaCache().getAll();

        final Map<Service, List<QuotaView>> quotasByService = quotas.stream()
                .filter(q -> q.getProject().isReal())
                .filter(q -> q.getTotalActual() > 0 || q.getOwnActual() > 0)
                .collect(Collectors.groupingBy(q -> q.getResource().getService()));
        final Map<Service, Set<NotificationEntry>> possibleNotificationsByService = new HashMap<>();
        final Set<NotificationEntry> allPossibleNotifications = new HashSet<>();
        final Set<NotificationEntry> notificationToSent = new HashSet<>();
        for (final Map.Entry<Service, List<QuotaView>> entry : quotasByService.entrySet()) {
            final Service service = entry.getKey();
            final List<QuotaView> serviceQuotas = entry.getValue();

            final TreeSet<Integer> sortedLimits = getLimitsByService(service);
            final int lowerLimit = sortedLimits.first();

            final Function<QuotaView, Long> actualF;
            final Function<QuotaView, Long> maxF;
            if (service.getSettings().usesProjectHierarchy()) {
                actualF = QuotaView::getTotalActual;
                maxF = QuotaView::getMax;
            } else {
                actualF = QuotaView::getOwnActual;
                maxF = QuotaView::getOwnMax;
            }

            final Predicate<QuotaView> overquoteWithZeroMaxFilter;

            if (servicesWithDisabledZeroMaxNotifications.contains(service.getKey())) {
                overquoteWithZeroMaxFilter = quotaView -> maxF.apply(quotaView) > 0;
            } else {
                overquoteWithZeroMaxFilter = quotaView -> true;
            }

            final Set<NotificationEntry> entries = serviceQuotas.stream()
                    .filter(q -> actualF.apply(q) > 0)
                    .filter(overquoteWithZeroMaxFilter)
                    .filter(q -> exceedPercentLimit(actualF.apply(q), maxF.apply(q), lowerLimit))
                    .flatMap(q -> {
                        final int percentUsage = percentUsage(actualF.apply(q), maxF.apply(q));
                        //get all limits less or equal than current usage
                        final NavigableSet<Integer> limits = sortedLimits.headSet(percentUsage, true);
                        notificationToSent.add(buildEntry(q, limits.last()));
                        return limits
                                .stream()
                                .map(limit -> buildEntry(q, limit));
                    })
                    .collect(Collectors.toSet());
            possibleNotificationsByService.put(service, entries);
            allPossibleNotifications.addAll(entries);
        }

        for (final Map.Entry<Service, Set<NotificationEntry>> entry : possibleNotificationsByService.entrySet()) {
            final Set<NotificationEntry> possibleNotifications = entry.getValue();
            possibleNotifications.stream()
                    .filter(notificationFilteringPredicate)
                    .filter(pn -> !alreadyNotified.contains(pn))
                    .forEach(notification -> {
                        if (notificationToSent.contains(notification)) {
                            final Hierarchy hierarchy = hierarchySupplier.get();
                            // find closest project with linked responsibles
                            Project project = notification.getProject();
                            Collection<Person> responsibles = hierarchy.getProjectReader().getLinkedResponsibles(project);
                            while (responsibles.isEmpty() && project.getMailList() == null && !project.isRoot()) {
                                project = project.getParent();
                                responsibles = hierarchy.getProjectReader().getLinkedResponsibles(project);
                            }

                            final String mailList = project.getMailList();
                            final Service service = notification.getSpec().getResource().getService();
                            boolean mdsNotification = MDS_KEYS.contains(service.getKey());
                            boolean mdbNotification = MDB_KEYS.contains(service.getKey());
                            boolean mdsOrMdb = mdbNotification || mdsNotification;
                            // if no project with responsibles found, send notification to service admins
                            final boolean toServiceAdmins = project.isRoot()
                                    && ((mailList == null && responsibles.isEmpty()) || mdsOrMdb);

                            if (toServiceAdmins) {
                                responsibles = hierarchy.getServiceReader().getAdmins(service);
                            }
                            // See DISPENSER-4273, DISPENSER-4440
                            boolean skipNotification = toServiceAdmins && mdsOrMdb;
                            try {
                                LOG.debug("Sending notification {} to {}", notification, responsibles);
                                final String notificationMessage = getNotificationMessage(service, notification, toServiceAdmins);
                                final Set<String> notificationRecipients = responsibles.stream().map(Person::getMail).collect(Collectors.toSet());
                                if (mailList != null) {
                                    notificationRecipients.add(mailList);
                                }
                                if (!skipNotification) {
                                    emailSender.sendMessage(getNotificationSubject(notification), notificationMessage, notificationRecipients);
                                }
                                notificationsDao.addNotification(notification);
                            } catch (RuntimeException e) {
                                LOG.error("Can't send notification", e);
                            }
                        } else {
                            try {
                                notificationsDao.addNotification(notification);
                            } catch (RuntimeException e) {
                                LOG.error("Can't save notification", e);
                            }
                        }
                    });
        }
        //remove fixed projects
        alreadyNotified.stream()
                .filter(an -> !allPossibleNotifications.contains(an))
                .forEach(notification -> notificationsDao.remove(notification));
        LOG.info("Notification sending has finished, took {} ms", stopwatch.elapsed(TimeUnit.MILLISECONDS));
    }

    private String getNotificationMessage(final Service service, final NotificationEntry notification, final boolean toServiceAdmins) {
        final long overquotingTs = notification.getOverquotingTs() != null ?
                notification.getOverquotingTs() :
                System.currentTimeMillis(); // timestamp is not set by LotsManager yet, use current time
        if (service.getKey().equals("nirvana")) {
            return MessageFormat.format(nirvanaNotificationTemplate,
                    toServiceAdmins ? 1 : 0,
                    notification.getSpec().getResource().getService().getName(),
                    notification.getProject().getName(),
                    notification.getProject().getPublicKey(),
                    notification.getSpec().getDescription(),
                    getBaseUrl(),
                    DateTimeUtils.formatToMskDateTime(overquotingTs),
                    notification.getQuotaLimit(),
                    notification.isOverquoting() ? 1 : 0);
        } else {
            return MessageFormat.format(commonNotificationTemplate,
                    toServiceAdmins ? 1 : 0,
                    notification.getSpec().getResource().getService().getName(),
                    notification.getProject().getName(),
                    notification.getProject().getPublicKey(),
                    notification.getSpec().getDescription(),
                    DateTimeUtils.formatToMskDateTime(overquotingTs),
                    notification.getQuotaLimit(),
                    solomon.getStatisticsLink(notification.getSpec(), notification.getProject())
            );
        }
    }

    private static TreeSet<Integer> getLimitsByService(final Service service) {
        final TreeSet<Integer> sortedLimits = PropertyUtils.readStringOptional(QUOTA_LIMITS_PROPERTY, service.getKey())
                .map(s -> Arrays.stream(s.split(",")))
                .orElse(Stream.empty())
                .map(Ints::tryParse)
                .filter(Objects::nonNull)
                .filter(x -> x > 0 && x < ONE_HUNDRED_PERCENT)
                .collect(Collectors.toCollection(TreeSet::new));

        sortedLimits.add(ONE_HUNDRED_PERCENT);

        return sortedLimits;
    }

    public static Set<String> getServicesWithDisabledZeroMaxNotifications(final String services) {
        return Stream.of(services.split(","))
                .collect(Collectors.toSet());
    }

    private static NotificationEntry buildEntry(final QuotaView quotaView, final int limit) {
        return new NotificationEntry(quotaView.getProject(), quotaView.getSpec(), limit, limit == ONE_HUNDRED_PERCENT ? quotaView.getLastOverquotingTs() : null);
    }

    private static boolean exceedPercentLimit(final long actual, final long max, final int percentLimit) {
        final BigDecimal maxBD = BigDecimal.valueOf(max);
        final BigDecimal actualBD = BigDecimal.valueOf(actual);
        final BigDecimal percentLimitBD = BigDecimal.valueOf(percentLimit);
        return actualBD.compareTo(maxBD.multiply(percentLimitBD).divide(ONE_HUNDRED_PERCENT_BD, RoundingMode.HALF_EVEN)) >= 0;
    }

    private static int percentUsage(final long actual, final long max) {
        if (max == 0) {
            return ONE_HUNDRED_PERCENT;
        } else {
            final BigDecimal maxBD = BigDecimal.valueOf(max);
            final BigDecimal actualBD = BigDecimal.valueOf(actual);
            return actualBD.multiply(ONE_HUNDRED_PERCENT_BD).divide(maxBD, RoundingMode.UP).intValue();
        }
    }

    @NotNull
    private String getBaseUrl() {
        return "https://" + environment.getDispenserHost();
    }

    @NotNull
    private String getNotificationSubject(@NotNull final NotificationEntry notification) {
        final String serviceName = notification.getSpec().getResource().getService().getName();
        final String prefix = String.format("[%s] %s: ", serviceName, notification.getProject().getName());
        if (notification.isOverquoting()) {
            return prefix + "исчерпание квоты";
        } else {
            return prefix + "превышен лимит потребления квоты в " + notification.getQuotaLimit() + " %";
        }
    }

    /**
     * Allows to specify which notifications to send with the following notation:
     * <ul>
     *   <li>nirvana - notifications for all quotas of service 'nirvana'<li/>
     *   <li>nirvana/yt-disk - notifications for quotas bound to resource 'yt-disk' of service 'nirvana'<li/>
     *   <li>nirvana/yt-disk/yt-disk-quota - notifications for quotas with concrete quota specification<li/>
     * <ul/>
     */
    private static final class NotificationQuotaKeysPredicate implements Predicate<NotificationEntry> {
        @NotNull
        private final String serviceKey;
        @Nullable
        private final String resourceKey;
        @Nullable
        private final String quotaSpecKey;

        private NotificationQuotaKeysPredicate(@NotNull final String serviceKey,
                                               @Nullable final String resourceKey,
                                               @Nullable final String quotaSpecKey) {
            this.serviceKey = serviceKey;
            this.resourceKey = resourceKey;
            this.quotaSpecKey = quotaSpecKey;
        }

        @NotNull
        public static Predicate<NotificationEntry> fromString(@NotNull final String string) {
            final String[] parts = StringUtils.split(string.trim(), '/');
            return new NotificationQuotaKeysPredicate(
                    parts[0],
                    parts.length > 1 ? parts[1] : null,
                    parts.length > 2 ? parts[2] : null
            );
        }

        @Override
        public boolean test(@NotNull final NotificationEntry notificationEntry) {
            return notificationEntry.getSpec().getResource().getService().getKey().equals(serviceKey)
                    && (resourceKey == null || notificationEntry.getSpec().getResource().getKey().getPublicKey().equals(resourceKey))
                    && (quotaSpecKey == null || notificationEntry.getSpec().getKey().getPublicKey().equals(quotaSpecKey));
        }
    }
}
