package ru.yandex.qe.dispenser.domain.request;

import java.util.Collections;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

import com.google.common.base.Stopwatch;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.aspect.SecondaryOperation;
import ru.yandex.qe.dispenser.domain.dao.DiJdbcTemplate;

public class RequestDbManager implements RequestManager {
    private static final Logger LOG = LoggerFactory.getLogger(RequestDbManager.class);

    private static final String UPDATE_RESULT_QUERY = "UPDATE request SET result = :result where id = :reqId";
    private static final String REMOVE_OLD_QUERY_PATTERN = "DELETE FROM request WHERE appearance_time < NOW() - interval '%s MINUTES'";
    private static final String CLEAR_QUERY = "TRUNCATE request CASCADE";
    private static final String LOCK_AND_GET_QUERY = "INSERT INTO request (id, result, appearance_time) VALUES (:reqId, :result, now()) ON CONFLICT(id) DO update SET result=request.result returning result";
    private static final String CREATE_REQUEST_QUERY = "INSERT INTO request (id, result, appearance_time) VALUES (:reqId, :result, now()) ON CONFLICT(id) DO update SET result=:result";
    private static final String NOWAIT_LOCK_QUERY = "select * from pg_try_advisory_xact_lock(:lockId) as locked";
    private static final String GET_PREV_RESULT_QUERY = "select * from request as r where r.id = :reqId";

    @Autowired
    private DiJdbcTemplate jdbcTemplate;

    /**
     * time to live for request id in minutes
     */
    private final long ttl;

    //todo make volatile?
    private final ReentrantLock[] locks;

    private static final AutoCloseable STUB = new AutoCloseable() {
        @Override
        public void close() throws Exception {
        }
    };

    public RequestDbManager(final long ttl, final int locksCount) {
        this.ttl = ttl;
        assert locksCount > 0;
        this.locks = new ReentrantLock[locksCount];
        for (int i = 0; i < locksCount; i++) {
            locks[i] = new ReentrantLock();
        }
    }

    @Override
    @SecondaryOperation
    public boolean removeOld() {
        //no idea why I can't set ttl via ps params
        return jdbcTemplate.update(String.format(REMOVE_OLD_QUERY_PATTERN, ttl), Collections.emptyMap()) > 0;
    }

    @Override
    public boolean clear() {
        return jdbcTemplate.update(CLEAR_QUERY) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean createRequest(@NotNull final String reqId, @NotNull final String data) {
        return jdbcTemplate.update(CREATE_REQUEST_QUERY, toParams(reqId, data)) > 0;
    }

    @Override
    public AutoCloseable localLock(@Nullable final String reqId, final long timeoutNanos) throws ConcurrentOperationException {
        //can't take lock for request without reqId
        if (reqId == null) {
            return STUB;
        }
        final Stopwatch stopwatch = Stopwatch.createStarted();
        final int lockId = lockId(reqId);
        try {
            if (!locks[lockId].tryLock(timeoutNanos, TimeUnit.NANOSECONDS)) {
                throw new ConcurrentOperationException("Can't get local lock for reqId = [" + reqId + "] [" + timeoutNanos + " ns.]");
            }
            return () -> locks[lockId].unlock();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        } finally {
            final long elapsedTimeMs = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            if (elapsedTimeMs < 100L) {
                LOG.debug("local lock took {} ms.", elapsedTimeMs);
            } else {
                LOG.warn("local lock took {} ms.", elapsedTimeMs);
            }
        }
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Optional<String> lockOrResult(final @NotNull String reqId) {
        final Optional<Boolean> isLocked = jdbcTemplate.query(NOWAIT_LOCK_QUERY, toParams(reqId), (rs, i) -> {
            return "t".equals(rs.getString("locked"));
        })
                .stream()
                .filter(Objects::nonNull)
                .findAny();

        if (!isLocked.isPresent() || !isLocked.get()) {
            throw new ConcurrentOperationException();
        }

        return jdbcTemplate.query(GET_PREV_RESULT_QUERY, toParams(reqId), (rs, i) -> rs.getString("result"))
                .stream()
                .findAny();
    }

    public MapSqlParameterSource toParams(@NotNull final String reqId) {
        return new MapSqlParameterSource()
                .addValue("reqId", reqId)
                .addValue("lockId", reqId.hashCode())
                .addValue("result", null);
    }

    @NotNull
    public MapSqlParameterSource toParams(@NotNull final String reqId, @Nullable final String data) {
        return toParams(reqId)
                .addValue("result", data);
    }

    private int lockId(@NotNull final String reqId) {
        return Math.abs(reqId.hashCode()) % locks.length;
    }
}
