package ru.yandex.qe.dispenser.domain.resources_model;

import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;

@JsonIgnoreProperties(ignoreUnknown = true)
public final class InternalResource {

    private final long resourceId;
    private final Set<Long> segmentIds;
    private final long bigOrderId;
    private final long amount;

    @JsonCreator
    public InternalResource(@JsonProperty("resourceId") long resourceId,
                            @JsonProperty("segmentIds") Set<Long> segmentIds,
                            @JsonProperty("bigOrderId") long bigOrderId,
                            @JsonProperty("amount") long amount) {
        this.resourceId = resourceId;
        this.segmentIds = segmentIds;
        this.bigOrderId = bigOrderId;
        this.amount = amount;
    }

    public static Builder builder() {
        return new Builder();
    }

    public long getResourceId() {
        return resourceId;
    }

    public Set<Long> getSegmentIds() {
        return segmentIds;
    }

    public long getBigOrderId() {
        return bigOrderId;
    }

    public long getAmount() {
        return amount;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        InternalResource that = (InternalResource) o;
        return resourceId == that.resourceId &&
                bigOrderId == that.bigOrderId &&
                amount == that.amount &&
                Objects.equals(segmentIds, that.segmentIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resourceId, segmentIds, bigOrderId, amount);
    }

    @Override
    public String toString() {
        return "InternalResource{" +
                "resourceId=" + resourceId +
                ", segmentIds=" + segmentIds +
                ", bigOrderId=" + bigOrderId +
                ", amount=" + amount +
                '}';
    }

    public static final class Builder {

        private final Set<Long> segmentIds = new HashSet<>();

        private Long resourceId;
        private Long bigOrderId;
        private Long amount;

        private Builder() {
        }

        public Builder resourceId(long resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder addSegmentId(long segmentId) {
            this.segmentIds.add(segmentId);
            return this;
        }

        public Builder addSegmentIds(Collection<? extends Long> segmentIds) {
            this.segmentIds.addAll(segmentIds);
            return this;
        }

        public Builder bigOrderId(long bigOrderId) {
            this.bigOrderId = bigOrderId;
            return this;
        }

        public Builder amount(long amount) {
            this.amount = amount;
            return this;
        }

        public InternalResource build() {
            Objects.requireNonNull(resourceId, "ResourceId is required");
            Objects.requireNonNull(bigOrderId, "BigOrderId is required");
            Objects.requireNonNull(amount, "Amount is required");
            return new InternalResource(resourceId, segmentIds, bigOrderId, amount);
        }

    }

}
