package ru.yandex.qe.dispenser.domain.resources_model;

import java.math.RoundingMode;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.ImmutableSet;

import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;

public final class ResourcesModelMapping {

    private final long id;
    private final Resource resource;
    private final Set<Segment> segments;
    private final long campaignId;
    private final ResourceModelMappingTarget target;
    private final RoundingMode rounding;

    protected ResourcesModelMapping(long id, Resource resource, Iterable<Segment> segments, long campaignId,
                                    ResourceModelMappingTarget target, RoundingMode rounding) {
        this.id = id;
        this.resource = Objects.requireNonNull(resource, "Resource is required");
        this.segments = ImmutableSet.copyOf(Objects.requireNonNull(segments, "Segments are required"));
        this.campaignId = campaignId;
        this.target = target;
        this.rounding = rounding;
    }

    public static ResourcesModelMapping.Builder builder() {
        return new ResourcesModelMapping.Builder();
    }

    public static ResourcesModelMapping.Builder builder(ResourcesModelMapping data) {
        return new ResourcesModelMapping.Builder(data);
    }

    public long getId() {
        return id;
    }

    public Resource getResource() {
        return resource;
    }

    public Set<Segment> getSegments() {
        return segments;
    }

    public long getCampaignId() {
        return campaignId;
    }

    public Optional<ResourceModelMappingTarget> getTarget() {
        return Optional.ofNullable(target);
    }

    public RoundingMode getRounding() {
        return rounding;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourcesModelMapping that = (ResourcesModelMapping) o;
        return id == that.id &&
                campaignId == that.campaignId &&
                Objects.equals(resource, that.resource) &&
                Objects.equals(segments, that.segments) &&
                Objects.equals(target, that.target);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, resource, segments, campaignId, target);
    }

    @Override
    public String toString() {
        return "ResourcesModelMapping{" +
                "id=" + id +
                ", resource=" + resource +
                ", segments=" + segments +
                ", campaignId=" + campaignId +
                ", target=" + target +
                '}';
    }

    public static final class Builder {

        private final Set<Segment> segments = new HashSet<>();

        private Long id;
        private Resource resource;
        private Long campaignId;
        private ResourceModelMappingTarget target;
        private RoundingMode rounding;

        private Builder() {
        }

        private Builder(ResourcesModelMapping initialData) {
            this.id = initialData.id;
            this.resource = initialData.resource;
            this.segments.addAll(initialData.segments);
            this.campaignId = initialData.campaignId;
            this.target = initialData.target;
            this.rounding = initialData.rounding;
        }

        public Builder id(long id) {
            this.id = id;
            return this;
        }

        public Builder resource(Resource resource) {
            this.resource = resource;
            return this;
        }

        public Builder addSegment(Segment segment) {
            this.segments.add(segment);
            return this;
        }

        public Builder addSegments(Collection< ? extends Segment> segments) {
            this.segments.addAll(segments);
            return this;
        }

        public Builder campaignId(long campaignId) {
            this.campaignId = campaignId;
            return this;
        }

        public Builder target(ResourceModelMappingTarget target) {
            this.target = target;
            return this;
        }

        public Builder rounding(RoundingMode rounding) {
            this.rounding = rounding;
            return this;
        }

        public Optional<Long> getId() {
            return Optional.ofNullable(id);
        }

        public Optional<Resource> getResource() {
            return Optional.ofNullable(resource);
        }

        public Set<Segment> getSegments() {
            return segments;
        }

        public Optional<Long> getCampaignId() {
            return Optional.ofNullable(campaignId);
        }

        public Optional<ResourceModelMappingTarget> getTarget() {
            return Optional.ofNullable(target);
        }

        public RoundingMode getRounding() {
            return rounding;
        }

        public ResourcesModelMapping build() {
            Objects.requireNonNull(id, "Id is required");
            Objects.requireNonNull(resource, "Resource is required");
            Objects.requireNonNull(campaignId, "CampaignId is required");
            return new ResourcesModelMapping(id, resource, segments, campaignId, target, rounding);
        }

        public ResourcesModelMapping build(long id) {
            Objects.requireNonNull(resource, "Resource is required");
            Objects.requireNonNull(campaignId, "CampaignId is required");
            return new ResourcesModelMapping(id, resource, segments, campaignId, target, rounding);
        }

    }

}
