package ru.yandex.qe.dispenser.domain.util;

import java.sql.Timestamp;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.qe.dispenser.api.v1.DiOrder;
import ru.yandex.qe.dispenser.domain.HistoryEvent;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequestHistoryEvent;
import ru.yandex.qe.dispenser.domain.dao.history.HistoryFilter;
import ru.yandex.qe.dispenser.domain.dao.history.request.QuotaChangeRequestHistoryFilter;

import static ru.yandex.qe.dispenser.domain.util.CollectionUtils.ids;

public final class HistoryDaoUtils {
    private HistoryDaoUtils() {}

    public static <T extends HistoryEvent> RelativePage<T> relativePage(final Stream<T> filteredStream, final RelativePageInfo pageInfo, final DiOrder order) {
        final Comparator<Long> idComparator = order.comparing(Long::longValue);
        final List<T> elements = filteredStream
                .sorted(order.comparing(HistoryEvent::getId))
                .filter(e -> idComparator.compare(e.getId(), pageInfo.getFromId()) >= 0)
                .collect(Collectors.toList());
        if (elements.size() > pageInfo.getPageSize()) {
            return RelativePage.middle(elements.subList(0, (int) pageInfo.getPageSize()));
        }
        return RelativePage.last(elements);
    }

    public static <T extends HistoryEvent> Predicate<T> asPredicate(final HistoryFilter historyFilter) {
        return e -> {
            if (historyFilter.getFrom().isPresent()) {
                if (e.getUpdated().isBefore(historyFilter.getFrom().get())) {
                    return false;
                }
            }
            if (historyFilter.getTo().isPresent()) {
                if (e.getUpdated().isAfter(historyFilter.getTo().get())) {
                    return false;
                }
            }
            if (!historyFilter.getPerformers().isEmpty()) {
                if (!ids(historyFilter.getPerformers()).contains(e.getPersonId())) {
                    return false;
                }
            }
            if (!historyFilter.getTvmIds().isEmpty()) {
                if (!historyFilter.getTvmIds().contains(e.getTvmId())) {
                    return false;
                }
            }
            return true;
        };
    }

    public static Predicate<QuotaChangeRequestHistoryEvent> asPredicate(final QuotaChangeRequestHistoryFilter historyFilter) {
        return e -> {
            if (!asPredicate((HistoryFilter) historyFilter).test(e)) {
                return false;
            }
            if (!historyFilter.getEventTypes().isEmpty()) {
                if (!historyFilter.getEventTypes().contains(e.getType())) {
                    return false;
                }
            }
            return true;
        };
    }

    public static String toCondition(final HistoryFilter filter, final boolean single) {
        final StringJoiner joiner = new StringJoiner(" AND ", " ", "");
        filter.getFrom().ifPresent(from -> joiner.add("updated >= :from"));
        filter.getTo().ifPresent(to -> joiner.add("updated <= :to"));
        if (!filter.getPerformers().isEmpty()) {
            joiner.add("person_id IN (:personIds)");
        }
        if (!filter.getTvmIds().isEmpty()) {
            joiner.add("tvm_id IN (:tvmIds)");
        }
        final String res = joiner.toString();
        if (single || StringUtils.isBlank(res)) {
            return res;
        } else {
            return " AND" + res;
        }
    }

    public static Map<String, Object> toParams(final HistoryFilter filter) {
        final Map<String, Object> params = new HashMap<>();
        params.put("personIds", ids(filter.getPerformers()));
        params.put("tvmIds", filter.getTvmIds());
        filter.getFrom().ifPresent(from -> params.put("from", Timestamp.from(from)));
        filter.getTo().ifPresent(to -> params.put("to", Timestamp.from(to)));
        return params;
    }

    public static String toCondition(final QuotaChangeRequestHistoryFilter filter, final boolean single) {
        final String sub = toCondition((HistoryFilter) filter, single);
        if (!filter.getEventTypes().isEmpty()) {
            final StringJoiner types = new StringJoiner(", ", "(", ")");
            for (int counter = 0; counter < filter.getEventTypes().size(); counter++) {
                types.add("cast (:type_" + counter + " as quota_request_history_event_type)");
            }
            final String condition = " event_type IN " + types.toString();
            if (StringUtils.isBlank(sub) && single) {
                return condition;
            } else {
                return " AND" + condition;
            }
        } else {
            return sub;
        }
    }

    public static Map<String, Object> toParams(final QuotaChangeRequestHistoryFilter filter) {
        final Map<String, Object> params = toParams((HistoryFilter) filter);

        final List<String> typeNames = filter.getEventTypes().stream().map(Enum::name).collect(Collectors.toList());
        for (int i = 0; i < typeNames.size(); i++) {
            params.put("type_" + i, typeNames.get(i));
        }
        return params;
    }

    public static String operatorByOrder(final DiOrder order) {
        switch (order) {
            case ASC:
                return ">=";
            case DESC:
                return "<=";
            default:
                throw new IllegalArgumentException("Incorrect order: " + order);
        }
    }
}
