package ru.yandex.qe.dispenser.domain.util;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collector;
import java.util.stream.Collectors;

import com.google.common.collect.LinkedHashMultimap;
import org.jetbrains.annotations.NotNull;

public enum MoreCollectors {
    ;

    @NotNull
    public static <T, K extends Enum<K>, V> Collector<T, ?, EnumMap<K, V>> toEnumMap(@NotNull final Function<? super T, ? extends K> keyMapper,
                                                                                     @NotNull final Function<? super T, ? extends V> valueMapper,
                                                                                     @NotNull final Class<K> keyClass) {
        return Collectors.toMap(keyMapper, valueMapper, MoreFunctions.throwingMerger(), () -> new EnumMap<>(keyClass));
    }

    @NotNull
    public static <T, K, U> Collector<T, ?, Map<K, U>> toLinkedMap(@NotNull final Function<? super T, ? extends K> keyMapper,
                                                                   @NotNull final Function<? super T, ? extends U> valueMapper) {
        return Collectors.toMap(keyMapper, valueMapper, (u, v) -> {
            throw new IllegalStateException(String.format("Duplicate key %s", u));
        }, LinkedHashMap::new);
    }

    @NotNull
    public static <T, K, U> Collector<T, ?, LinkedHashMultimap<K, U>> toLinkedMultimap(@NotNull final Function<? super T, ? extends K> keyMapper,
                                                                             @NotNull final Function<? super T, ? extends U> valueMapper) {
        return new Collector<T, LinkedHashMultimap<K, U>, LinkedHashMultimap<K, U>>() {
            @NotNull
            @Override
            public Supplier<LinkedHashMultimap<K, U>> supplier() {
                return LinkedHashMultimap::create;
            }

            @NotNull
            @Override
            public BiConsumer<LinkedHashMultimap<K, U>, T> accumulator() {
                return (map, element) -> map.put(keyMapper.apply(element), valueMapper.apply(element));
            }

            @NotNull
            @Override
            public BinaryOperator<LinkedHashMultimap<K, U>> combiner() {
                return (map1, map2) -> {
                    map1.putAll(map2);
                    return map1;
                };
            }

            @NotNull
            @Override
            public Function<LinkedHashMultimap<K, U>, LinkedHashMultimap<K, U>> finisher() {
                return Function.identity();
            }

            @NotNull
            @Override
            public Set<Characteristics> characteristics() {
                return EnumSet.of(Characteristics.IDENTITY_FINISH);
            }
        };
    }

    @NotNull
    public static <T, K, U> Collector<T, ?, Map<K, U>> toLeftMap(@NotNull final Function<? super T, ? extends K> keyMapper,
                                                                 @NotNull final Function<? super T, ? extends U> valueMapper) {
        return Collectors.toMap(keyMapper, valueMapper, MoreFunctions.leftProjection());
    }

    @NotNull
    public static <T, K, U> Collector<T, ?, Map<K, U>> toRightMap(@NotNull final Function<? super T, ? extends K> keyMapper,
                                                                  @NotNull final Function<? super T, ? extends U> valueMapper) {
        return Collectors.toMap(keyMapper, valueMapper, MoreFunctions.rightProjection());
    }

    @NotNull
    public static <T, K, V> Collector<T, ?, Map<K, List<V>>> toMultiValueMap(@NotNull final Function<? super T, ? extends K> keyMapper,
                                                                             @NotNull final Function<? super T, ? extends V> valueMapper) {
        return Collector.of(HashMap::new, (map, entry) -> map.computeIfAbsent(keyMapper.apply(entry), k -> new ArrayList<>()).add(valueMapper.apply(entry)), (map1, map2) -> {
            map1.putAll(map2);
            return map1;
        });
    }
}
