package ru.yandex.qe.dispenser.domain.util;

import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Collection;
import java.util.Objects;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.v1.request.quota.Body;
import ru.yandex.qe.dispenser.api.v1.request.quota.BodyUpdate;
import ru.yandex.qe.dispenser.api.v1.request.quota.ChangeBody;
import ru.yandex.qe.dispenser.domain.exception.SingleMessageException;

@ParametersAreNonnullByDefault
public final class ValidationUtils {
    private static final Pattern REQ_ID_PATTERN = Pattern.compile("[A-Za-z0-9_-]+"); //thread safe
    private static final Pattern KEY_PATTERN = Pattern.compile("[A-Za-z0-9-_]+");
    private static final Pattern PROJECT_KEY_PATTERN = Pattern.compile("[A-Za-z0-9-_.]+");
    private static final Pattern ENTITY_KEY_PATTERN = REQ_ID_PATTERN;
    private static final Pattern YA_TEAM_MAIL_PATTERN = Pattern.compile("[A-Z0-9a-z_-]+@yandex-team\\.ru");

    private ValidationUtils() {
    }

    public static String validateReqId(@NotNull final String reqId) {
        return validate(reqId, REQ_ID_PATTERN, "Wrong reqId format");
    }

    public static String validateEntityKey(@NotNull final String key) {
        return validate(key, ENTITY_KEY_PATTERN, "Wrong entity key format");
    }

    public static String validateProjectKey(@NotNull final String key) {
        return validate(key, PROJECT_KEY_PATTERN, "Wrong project key format");
    }

    public static String validateServiceKey(@NotNull final String key) {
        return validateKey(key, "Wrong service key format");
    }

    public static String validateResourceKey(@NotNull final String key) {
        return validateKey(key, "Wrong resource key format");
    }

    public static String validateKey(@NotNull final String key, @NotNull final String msg) {
        return validate(key, KEY_PATTERN, msg);
    }

    private static String validate(@NotNull final String s, @NotNull final Pattern p, @NotNull final String msg) {
        if (!check(s, p)) {
            throw new IllegalArgumentException(msg + " ('" + s + "' does not match regexp '" + p + "')");
        }
        return s;
    }

    private static boolean check(@NotNull final String s, @NotNull final Pattern p) {
        return p.matcher(s).matches();
    }

    @Nullable
    public static Integer validateAbcServiceId(@Nullable final Integer abcServiceId) {
        // TODO check that ID is valid via ABC API
        if (abcServiceId != null && abcServiceId <= 0) {
            throw SingleMessageException.illegalArgument("valid.abc.service.id.required");
        }
        return abcServiceId;
    }

    /**
     * Similar to {@link java.util.Objects#requireNonNull(java.lang.Object, java.lang.String)}, but throws IllegalArgumentException
     * to be mapped then to 400 "Bad Request" response code in {@link ru.yandex.qe.bus.exception.ExceptionToResponseMapper}.
     **/
    @NotNull
    public static <T> T requireNonNull(@Nullable final T t, @NotNull final String message) {
        if (t == null) {
            throw new IllegalArgumentException(message);
        }
        return t;
    }

    public static <T> void requireEquals(final T t1, final T t2, @NotNull final String message) {
        if (!Objects.equals(t1, t2)) {
            throw new IllegalArgumentException(message);
        }
    }

    public static boolean isValidURL(final String urlToTest) {
        final URL url;

        try {
            final String encodedUrl = urlToTest.replace("|", "%7C"); // DISPENSER-1886
            url = new URL(encodedUrl);
        } catch (MalformedURLException e) {
            return false;
        }

        try {
            url.toURI();
        } catch (URISyntaxException e) {
            return false;
        }

        return true;
    }

    public static String validateYaTeamMail(final String mail) {
        return validate(mail, YA_TEAM_MAIL_PATTERN, "Invalid mail format");
    }

    public static <T> T requireNonNullFieldLocalized(@Nullable final T field, final String fieldName) {
        if (field == null) {
            throw SingleMessageException.illegalArgument("validation." + fieldName + ".required");
        }
        return field;
    }

    public static void requireNotEmptyFieldLocalized(@NotNull final Collection<?> field, final String fieldName) {
        if (field.isEmpty()) {
            throw SingleMessageException.illegalArgument("validation." + fieldName + ".required.not.empty");
        }
    }

    public static void validateQuotaChangeRequestCreateBody(final Body body) {
        requireNonNullFieldLocalized(body.getType(), "type");
        requireNonNullFieldLocalized(body.getChanges(), "changes");
        requireNotEmptyFieldLocalized(body.getChanges(), "changes");
        validateChangeBodies(body.getChanges());
        requireNonNullFieldLocalized(body.getProjectKey(), "projectKey");
    }

    public static void validateQuotaChangeRequestUpdateBody(final BodyUpdate update) {
        if (update.getChanges() != null && !update.getChanges().isEmpty()) {
            validateChangeBodies(update.getChanges());
        }
    }

    public static void validateChangeBodies(final Collection<ChangeBody> changeBodies) {
        changeBodies.forEach(ValidationUtils::validateChangeBody);
    }

    public static void validateChangeBody(final ChangeBody changeBody) {
        requireNonNullFieldLocalized(changeBody.getServiceKey(), "change.serviceKey");
        requireNonNullFieldLocalized(changeBody.getResourceKey(), "change.resourceKey");
        requireNonNullFieldLocalized(changeBody.getSegmentKeys(), "change.segmentKeys");
        requireNonNullFieldLocalized(changeBody.getAmount(), "change.amount");
    }
}
