package ru.yandex.qe.dispenser.domain.util.clickhouse;


import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.fluent.Executor;
import org.apache.http.client.fluent.Request;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.util.EntityUtils;


public final class ClickHouseClient {

    private final String host;
    private final int port;
    private final int connectionTimeout;
    private final String database;
    private final Executor executor;

    private ClickHouseClient(final String host, final int port, final int connectionTimeout, final String user,
                             final String password, final String database) {
        if (host == null) {
            throw new ClickHouseClientException("Host cannot be null!");
        }
        this.host = host;
        if (port <= 0) {
            throw new ClickHouseClientException("Invalid port!");
        }
        this.port = port;
        if (connectionTimeout < 0) {
            throw new ClickHouseClientException("Invalid connetion timeout!");
        }
        this.connectionTimeout = connectionTimeout;
        if (user == null || user.isEmpty()) {
            throw new ClickHouseClientException("User cannot be null or empty!");
        }
        if (password == null) {
            throw new ClickHouseClientException("Password cannot be null!");
        }
        if (database == null || database.isEmpty()) {
            throw new ClickHouseClientException("Database cannot be null or empty!");
        }
        this.database = database;
        executor = Executor.newInstance().auth(user, password);
    }

    private String request(final String query, final Function<URI, Request> requestBuilder) {
        final URIBuilder builder = new URIBuilder();
        builder.setScheme("http").setHost(host).setPort(port);
        if (query != null && !query.isEmpty()) {
            builder.setParameter("query", query);
        }
        try {
            final HttpResponse r = executor.execute(requestBuilder.apply(builder.build()).connectTimeout(connectionTimeout)).returnResponse();
            final String result = EntityUtils.toString(r.getEntity());
            if (r.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                return result;
            } else {
                throw new ClickHouseClientException(result);
            }
        } catch (IOException | URISyntaxException e) {
            throw new ClickHouseClientException(String.format("Cannot connect to ClickHouse host %1$s:%2$d!", host, port), e);
        }
    }

    private String getRequest(final String query) {
        return request(query, Request::Get);
    }

    private String postRequest(final String query) {
        return request(query, Request::Post);
    }

    private String postRequest(final String query, final InputStream data) {
        final InputStreamEntity entity = new InputStreamEntity(data, -1, ContentType.APPLICATION_OCTET_STREAM);
        entity.setChunked(true);
        return request(query, uri -> Request.Post(uri).body(entity));
    }

    public boolean ping() {
        try {
            return getRequest(null).equals("Ok.\n");
        } catch (ClickHouseClientException ignore) {
            return false;
        }
    }

    private String ifNotExists(final boolean failIfExists) {
        return !failIfExists ? "IF NOT EXISTS" : "";
    }

    private String ifExists(final boolean failIfExists) {
        return !failIfExists ? "IF EXISTS" : "";
    }

    public void createDatabase(final boolean failIfExists) {
        postRequest(String.format("CREATE DATABASE %1$s %2$s", ifNotExists(failIfExists), database));
    }

    public boolean tableExists(final String name) {
        return getRequest(String.format("EXISTS TABLE %1$s.%2$s FORMAT CSV", database, name)).equals("1\n");
    }

    public void dropTable(final String name, final boolean failIfExists) {
        postRequest(String.format("DROP TABLE %1$s %2$s.%3$s", ifExists(failIfExists), database, name));
    }

    public void createTable(final String name, final ClickHouseEngine engine, final boolean failIfExists, final Map<String, ClickHouseDataType> fields) {
        final String fieldsDefinitions = fields.entrySet().stream().
                map(entry -> entry.getKey() + " " + entry.getValue().name()).
                collect(Collectors.joining(","));
        postRequest(String.format("CREATE TABLE %1$s %2$s.%3$s (%4$s) ENGINE = %5$s",
                ifNotExists(failIfExists),
                database,
                name,
                fieldsDefinitions,
                engine.name()
        ));
    }

    public void insert(final String name, final InsertDataProvider provider, final String... fields) {
        final String projection = fields.length == 0 ? "" : "(" + Arrays.stream(fields).collect(Collectors.joining(",")) + ")";
        postRequest(String.format("INSERT INTO %1$s.%2$s %3$s FORMAT TabSeparated", database, name, projection), provider);
    }

    public static final class Builder {

        private String host;
        private int port;
        private String database;
        private int connectionTimeout = 1500;
        private String user = "default";
        private String password = "";

        public Builder setHost(final String host) {
            this.host = host;
            return this;
        }

        public Builder setPort(final int port) {
            this.port = port;
            return this;
        }

        public Builder setDatabase(final String database) {
            this.database = database;
            return this;
        }

        public Builder setConnectionTimeout(final int connectionTimeout) {
            this.connectionTimeout = connectionTimeout;
            return this;
        }

        public Builder setUser(final String user) {
            this.user = user;
            return this;
        }

        public Builder setPassword(final String password) {
            this.password = password;
            return this;
        }

        public ClickHouseClient build() {
            return new ClickHouseClient(host, port, connectionTimeout, user, password, database);
        }

    }

}
