package ru.yandex.qe.dispenser.solomon;

import java.util.Arrays;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaSpec;

public final class Solomon {
    private static final String STATISTICS_LINK_TEMPLATE = "https://solomon.yandex-team.ru/?project=%s&cluster=%s&service=%s&graph=auto&l.sensor=%s&l.attribute=actual|max&b=31d&stack=0&secondaryGraphMode=none";

    private static final String GPU_YT_STATISTICS_LINK_TEMPLATE = "https://solomon.yandex-team.ru/?project=yt&cluster=hahn&service=yt_scheduler&graph=gpu-utilization&l.tree=%s&l.pool=nirvana-%s&autorefresh=y&b=31d&e=";

    private static final Map<String, String> GPU_RESOURCE_KEY_TO_YT_TREE = Map.of(
            "geforce-1080ti-gpu", "gpu_geforce_1080ti",
            "tesla-k40-gpu", "gpu_tesla_k40",
            "tesla-m40-gpu", "gpu_tesla_m40",
            "tesla-p40-gpu", "gpu_tesla_p40",
            "tesla-v100-gpu", "gpu_tesla_v100",
            "tesla-a100-80g-gpu", "gpu_tesla_a100_80g",
            "tesla-a100-gpu", "gpu_tesla_a100",
            "tesla-v100-nvlink-gpu", "gpu_tesla_v100_nvlink"
    );
    private static final String NIRVANA = "nirvana";

    @Value("${statistics.solomon.project}")
    private String project;
    @Value("${statistics.solomon.cluster}")
    private String cluster;
    @Value("${statistics.solomon.service}")
    private String service;

    @NotNull
    public String getStatisticsLink(@NotNull final Quota.Key quotaKey) {
        return getStatisticsLink(quotaKey.getSpec(), quotaKey.getProject());
    }

    @NotNull
    public String getStatisticsLink(@NotNull final QuotaSpec quotaSpec, @NotNull Project quotaProject) {
        final String resourceKey = quotaSpec.getResource().getPublicKey();
        final String serviceKey = quotaSpec.getResource().getService().getKey();

        if (NIRVANA.equals(serviceKey) && GPU_RESOURCE_KEY_TO_YT_TREE.containsKey(resourceKey)) {
            return String.format(GPU_YT_STATISTICS_LINK_TEMPLATE,
                    GPU_RESOURCE_KEY_TO_YT_TREE.get(resourceKey), quotaProject.getPublicKey());
        }

        return String.format(STATISTICS_LINK_TEMPLATE,
                        project, cluster, service, getSensorName(quotaSpec, quotaProject)
                )
                .replaceAll(" ", "%20")
                .replaceAll("\\|", "%7C");
    }

    @NotNull
    public static String getSegmentsLabelValue(@NotNull final Quota.@NotNull Key quotaKey) {
        return quotaKey.getSegments().stream()
                .map(segment -> segment.getKey().getSegmentation().getName() + ":" +
                        (segment.isAggregationSegment() ? "aggregation" : segment.getName()))
                .collect(Collectors.joining(", "));
    }

    @Nonnull
    public static String getSensorName(@Nonnull final Quota.Key quotaKey) {
        return getSensorName(quotaKey.getSpec(), quotaKey.getProject());
    }

    @Nonnull
    public static String getSensorName(@NotNull final QuotaSpec quotaSpec, @NotNull Project quotaProject) {
        final String sk = quotaSpec.getResource().getService().getKey();
        final String rk = quotaSpec.getResource().getKey().getPublicKey();
        if ("cluster-api".equals(sk)) {
            return getOldstyleSensorName(quotaSpec, quotaProject);
        }
        // TODO unused resources, do not report data to Solomon
        if (NIRVANA.equals(sk) && Arrays.asList("yt-chunk", "yt-node").contains(rk)) {
            return getOldstyleSensorName(quotaSpec, quotaProject);
        }
        return getNewstyleSensorName(quotaSpec, quotaProject);
    }

    @Nonnull
    private static String getNewstyleSensorName(@NotNull final QuotaSpec quotaSpec, @NotNull Project quotaProject) {
        final String pk = quotaProject.getPublicKey();
        final String sk = quotaSpec.getResource().getService().getKey();
        final String rk = quotaSpec.getResource().getKey().getPublicKey();
        final String qsk = quotaSpec.getKey().getPublicKey();
        return String.format("%s: %s/%s/%s", pk, sk, rk, qsk);
    }

    @Nonnull
    @Deprecated
    private static String getOldstyleSensorName(@NotNull final QuotaSpec quotaSpec, @NotNull Project quotaProject) {
        final String pk = quotaProject.getPublicKey();
        final String sn = quotaSpec.getResource().getService().getName();
        final String rn = quotaSpec.getResource().getName();
        final String sd = quotaSpec.getDescription();
        return String.format("%s/%s/%s (%s)", sn, pk, sd, rn);
    }
}
