package ru.yandex.qe.dispenser.flyway;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.sql.DataSource;

import com.google.common.base.Preconditions;
import org.flywaydb.core.Flyway;
import org.flywaydb.core.api.ErrorDetails;
import org.flywaydb.core.api.Location;
import org.flywaydb.core.api.MigrationInfo;
import org.flywaydb.core.api.MigrationVersion;
import org.flywaydb.core.api.configuration.ClassicConfiguration;
import org.flywaydb.core.internal.info.MigrationInfoImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.flywaydb.core.api.MigrationType.BASELINE;
import static org.flywaydb.core.api.MigrationType.SCHEMA;

public class FlywayMigration {

    public static final String DEFAULT_TABLE = "schema_version";

    public static final String DEFAULT_SCHEMAS = "";

    public static final Action DEFAULT_ACTION = FlywayMigration.Action.NONE;

    public static final String DEFAULT_INIT_LOCATIONS = "db/init";

    public static final String DEFAULT_LOCATIONS = "db/migration";

    public static final String DEFAULT_PLACEHOLDER_PREFIX = "${";

    public static final String DEFAULT_PLACEHOLDER_SUFFIX = "}";

    private static Logger LOG = LoggerFactory.getLogger(FlywayMigration.class);

    private DataSource dataSource;

    private Action action;

    private Location[] initLocations;

    private Location[] locations;

    private String[] schemas;

    private String table;

    private Map<String, String> placeholders;

    private String placeholderPrefix;

    private String placeholderSuffix;

    private boolean failOnMissingMigration;

    public FlywayMigration() {
        setAction(DEFAULT_ACTION);
        setInitLocations(DEFAULT_INIT_LOCATIONS);
        setLocations(DEFAULT_LOCATIONS);
        setSchemas(DEFAULT_SCHEMAS);
        setTable(DEFAULT_TABLE);
        setPlaceholderPrefix(DEFAULT_PLACEHOLDER_PREFIX);
        setPlaceholderSuffix(DEFAULT_PLACEHOLDER_SUFFIX);
        setPlaceholders(Collections.<String, String>emptyMap());
    }

    public void setAction(Action action) {
        if (action == null) {
            this.action = DEFAULT_ACTION;
        }
        this.action = action;
    }

    public void setDataSource(DataSource dataSource) {
        Preconditions.checkNotNull(dataSource);
        this.dataSource = dataSource;
    }

    public void setTable(String table) {
        Preconditions.checkNotNull(table);
        Preconditions.checkState(!table.trim().isEmpty());
        this.table = table;
    }

    public void setLocations(String locations) {
        Preconditions.checkNotNull(locations);
        Preconditions.checkState(!locations.trim().isEmpty());
        this.locations = Arrays.stream(locations.split(",")).map(Location::new).toArray(Location[]::new);
    }

    public void setInitLocations(String initLocations) {
        Preconditions.checkNotNull(initLocations);
        this.initLocations = initLocations.trim().isEmpty() ? new Location[0] :
                Arrays.stream(initLocations.split(",")).map(Location::new).toArray(Location[]::new);
    }

    public void setSchemas(String schemas) {
        Preconditions.checkNotNull(schemas);
        this.schemas = schemas.trim().isEmpty() ? new String[0] : schemas.split(",");
    }

    public void setPlaceholders(Map<String, String> placeholders) {
        Preconditions.checkNotNull(placeholders);
        this.placeholders = placeholders;
    }

    public void setPlaceholderPrefix(String placeholderPrefix) {
        Preconditions.checkNotNull(placeholderPrefix);
        Preconditions.checkState(!placeholderPrefix.trim().isEmpty());
        this.placeholderPrefix = placeholderPrefix;
    }

    public void setPlaceholderSuffix(String placeholderSuffix) {
        Preconditions.checkNotNull(placeholderSuffix);
        Preconditions.checkState(!placeholderSuffix.trim().isEmpty());
        this.placeholderSuffix = placeholderSuffix;
    }

    public void setFailOnMissingMigration(boolean failOnMissingMigration) {
        this.failOnMissingMigration = failOnMissingMigration;
    }

    public void init() {
        Preconditions.checkNotNull(dataSource);
        LOG.info("Start init action - {}", action);

        if (action == FlywayMigration.Action.NONE) {
            return;
        }

        // Clean
        final Flyway metaInitializer = createMetaInitializer();
        if ((action == FlywayMigration.Action.CLEAN_INIT_MIGRATE || action == FlywayMigration.Action.CLEAN_MIGRATE || action == FlywayMigration.Action.CLEAN)
                && metaInitializer.info().current() != null) {
            LOG.info("Clean db");
            metaInitializer.clean();
        }

        if (action == FlywayMigration.Action.CLEAN) {
            return;
        }

        // Init metadata (e.g. schema_version table)
        if (metaInitializer.info().current() == null) {
            LOG.info("Init db meta");
            metaInitializer.baseline();
        }

        // Initial migrations
        final MigrationInfo[] initInfos = initSchemaIfNeeded(metaInitializer.info().current());

        // Regular migrations
        final Flyway migrator = createMigrator();
        LOG.info("Migrate db schema");
        migrator.migrate();
        LOG.info("Validate db schema");
        validate(migrator.info().all(), initInfos);
        LOG.info("Finish init action - {}", action);
    }

    @Nonnull
    private MigrationInfo[] initSchemaIfNeeded(@Nonnull MigrationInfo lastMetaMigration) {
        final boolean initDbSchema = initLocations.length > 0 &&
                (action == FlywayMigration.Action.CLEAN_INIT_MIGRATE || (action == FlywayMigration.Action.INIT_MIGRATE && lastMetaMigration.getType() == BASELINE));
        final Flyway initializer = initDbSchema || failOnMissingMigration ? createInitializer() : null;
        if (initDbSchema) {
            LOG.info("Init db schema");
            initializer.migrate();
        }
        return initializer == null ? new MigrationInfo[0] : initializer.info().all();
    }

    private Flyway createMetaInitializer() {
        ClassicConfiguration configuration = new ClassicConfiguration();
        configuration.setDataSource(dataSource);
        configuration.setSchemas(schemas);
        configuration.setLocations(new Location("db/meta"));
        configuration.setTable(table);
        configuration.setPlaceholderPrefix(placeholderPrefix);
        configuration.setPlaceholderSuffix(placeholderSuffix);
        configuration.setPlaceholders(placeholders);
        return new Flyway(configuration);
    }

    private Flyway createInitializer() {
        ClassicConfiguration configuration = new ClassicConfiguration();
        configuration.setDataSource(dataSource);
        configuration.setLocations(initLocations);
        configuration.setSchemas(schemas);
        configuration.setTable(table);
        configuration.setPlaceholderPrefix(placeholderPrefix);
        configuration.setPlaceholderSuffix(placeholderSuffix);
        configuration.setPlaceholders(placeholders);
        return new Flyway(configuration);
    }

    private Flyway createMigrator() {
        ClassicConfiguration configuration = new ClassicConfiguration();
        configuration.setValidateOnMigrate(false);
        configuration.setLocations(locations);
        configuration.setDataSource(dataSource);
        configuration.setSchemas(schemas);
        configuration.setTable(table);
        configuration.setPlaceholderPrefix(placeholderPrefix);
        configuration.setPlaceholderSuffix(placeholderSuffix);
        configuration.setPlaceholders(placeholders);
        return new Flyway(configuration);
    }

    private void validate(@Nonnull MigrationInfo[] migrationInfos, @Nonnull MigrationInfo[] initializerInfos) {
        final SortedMap<MigrationVersion, MigrationInfo> initMigrations = Arrays.stream(initializerInfos)
                .collect(Collectors.toMap(
                        MigrationInfo::getVersion,
                        Function.identity(),
                        (a, b) -> { throw new IllegalStateException(); },
                        TreeMap::new
                ));

        MigrationInfo firstMigrationInfo = null;
        for (MigrationInfo migrationInfo : migrationInfos) {
            if ((migrationInfo.getState().isApplied()) && (migrationInfo instanceof MigrationInfoImpl)) {
                MigrationInfoImpl migrationInfoImpl = (MigrationInfoImpl) migrationInfo;
                if ((migrationInfoImpl.getType() != SCHEMA)
                        && (migrationInfoImpl.getType() != BASELINE)) {
                    firstMigrationInfo = migrationInfo;
                }
                if ((migrationInfoImpl.getResolvedMigration() == null)
                        && (migrationInfoImpl.getType() != SCHEMA)
                        && (migrationInfoImpl.getType() != BASELINE)) {
                    final MigrationInfoImpl initMigrationInfoImpl = (MigrationInfoImpl) initMigrations.get(migrationInfoImpl.getVersion());
                    if (initMigrationInfoImpl != null && initMigrationInfoImpl.getResolvedMigration() != null) {
                        continue;
                    }

                    if (failOnMissingMigration) {
                        if (initMigrations.isEmpty() || migrationInfo.getVersion().compareTo(initMigrations.lastKey()) >= 0) {
                            throw new IllegalStateException("Detected applied migration missing on the classpath: " + migrationInfoImpl.getVersion());
                        }
                        // If we have a missing migration which is older than the last init migration available, don't fail
                        LOG.warn("Detected **old** applied migration missing on the classpath: " + migrationInfoImpl.getVersion());
                    } else {
                        LOG.warn("Detected applied migration missing on the classpath: " + migrationInfoImpl.getVersion());
                    }

                    continue;
                }
                ErrorDetails validate = migrationInfoImpl.validate();
                if (validate != null) {
                    throw new IllegalStateException(validate.errorMessage);
                }
            } else if (!(migrationInfo.getState().isApplied())
                    && ((action == FlywayMigration.Action.MIGRATE)
                    || (action == FlywayMigration.Action.CLEAN_MIGRATE)
                    || ((action == FlywayMigration.Action.INIT_MIGRATE) || (action == FlywayMigration.Action.CLEAN_INIT_MIGRATE)) && firstMigrationInfo != null)) {
                throw new IllegalStateException("Out of order migration on the classpath: " + migrationInfo.getVersion());
            }
        }
    }

    @Override
    public String toString() {
        return "FlywayMigration{" +
                "action=" + action +
                ", dataSource=" + dataSource +
                ", initLocations=" + Arrays.toString(initLocations) +
                ", locations=" + Arrays.toString(locations) +
                ", schemas=" + Arrays.toString(schemas) +
                ", table='" + table + '\'' +
                ", placeholderPrefix='" + placeholderPrefix + '\'' +
                ", placeholderSuffix='" + placeholderSuffix + '\'' +
                ", placeholders=" + placeholders +
                '}';
    }

    public static enum Action {
        NONE,
        CLEAN,
        CLEAN_MIGRATE,
        CLEAN_INIT_MIGRATE,
        INIT_MIGRATE,
        MIGRATE
    }
}
