package ru.yandex.qe.jetty.metrics;

import java.io.IOException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.base.Stopwatch;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.handler.HandlerWrapper;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.qe.dispenser.solomon.SolomonHolder;

public class InstrumentedHandler extends HandlerWrapper {

    private static final AtomicLong HANDLER_INDEX = new AtomicLong(0);

    private final Rate dispatchRate;
    private final GaugeInt64 dispatchInProgress;
    private final Histogram dispatchDuration;

    public InstrumentedHandler(SolomonHolder solomonHolder) {
        MetricRegistry metricRegistry = solomonHolder.getRootRegistry();
        String handlerIndex = String.valueOf(HANDLER_INDEX.incrementAndGet());
        this.dispatchRate = metricRegistry.rate("jetty.handler.dispatch_rate",
                Labels.of("handler_index", handlerIndex));
        this.dispatchInProgress = metricRegistry.gaugeInt64("jetty.handler.dispatch_in_progress",
                Labels.of("handler_index", handlerIndex));
        this.dispatchDuration = metricRegistry.histogramRate("jetty.handler.dispatch_duration_millis",
                Labels.of("handler_index", handlerIndex), Histograms.exponential(22, 2.0d, 1.0d));
    }

    @Override
    public void handle(String path,
                       Request request,
                       HttpServletRequest httpRequest,
                       HttpServletResponse httpResponse) throws IOException, ServletException {
        dispatchInProgress.add(1L);
        dispatchRate.inc();
        Stopwatch stopwatch = Stopwatch.createStarted();
        try {
            super.handle(path, request, httpRequest, httpResponse);
        } finally {
            dispatchInProgress.add(-1L);
            long dispatchedIn = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            dispatchDuration.record(dispatchedIn);
        }
    }

}
