package ru.yandex.qe.jetty.server;

import java.util.ArrayList;
import java.util.EnumSet;

import javax.servlet.DispatcherType;
import javax.servlet.Filter;
import javax.servlet.annotation.WebFilter;
import javax.servlet.annotation.WebInitParam;

import org.eclipse.jetty.http.pathmap.ServletPathSpec;
import org.eclipse.jetty.servlet.FilterHolder;
import org.eclipse.jetty.servlet.FilterMapping;
import org.eclipse.jetty.servlet.Source;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author lvovich
 */
public class AnnotationConfiguredFilter {

    private final static Logger LOG = LoggerFactory.getLogger(AnnotationConfiguredFilter.class);


    private final FilterHolder filterHolder;
    private final FilterMapping filterMapping;

    private AnnotationConfiguredFilter(final FilterHolder filterHolder, final FilterMapping filterMapping) {
        this.filterHolder = filterHolder;
        this.filterMapping = filterMapping;
    }

    public FilterHolder getFilterHolder() {
        return filterHolder;
    }

    public FilterMapping getFilterMapping() {
        return filterMapping;
    }

    public static AnnotationConfiguredFilter construct(final Class<?> rawClazz) {
        Class<? extends Filter> clazz = (Class<? extends Filter>) rawClazz;
        if (!Filter.class.isAssignableFrom(clazz))
        {
            LOG.warn(clazz.getName()+" is not assignable from javax.servlet.Filter");
            return null;
        }
        WebFilter filterAnnotation = (WebFilter)clazz.getAnnotation(WebFilter.class);

        if (filterAnnotation.value().length > 0 && filterAnnotation.urlPatterns().length > 0)
        {
            LOG.warn(clazz.getName()+" defines both @WebFilter.value and @WebFilter.urlPatterns");
            return null;
        }

        String name = (filterAnnotation.filterName().equals("")?clazz.getName():filterAnnotation.filterName());
        String[] urlPatterns = filterAnnotation.value();
        if (urlPatterns.length == 0)
            urlPatterns = filterAnnotation.urlPatterns();

        final FilterHolder holder = new FilterHolder(new Source(Source.Origin.ANNOTATION, clazz.getName()));
        holder.setName(name);
        holder.setHeldClass(clazz);
        holder.setDisplayName(filterAnnotation.displayName());
        for (WebInitParam ip:  filterAnnotation.initParams())
        {
            holder.setInitParameter(ip.name(), ip.value());
        }
        holder.setAsyncSupported(filterAnnotation.asyncSupported());

        FilterMapping mapping = new FilterMapping();
        mapping.setFilterName(holder.getName());
        if (urlPatterns.length > 0)
        {
            ArrayList<String> paths = new ArrayList<String>();
            for (String s:urlPatterns)
            {
                paths.add(ServletPathSpec.normalize(s));
            }
            mapping.setPathSpecs(paths.toArray(new String[paths.size()]));
        }

        if (filterAnnotation.servletNames().length > 0)
        {
            ArrayList<String> names = new ArrayList<String>();
            for (String s : filterAnnotation.servletNames())
            {
                names.add(s);
            }
            mapping.setServletNames(names.toArray(new String[names.size()]));
        }

        EnumSet<DispatcherType> dispatcherSet = EnumSet.noneOf(DispatcherType.class);
        for (DispatcherType d : filterAnnotation.dispatcherTypes())
        {
            dispatcherSet.add(d);
        }
        mapping.setDispatcherTypes(dispatcherSet);

        holder.setAsyncSupported(filterAnnotation.asyncSupported());

        return new AnnotationConfiguredFilter(holder, mapping);

    }
}
