package ru.yandex.qe.dispenser.ws.owning_cost

import org.junit.jupiter.api.Assertions
import org.junit.jupiter.api.Test
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Value
import ru.yandex.qe.dispenser.api.v1.*
import ru.yandex.qe.dispenser.api.v1.request.quota.BodyUpdate.BodyUpdateBuilder
import ru.yandex.qe.dispenser.domain.*
import ru.yandex.qe.dispenser.domain.dao.bot.settings.BotCampaignGroupDao
import ru.yandex.qe.dispenser.domain.dao.person.PersonDao
import ru.yandex.qe.dispenser.domain.dao.resource.ResourceDao
import ru.yandex.qe.dispenser.domain.dao.resource.segmentation.ResourceSegmentationDao
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentDao
import ru.yandex.qe.dispenser.domain.dao.segmentation.SegmentationDao
import ru.yandex.qe.dispenser.domain.dao.service.ServiceDao
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy
import ru.yandex.qe.dispenser.ws.logic.BaseQuotaRequestTest
import ru.yandex.qe.dispenser.ws.logic.QuotaChangeRequestValidationTest
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.QuotaChangeOwningCostRefreshManager
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.ProviderOwningCostFormula
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.SolomonOwningCostFormula
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.YtOwningCostFormula
import java.math.BigDecimal
import java.math.RoundingMode
import java.time.LocalDate
import java.time.Month
import java.util.*

class OwningCost2022MappingTest(
    @Autowired private val quotaChangeOwningCostRefreshManager: QuotaChangeOwningCostRefreshManager,
    @Autowired private val botCampaignGroupDao: BotCampaignGroupDao,
    @Autowired private val resourceDao: ResourceDao,
    @Autowired private val segmentationDao: SegmentationDao,
    @Autowired private val segmentDao: SegmentDao,
    @Autowired private val serviceDao: ServiceDao,
    @Autowired private val resourceSegmentationDao: ResourceSegmentationDao,
    @Autowired private val personDao: PersonDao,
    @Value("\${dispenser.distbuild.segmentation.key}") private val distBuildSegmentationKey: String,
    @Value("\${dispenser.location.segmentation.key}") private val locationSegmentationKey: String,
    @Value("\${dispenser.yt.segmentation.key}") private val ytSegmentationKey: String,
    @Value("\${dispenser.ydb.segmentation.key}") private val ydbSegmentationKey: String
): BaseQuotaRequestTest() {

    @Test
    fun testMds() {
        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val mdsService = prepareMdsService()
        updateHierarchy()
        resourceDao.create(Resource.Builder("replicas_x2_hdd", mdsService).name("HDD x2")
            .type(DiResourceType.STORAGE).build())
        resourceDao.create(Resource.Builder("replicas_x2_ssd", mdsService).name("SSD x2")
            .type(DiResourceType.STORAGE).build())
        resourceDao.create(Resource.Builder("replicas_x3_hdd", mdsService).name("HDD x3")
            .type(DiResourceType.STORAGE).build())
        resourceDao.create(Resource.Builder("replicas_x3_ssd", mdsService).name("SSD x3")
            .type(DiResourceType.STORAGE).build())
        resourceDao.create(Resource.Builder("lrc_x1_5_hdd", mdsService).name("HDD lrc x1.5")
            .type(DiResourceType.STORAGE).build())
        updateHierarchy()
        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)
                .changes("mds_new", "replicas_x2_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(1L, DiUnit.GIBIBYTE))
                .changes("mds_new", "replicas_x2_ssd", bigOrderTwo.id, emptySet(), DiAmount.of(2L, DiUnit.GIBIBYTE))
                .changes("mds_new", "replicas_x3_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(3L, DiUnit.GIBIBYTE))
                .changes("mds_new", "replicas_x3_ssd", bigOrderTwo.id, emptySet(), DiAmount.of(4L, DiUnit.GIBIBYTE))
                .changes("mds_new", "lrc_x1_5_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(5L, DiUnit.GIBIBYTE))
                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(5, changes.size)
        val owningCostByResourceKey = changes.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["replicas_x2_hdd"])
        Assertions.assertEquals(BigDecimal("2.84800032").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["replicas_x2_ssd"])
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["replicas_x3_hdd"])
        Assertions.assertEquals(BigDecimal("4.27199976").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["replicas_x3_ssd"])
        Assertions.assertEquals(BigDecimal("0.16750008").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["lrc_x1_5_hdd"])
        quotaChangeOwningCostRefreshManager.refresh()
        val fetchedRequest2 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest2)
        val changes2 = fetchedRequest2.changes
        Assertions.assertEquals(5, changes2.size)
        val owningCostByResourceKey2 = changes2.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["replicas_x2_hdd"])
        Assertions.assertEquals(BigDecimal("2.84800032").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["replicas_x2_ssd"])
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["replicas_x3_hdd"])
        Assertions.assertEquals(BigDecimal("4.27199976").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["replicas_x3_ssd"])
        Assertions.assertEquals(BigDecimal("0.16750008").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["lrc_x1_5_hdd"])
        dispenser().quotaChangeRequests()
            .byId(quotaRequests.first.id)
            .update(BodyUpdateBuilder()
                .changes("mds_new", "replicas_x2_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(10L, DiUnit.GIBIBYTE))
                .changes("mds_new", "replicas_x2_ssd", bigOrderTwo.id, emptySet(), DiAmount.of(20L, DiUnit.GIBIBYTE))
                .changes("mds_new", "replicas_x3_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(30L, DiUnit.GIBIBYTE))
                .changes("mds_new", "replicas_x3_ssd", bigOrderTwo.id, emptySet(), DiAmount.of(40L, DiUnit.GIBIBYTE))
                .changes("mds_new", "lrc_x1_5_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(50L, DiUnit.GIBIBYTE))
                .build())
            .performBy(AMOSOV_F)
        val fetchedRequest3 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest3)
        val changes3 = fetchedRequest3.changes
        Assertions.assertEquals(5, changes3.size)
        val owningCostByResourceKey3 = changes3.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(10L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["replicas_x2_hdd"])
        Assertions.assertEquals(BigDecimal("2.84800032").multiply(BigDecimal(20L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["replicas_x2_ssd"])
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(30L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["replicas_x3_hdd"])
        Assertions.assertEquals(BigDecimal("4.27199976").multiply(BigDecimal(40L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["replicas_x3_ssd"])
        Assertions.assertEquals(BigDecimal("0.16750008").multiply(BigDecimal(50L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["lrc_x1_5_hdd"])
    }

    @Test
    fun testAvatars() {
        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val avatarsService = prepareAvatarsService()
        updateHierarchy()
        resourceDao.create(Resource.Builder("replicas_x2_hdd", avatarsService).name("HDD x2")
            .type(DiResourceType.STORAGE).build())
        resourceDao.create(Resource.Builder("replicas_x2_ssd", avatarsService).name("SSD x2")
            .type(DiResourceType.STORAGE).build())
        resourceDao.create(Resource.Builder("replicas_x3_hdd", avatarsService).name("HDD x3")
            .type(DiResourceType.STORAGE).build())
        resourceDao.create(Resource.Builder("replicas_x3_ssd", avatarsService).name("SSD x3")
            .type(DiResourceType.STORAGE).build())
        resourceDao.create(Resource.Builder("lrc_x1_5_hdd", avatarsService).name("HDD lrc x1.5")
            .type(DiResourceType.STORAGE).build())
        updateHierarchy()
        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)
                .changes("avatars", "replicas_x2_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(1L, DiUnit.GIBIBYTE))
                .changes("avatars", "replicas_x2_ssd", bigOrderTwo.id, emptySet(), DiAmount.of(2L, DiUnit.GIBIBYTE))
                .changes("avatars", "replicas_x3_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(3L, DiUnit.GIBIBYTE))
                .changes("avatars", "replicas_x3_ssd", bigOrderTwo.id, emptySet(), DiAmount.of(4L, DiUnit.GIBIBYTE))
                .changes("avatars", "lrc_x1_5_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(5L, DiUnit.GIBIBYTE))
                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(5, changes.size)
        val owningCostByResourceKey = changes.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["replicas_x2_hdd"])
        Assertions.assertEquals(BigDecimal("2.84800032").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["replicas_x2_ssd"])
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["replicas_x3_hdd"])
        Assertions.assertEquals(BigDecimal("4.27199976").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["replicas_x3_ssd"])
        Assertions.assertEquals(BigDecimal("0.16750008").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["lrc_x1_5_hdd"])
        quotaChangeOwningCostRefreshManager.refresh()
        val fetchedRequest2 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest2)
        val changes2 = fetchedRequest2.changes
        Assertions.assertEquals(5, changes2.size)
        val owningCostByResourceKey2 = changes2.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["replicas_x2_hdd"])
        Assertions.assertEquals(BigDecimal("2.84800032").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["replicas_x2_ssd"])
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["replicas_x3_hdd"])
        Assertions.assertEquals(BigDecimal("4.27199976").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["replicas_x3_ssd"])
        Assertions.assertEquals(BigDecimal("0.16750008").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["lrc_x1_5_hdd"])
        dispenser().quotaChangeRequests()
            .byId(quotaRequests.first.id)
            .update(BodyUpdateBuilder()
                .changes("avatars", "replicas_x2_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(10L, DiUnit.GIBIBYTE))
                .changes("avatars", "replicas_x2_ssd", bigOrderTwo.id, emptySet(), DiAmount.of(20L, DiUnit.GIBIBYTE))
                .changes("avatars", "replicas_x3_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(30L, DiUnit.GIBIBYTE))
                .changes("avatars", "replicas_x3_ssd", bigOrderTwo.id, emptySet(), DiAmount.of(40L, DiUnit.GIBIBYTE))
                .changes("avatars", "lrc_x1_5_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(50L, DiUnit.GIBIBYTE))
                .build())
            .performBy(AMOSOV_F)
        val fetchedRequest3 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest3)
        val changes3 = fetchedRequest3.changes
        Assertions.assertEquals(5, changes3.size)
        val owningCostByResourceKey3 = changes3.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(10L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["replicas_x2_hdd"])
        Assertions.assertEquals(BigDecimal("2.84800032").multiply(BigDecimal(20L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["replicas_x2_ssd"])
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(30L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["replicas_x3_hdd"])
        Assertions.assertEquals(BigDecimal("4.27199976").multiply(BigDecimal(40L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["replicas_x3_ssd"])
        Assertions.assertEquals(BigDecimal("0.16750008").multiply(BigDecimal(50L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["lrc_x1_5_hdd"])
    }

    @Test
    fun testS3() {
        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val s3Service = prepareS3Service()
        updateHierarchy()
        resourceDao.create(Resource.Builder("x2_hdd", s3Service).name("HDD x2")
            .type(DiResourceType.STORAGE).build())
        resourceDao.create(Resource.Builder("x3_hdd", s3Service).name("HDD x3")
            .type(DiResourceType.STORAGE).build())
        updateHierarchy()
        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)
                .changes("s3", "x2_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(1L, DiUnit.GIBIBYTE))
                .changes("s3", "x3_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(2L, DiUnit.GIBIBYTE))
                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(2, changes.size)
        val owningCostByResourceKey = changes.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["x2_hdd"])
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["x3_hdd"])
        quotaChangeOwningCostRefreshManager.refresh()
        val fetchedRequest2 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest2)
        val changes2 = fetchedRequest2.changes
        Assertions.assertEquals(2, changes2.size)
        val owningCostByResourceKey2 = changes2.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["x2_hdd"])
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["x3_hdd"])
        dispenser().quotaChangeRequests()
            .byId(quotaRequests.first.id)
            .update(BodyUpdateBuilder()
                .changes("s3", "x2_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(10L, DiUnit.GIBIBYTE))
                .changes("s3", "x3_hdd", bigOrderTwo.id, emptySet(), DiAmount.of(20L, DiUnit.GIBIBYTE))
                .build())
            .performBy(AMOSOV_F)
        val fetchedRequest3 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest3)
        val changes3 = fetchedRequest3.changes
        Assertions.assertEquals(2, changes3.size)
        val owningCostByResourceKey3 = changes3.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(10L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["x2_hdd"])
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(20L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["x3_hdd"])
    }

    @Test
    fun distBuildOwningCostFormulaTest() {
        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val service = serviceDao.create(Service
            .withKey(DISTBUILD)
            .withName("DISTBUILD")
            .withAbcServiceId(5656)
            .withPriority(190)
            .build())
        updateHierarchy()
        val cpu = resourceDao.create(Resource.Builder(CPU, service).name(CPU).type(DiResourceType.PROCESSOR).build())
        val segmentation = segmentationDao.create(Segmentation.Builder(distBuildSegmentationKey)
            .name(distBuildSegmentationKey)
            .description(distBuildSegmentationKey)
            .priority(1)
            .build())
        updateHierarchy()
        val distBuildAutocheck = segmentDao.create(Segment.Builder("distbuild_autocheck", segmentation)
            .priority(1)
            .name("distbuild_autocheck")
            .description("abc")
            .build())
        val distBuildUser = segmentDao.create(Segment.Builder("distbuild_user", segmentation)
            .priority(1)
            .name("distbuild_user")
            .description("abc")
            .build())
        updateHierarchy()
        resourceSegmentationDao.setSegmentations(cpu, setOf(segmentation))
        updateHierarchy()
        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)
                .changes(DISTBUILD, CPU, bigOrderTwo.id, setOf(distBuildAutocheck.publicKey), DiAmount.of(100L, DiUnit.CORES))
                .changes(DISTBUILD, CPU, bigOrderTwo.id, setOf(distBuildUser.publicKey), DiAmount.of(100L, DiUnit.CORES))
                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        updateHierarchy()
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(2, changes.size)
        val owningCostByResourceKey = changes.associateBy { Pair(it.resource.key, it.segmentKeys.first()) }
            .mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("532.22").multiply(BigDecimal(100L),
            ProviderOwningCostFormula.MATH_CONTEXT).setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey[Pair(CPU, distBuildAutocheck.publicKey)])
        Assertions.assertEquals(BigDecimal("462.8").multiply(BigDecimal(100L),
            ProviderOwningCostFormula.MATH_CONTEXT).setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey[Pair(CPU, distBuildUser.publicKey)])
    }

    @Test
    fun testNirvana() {
        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val service = Hierarchy.get().serviceReader.read(NIRVANA)
        updateHierarchy()
        resourceDao.create(Resource.Builder("s3_hdd_x2", service).name("MDS S3 HDD x2")
            .type(DiResourceType.STORAGE).build())
        updateHierarchy()
        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)
                .changes("nirvana", "s3_hdd_x2", bigOrderTwo.id, emptySet(), DiAmount.of(1L, DiUnit.GIBIBYTE))
                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(1, changes.size)
        val owningCostByResourceKey = changes.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["s3_hdd_x2"])
        quotaChangeOwningCostRefreshManager.refresh()
        val fetchedRequest2 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest2)
        val changes2 = fetchedRequest2.changes
        Assertions.assertEquals(1, changes2.size)
        val owningCostByResourceKey2 = changes2.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["s3_hdd_x2"])
        dispenser().quotaChangeRequests()
            .byId(quotaRequests.first.id)
            .update(BodyUpdateBuilder()
                .changes("nirvana", "s3_hdd_x2", bigOrderTwo.id, emptySet(), DiAmount.of(10L, DiUnit.GIBIBYTE))
                .build())
            .performBy(AMOSOV_F)
        val fetchedRequest3 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest3)
        val changes3 = fetchedRequest3.changes
        Assertions.assertEquals(1, changes3.size)
        val owningCostByResourceKey3 = changes3.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(10L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["s3_hdd_x2"])
    }

    @Test
    fun testSolomon() {
        val cpuPrice = BigDecimal("162.81")
        val ramPrice = BigDecimal("25.12")
        val FLOW = "metrics_write_flow_segmented"
        val METRICS_COUNT = "metrics_stored_count"
        val ALERTS_COUNT = "alerts_count"

        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val service = prepareSolomonService()
        updateHierarchy()
        val metricsStoredCount = createResource(service, METRICS_COUNT, DiResourceType.ENUMERABLE)
        val metricsWriteFlow = createResource(service, FLOW, DiResourceType.ENUMERABLE)
        val alertsCount = createResource(service, ALERTS_COUNT, DiResourceType.ENUMERABLE)
        val locationSegmentation = Hierarchy.get().segmentationReader.read(Segmentation.Key(locationSegmentationKey))
        updateHierarchy()
        createSegment(locationSegmentation, "SAS")
        createSegment(locationSegmentation, "VLA")
        createSegment(locationSegmentation, "MAN")
        createSegment(locationSegmentation, "MYT")
        createSegment(locationSegmentation, "IVA")
        createSegment(locationSegmentation, "KLG")
        resourceSegmentationDao.setSegmentations(metricsWriteFlow, setOf(locationSegmentation))
        updateHierarchy()
        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)
                .changes("solomon", metricsStoredCount.publicKey, bigOrderTwo.id, emptySet(), DiAmount.of(1_000_000L, DiUnit.COUNT))
                .changes("solomon", metricsWriteFlow.publicKey, bigOrderTwo.id, setOf("SAS"), DiAmount.of(200_000L, DiUnit.COUNT))
                .changes("solomon", metricsWriteFlow.publicKey, bigOrderTwo.id, setOf("VLA"), DiAmount.of(200_000L, DiUnit.COUNT))
                .changes("solomon", metricsWriteFlow.publicKey, bigOrderTwo.id, setOf("KLG"), DiAmount.of(200_000L, DiUnit.COUNT))
                .changes("solomon", alertsCount.publicKey, bigOrderTwo.id, emptySet(), DiAmount.of(30000L, DiUnit.COUNT))
                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(5, changes.size)
        val owningCostByResourceKey = changes.associateBy { Pair(it.resource.key, it.segmentKeys) }
            .mapValues { it.value.owningCost }

        val math = SolomonOwningCostFormula.MATH_CONTEXT
        val flowPerServer = BigDecimal(600_000L).divide(BigDecimal(4_000_000L), math)
            .multiply(BigDecimal(56), math)
            .multiply(BigDecimal(2), math)
        val flowCores = BigDecimal(2) * flowPerServer.setScale(0, RoundingMode.UP)
        val flowRam = flowCores * BigDecimal(8)
        val alertsCores = BigDecimal(30_000L).divide(BigDecimal(18_750L), math)
            .setScale(0, RoundingMode.UP)
        val alertsRam = alertsCores * BigDecimal(8)

        Assertions.assertEquals(
            flowCores.multiply(cpuPrice, math).add(flowRam.multiply(ramPrice, math), math)
                .divide(BigDecimal(3), math).setScale(0, RoundingMode.UP).toString(),
            owningCostByResourceKey[Pair(FLOW, setOf("SAS"))])
        Assertions.assertEquals(
            flowCores.multiply(cpuPrice, math).add(flowRam.multiply(ramPrice, math), math)
                .divide(BigDecimal(3), math).setScale(0, RoundingMode.UP).toString(),
            owningCostByResourceKey[Pair(FLOW, setOf("VLA"))])
        Assertions.assertEquals(
            flowCores.multiply(cpuPrice, math).add(flowRam.multiply(ramPrice, math), math)
                .divide(BigDecimal(3), math).setScale(0, RoundingMode.UP).toString(),
            owningCostByResourceKey[Pair(FLOW, setOf("KLG"))])
        Assertions.assertEquals(SolomonOwningCostFormula.DEFAULT_OWNING_COST.toString(),
            owningCostByResourceKey[Pair(METRICS_COUNT, setOf())])
        Assertions.assertEquals(
            alertsCores.multiply(cpuPrice, math).add(alertsRam.multiply(ramPrice, math), math)
                .setScale(0, RoundingMode.UP).toString(),
            owningCostByResourceKey[Pair(ALERTS_COUNT, setOf())])
    }

    @Test
    fun logbrokerOwningCostFormulaTest() {
        val campaign = prepareCampaign()
        val service = serviceDao.create(Service.withKey(LOGBROKER)
            .withName("Logbroker")
            .withAbcServiceId(LOGBROKER_ABC_SERVICE_ID)
            .withSettings(Service.Settings.builder()
                .accountActualValuesInQuotaDistribution(true)
                .requireZeroQuotaUsageForProjectDeletion(true)
                .build())
            .build())
        val slonnn = personDao.readPersonByLogin(SLONNN.login)
        serviceDao.attachAdmin(service, slonnn)
        serviceDao.attachAdmin(service, personDao.readPersonByLogin(ZOMB_MOBSEARCH))
        updateHierarchy()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val dataflowBinaryResource = resourceDao.create(Resource.Builder("data_flow_binary", service)
            .name("data_flow_binary").type(DiResourceType.BINARY_TRAFFIC).build())
        updateHierarchy()
        val location = segmentationDao.create(Segmentation.Builder("logbroker").name("logbroker")
            .description("logbroker").priority(1).build())
        updateHierarchy()
        val lbkx = segmentDao.create(Segment.Builder("lbkx", location).priority(1.toShort())
            .name("lbkx").description("lbkx").build())
        val logbrokerSAS = segmentDao.create(Segment.Builder("logbroker_SAS", location).priority(1.toShort())
            .name("logbroker_SAS").description("logbroker_SAS").build())
        val logbrokerVLA = segmentDao.create(Segment.Builder("logbroker_VLA", location).priority(1.toShort())
            .name("logbroker_VLA").description("logbroker_VLA").build())
        val logbrokerKLG = segmentDao.create(Segment.Builder("logbroker_KLG", location).priority(1.toShort())
            .name("logbroker_KLG").description("logbroker_KLG").build())
        val logbrokerPrestableKLG = segmentDao.create(Segment.Builder("logbroker-prestable_KLG", location)
            .priority(1.toShort()).name("logbroker-prestable_KLG").description("logbroker-prestable_KLG").build())
        val logbrokerYC = segmentDao.create(Segment.Builder("logbroker-yc", location).priority(1.toShort())
            .name("logbroker-yc").description("logbroker-yc").build())
        val logbrokerPreprodYC = segmentDao.create(Segment.Builder("logbroker-yc-preprod", location)
            .priority(1.toShort()).name("logbroker-yc-preprod").description("logbroker-yc-preprod").build())
        updateHierarchy()
        resourceSegmentationDao.setSegmentations(dataflowBinaryResource, setOf(location))
        updateHierarchy()
        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(lbkx.publicKey), DiAmount.of(1L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerSAS.publicKey), DiAmount.of(2L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerVLA.publicKey), DiAmount.of(3L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerKLG.publicKey), DiAmount.of(4L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerPrestableKLG.publicKey), DiAmount.of(5L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerYC.publicKey), DiAmount.of(6L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerPreprodYC.publicKey), DiAmount.of(7L, DiUnit.MIBPS))
                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        updateHierarchy()
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(7, changes.size)
        val owningCostByResourceKey = changes.associateBy { Pair(it.resource.key, it.segmentKeys.first()) }
            .mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("1735.17").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey[Pair("data_flow_binary", "lbkx")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey[Pair("data_flow_binary", "logbroker_SAS")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey[Pair("data_flow_binary", "logbroker_VLA")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey[Pair("data_flow_binary", "logbroker_KLG")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey[Pair("data_flow_binary", "logbroker-prestable_KLG")])
        Assertions.assertEquals(BigDecimal("0").multiply(BigDecimal(6L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey[Pair("data_flow_binary", "logbroker-yc")])
        Assertions.assertEquals(BigDecimal("0").multiply(BigDecimal(7L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey[Pair("data_flow_binary", "logbroker-yc-preprod")])
        quotaChangeOwningCostRefreshManager.refresh()
        updateHierarchy()
        val fetchedRequest2 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest2)
        val changes2 = fetchedRequest2.changes
        Assertions.assertEquals(7, changes2.size)
        val owningCostByResourceKey2 = changes2.associateBy { Pair(it.resource.key, it.segmentKeys.first()) }
            .mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("1735.17").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey2[Pair("data_flow_binary", "lbkx")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey2[Pair("data_flow_binary", "logbroker_SAS")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey2[Pair("data_flow_binary", "logbroker_VLA")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey2[Pair("data_flow_binary", "logbroker_KLG")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey2[Pair("data_flow_binary", "logbroker-prestable_KLG")])
        Assertions.assertEquals(BigDecimal("0").multiply(BigDecimal(6L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey2[Pair("data_flow_binary", "logbroker-yc")])
        Assertions.assertEquals(BigDecimal("0").multiply(BigDecimal(7L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey2[Pair("data_flow_binary", "logbroker-yc-preprod")])
        dispenser().quotaChangeRequests()
            .byId(quotaRequests.first.id)
            .update(BodyUpdateBuilder()
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(lbkx.publicKey), DiAmount.of(10L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerSAS.publicKey), DiAmount.of(20L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerVLA.publicKey), DiAmount.of(30L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerKLG.publicKey), DiAmount.of(40L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerPrestableKLG.publicKey), DiAmount.of(50L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerYC.publicKey), DiAmount.of(60L, DiUnit.MIBPS))
                .changes(LOGBROKER, "data_flow_binary", bigOrderTwo.id, setOf(logbrokerPreprodYC.publicKey), DiAmount.of(70L, DiUnit.MIBPS))
                .build())
            .performBy(AMOSOV_F)
        updateHierarchy()
        val fetchedRequest3 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest3)
        val changes3 = fetchedRequest3.changes
        Assertions.assertEquals(7, changes3.size)
        val owningCostByResourceKey3 = changes3.associateBy { Pair(it.resource.key, it.segmentKeys.first()) }
            .mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("1735.17").multiply(BigDecimal(10L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey3[Pair("data_flow_binary", "lbkx")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(20L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey3[Pair("data_flow_binary", "logbroker_SAS")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(30L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey3[Pair("data_flow_binary", "logbroker_VLA")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(40L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey3[Pair("data_flow_binary", "logbroker_KLG")])
        Assertions.assertEquals(BigDecimal("542.24").multiply(BigDecimal(50L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey3[Pair("data_flow_binary", "logbroker-prestable_KLG")])
        Assertions.assertEquals(BigDecimal("0").multiply(BigDecimal(60L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey3[Pair("data_flow_binary", "logbroker-yc")])
        Assertions.assertEquals(BigDecimal("0").multiply(BigDecimal(70L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(),
            owningCostByResourceKey3[Pair("data_flow_binary", "logbroker-yc-preprod")])
    }

    @Test
    fun testMdb() {
        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val service = serviceDao.create(Service.withKey("dbaas")
            .withName("MDB DBAAS")
            .withAbcServiceId(MDB_ABC_SERVICE_ID)
            .withSettings(Service.Settings.builder()
                .accountActualValuesInQuotaDistribution(true)
                .requireZeroQuotaUsageForProjectDeletion(true)
                .build())
            .build())
        val slonnn = personDao.readPersonByLogin(SLONNN.login)
        serviceDao.attachAdmin(service, slonnn)
        serviceDao.attachAdmin(service, personDao.readPersonByLogin(ZOMB_MOBSEARCH))
        updateHierarchy()
        resourceDao.create(Resource.Builder("s3_hdd_x3", service).name("S3 HDD x3")
            .type(DiResourceType.STORAGE).build())
        resourceDao.create(Resource.Builder("cpu_all", service).name("CPU")
            .type(DiResourceType.PROCESSOR).build())
        resourceDao.create(Resource.Builder("ram_all", service).name("RAM")
            .type(DiResourceType.MEMORY).build())
        resourceDao.create(Resource.Builder("hdd_all", service).name("HDD")
            .type(DiResourceType.STORAGE).build())
        resourceDao.create(Resource.Builder("ssd_all", service).name("SSD")
            .type(DiResourceType.STORAGE).build())
        updateHierarchy()
        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)
                .changes("dbaas", "s3_hdd_x3", bigOrderTwo.id, emptySet(), DiAmount.of(1L, DiUnit.GIBIBYTE))
                .changes("dbaas", "cpu_all", bigOrderTwo.id, emptySet(), DiAmount.of(2L, DiUnit.CORES))
                .changes("dbaas", "ram_all", bigOrderTwo.id, emptySet(), DiAmount.of(3L, DiUnit.GIBIBYTE))
                .changes("dbaas", "hdd_all", bigOrderTwo.id, emptySet(), DiAmount.of(4L, DiUnit.GIBIBYTE))
                .changes("dbaas", "ssd_all", bigOrderTwo.id, emptySet(), DiAmount.of(5L, DiUnit.GIBIBYTE))
                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(5, changes.size)
        val owningCostByResourceKey = changes.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["s3_hdd_x3"])
        Assertions.assertEquals(BigDecimal("211.653").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["cpu_all"])
        Assertions.assertEquals(BigDecimal("32.656").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["ram_all"])
        Assertions.assertEquals(BigDecimal("0.23").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["hdd_all"])
        Assertions.assertEquals(BigDecimal("3.132").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey["ssd_all"])
        quotaChangeOwningCostRefreshManager.refresh()
        val fetchedRequest2 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest2)
        val changes2 = fetchedRequest2.changes
        Assertions.assertEquals(5, changes2.size)
        val owningCostByResourceKey2 = changes2.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["s3_hdd_x3"])
        Assertions.assertEquals(BigDecimal("211.653").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["cpu_all"])
        Assertions.assertEquals(BigDecimal("32.656").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["ram_all"])
        Assertions.assertEquals(BigDecimal("0.23").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["hdd_all"])
        Assertions.assertEquals(BigDecimal("3.132").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2["ssd_all"])
        dispenser().quotaChangeRequests()
            .byId(quotaRequests.first.id)
            .update(BodyUpdateBuilder()
                .changes("dbaas", "s3_hdd_x3", bigOrderTwo.id, emptySet(), DiAmount.of(10L, DiUnit.GIBIBYTE))
                .changes("dbaas", "cpu_all", bigOrderTwo.id, emptySet(), DiAmount.of(20L, DiUnit.CORES))
                .changes("dbaas", "ram_all", bigOrderTwo.id, emptySet(), DiAmount.of(30L, DiUnit.GIBIBYTE))
                .changes("dbaas", "hdd_all", bigOrderTwo.id, emptySet(), DiAmount.of(40L, DiUnit.GIBIBYTE))
                .changes("dbaas", "ssd_all", bigOrderTwo.id, emptySet(), DiAmount.of(50L, DiUnit.GIBIBYTE))
                .build())
            .performBy(AMOSOV_F)
        val fetchedRequest3 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest3)
        val changes3 = fetchedRequest3.changes
        Assertions.assertEquals(5, changes3.size)
        val owningCostByResourceKey3 = changes3.associateBy { it.resource.key }.mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.3141").multiply(BigDecimal(10L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["s3_hdd_x3"])
        Assertions.assertEquals(BigDecimal("211.653").multiply(BigDecimal(20L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["cpu_all"])
        Assertions.assertEquals(BigDecimal("32.656").multiply(BigDecimal(30L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["ram_all"])
        Assertions.assertEquals(BigDecimal("0.23").multiply(BigDecimal(40L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["hdd_all"])
        Assertions.assertEquals(BigDecimal("3.132").multiply(BigDecimal(50L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3["ssd_all"])
    }

    @Test
    fun testSandbox() {
        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val service = serviceDao.create(Service
            .withKey(SANDBOX)
            .withName("Sandbox")
            .withAbcServiceId(SANDBOX_ABC_SERVICE_ID)
            .withPriority(190)
            .build())
        updateHierarchy()
        val s3HddX2 = resourceDao.create(Resource.Builder("s3_hdd_x2", service).name("MDS S3 HDD x2")
            .type(DiResourceType.STORAGE).build())
        val cpuLinuxHdd = resourceDao.create(Resource.Builder("cpu_linux_hdd", service).name("CPU linux HDD")
            .type(DiResourceType.PROCESSOR).build())
        val cpuLinuxSsd = resourceDao.create(Resource.Builder("cpu_linux_ssd", service).name("CPU linux SSD")
            .type(DiResourceType.PROCESSOR).build())
        val cpuMacos = resourceDao.create(Resource.Builder("cpu_macos", service).name("CPU macos")
            .type(DiResourceType.PROCESSOR).build())
        val cpuWindows = resourceDao.create(Resource.Builder("cpu_windows", service).name("CPU windows")
            .type(DiResourceType.PROCESSOR).build())
        updateHierarchy()
        val location = segmentationDao.create(Segmentation.Builder("logbroker").name("logbroker")
            .description("logbroker").priority(1).build())
        updateHierarchy()
        val generic = segmentDao.create(Segment.Builder("sandbox_generic", location).priority(1.toShort())
            .name("sandbox_generic").description("sandbox_generic").build())
        val browser = segmentDao.create(Segment.Builder("sandbox_browser", location).priority(1.toShort())
            .name("sandbox_browser").description("sandbox_browser").build())
        val yabs = segmentDao.create(Segment.Builder("sandbox_yabs", location).priority(1.toShort())
            .name("sandbox_yabs").description("sandbox_yabs").build())
        val mobileMonorepo = segmentDao.create(Segment.Builder("sandbox_mobile_monorepo", location).priority(1.toShort())
            .name("sandbox_mobile_monorepo").description("sandbox_mobile_monorepo").build())
        updateHierarchy()
        resourceSegmentationDao.setSegmentations(cpuLinuxHdd, setOf(location))
        resourceSegmentationDao.setSegmentations(cpuLinuxSsd, setOf(location))
        resourceSegmentationDao.setSegmentations(cpuMacos, setOf(location))
        resourceSegmentationDao.setSegmentations(cpuWindows, setOf(location))
        updateHierarchy()
        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)
                .changes(SANDBOX, s3HddX2.publicKey, bigOrderTwo.id, emptySet(), DiAmount.of(1L, DiUnit.GIBIBYTE))
                .changes(SANDBOX, cpuLinuxHdd.publicKey, bigOrderTwo.id, setOf(generic.publicKey), DiAmount.of(2L, DiUnit.CORES))
                .changes(SANDBOX, cpuLinuxHdd.publicKey, bigOrderTwo.id, setOf(browser.publicKey), DiAmount.of(3L, DiUnit.CORES))
                .changes(SANDBOX, cpuLinuxHdd.publicKey, bigOrderTwo.id, setOf(yabs.publicKey), DiAmount.of(4L, DiUnit.CORES))
                .changes(SANDBOX, cpuLinuxSsd.publicKey, bigOrderTwo.id, setOf(generic.publicKey), DiAmount.of(5L, DiUnit.CORES))
                .changes(SANDBOX, cpuLinuxSsd.publicKey, bigOrderTwo.id, setOf(browser.publicKey), DiAmount.of(6L, DiUnit.CORES))
                .changes(SANDBOX, cpuLinuxSsd.publicKey, bigOrderTwo.id, setOf(yabs.publicKey), DiAmount.of(7L, DiUnit.CORES))
                .changes(SANDBOX, cpuMacos.publicKey, bigOrderTwo.id, setOf(generic.publicKey), DiAmount.of(8L, DiUnit.CORES))
                .changes(SANDBOX, cpuMacos.publicKey, bigOrderTwo.id, setOf(browser.publicKey), DiAmount.of(9L, DiUnit.CORES))
                .changes(SANDBOX, cpuMacos.publicKey, bigOrderTwo.id, setOf(mobileMonorepo.publicKey), DiAmount.of(10L, DiUnit.CORES))
                .changes(SANDBOX, cpuWindows.publicKey, bigOrderTwo.id, setOf(generic.publicKey), DiAmount.of(11L, DiUnit.CORES))
                .changes(SANDBOX, cpuWindows.publicKey, bigOrderTwo.id, setOf(browser.publicKey), DiAmount.of(12L, DiUnit.CORES))
                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(12, changes.size)
        val owningCostByResourceKey = changes.associateBy { Pair(it.resource.key, it.segmentKeys.firstOrNull()) }
            .mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("s3_hdd_x2", null)])
        Assertions.assertEquals(BigDecimal("350.52").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("cpu_linux_hdd", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("350.52").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("cpu_linux_hdd", "sandbox_browser")])
        Assertions.assertEquals(BigDecimal("350.52").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("cpu_linux_hdd", "sandbox_yabs")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("cpu_linux_ssd", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(6L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("cpu_linux_ssd", "sandbox_browser")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(7L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("cpu_linux_ssd", "sandbox_yabs")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(8L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("cpu_macos", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(9L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("cpu_macos", "sandbox_browser")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(10L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("cpu_macos", "sandbox_mobile_monorepo")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(11L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("cpu_windows", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(12L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("cpu_windows", "sandbox_browser")])
        quotaChangeOwningCostRefreshManager.refresh()
        val fetchedRequest2 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest2)
        val changes2 = fetchedRequest2.changes
        Assertions.assertEquals(12, changes2.size)
        val owningCostByResourceKey2 = changes2.associateBy { Pair(it.resource.key, it.segmentKeys.firstOrNull()) }
            .mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("s3_hdd_x2", null)])
        Assertions.assertEquals(BigDecimal("350.52").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("cpu_linux_hdd", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("350.52").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("cpu_linux_hdd", "sandbox_browser")])
        Assertions.assertEquals(BigDecimal("350.52").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("cpu_linux_hdd", "sandbox_yabs")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("cpu_linux_ssd", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(6L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("cpu_linux_ssd", "sandbox_browser")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(7L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("cpu_linux_ssd", "sandbox_yabs")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(8L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("cpu_macos", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(9L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("cpu_macos", "sandbox_browser")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(10L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("cpu_macos", "sandbox_mobile_monorepo")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(11L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("cpu_windows", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(12L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("cpu_windows", "sandbox_browser")])
        dispenser().quotaChangeRequests()
            .byId(quotaRequests.first.id)
            .update(BodyUpdateBuilder()
                .changes(SANDBOX, s3HddX2.publicKey, bigOrderTwo.id, emptySet(), DiAmount.of(10L, DiUnit.GIBIBYTE))
                .changes(SANDBOX, cpuLinuxHdd.publicKey, bigOrderTwo.id, setOf(generic.publicKey), DiAmount.of(20L, DiUnit.CORES))
                .changes(SANDBOX, cpuLinuxHdd.publicKey, bigOrderTwo.id, setOf(browser.publicKey), DiAmount.of(30L, DiUnit.CORES))
                .changes(SANDBOX, cpuLinuxHdd.publicKey, bigOrderTwo.id, setOf(yabs.publicKey), DiAmount.of(40L, DiUnit.CORES))
                .changes(SANDBOX, cpuLinuxSsd.publicKey, bigOrderTwo.id, setOf(generic.publicKey), DiAmount.of(50L, DiUnit.CORES))
                .changes(SANDBOX, cpuLinuxSsd.publicKey, bigOrderTwo.id, setOf(browser.publicKey), DiAmount.of(60L, DiUnit.CORES))
                .changes(SANDBOX, cpuLinuxSsd.publicKey, bigOrderTwo.id, setOf(yabs.publicKey), DiAmount.of(70L, DiUnit.CORES))
                .changes(SANDBOX, cpuMacos.publicKey, bigOrderTwo.id, setOf(generic.publicKey), DiAmount.of(80L, DiUnit.CORES))
                .changes(SANDBOX, cpuMacos.publicKey, bigOrderTwo.id, setOf(browser.publicKey), DiAmount.of(90L, DiUnit.CORES))
                .changes(SANDBOX, cpuMacos.publicKey, bigOrderTwo.id, setOf(mobileMonorepo.publicKey), DiAmount.of(100L, DiUnit.CORES))
                .changes(SANDBOX, cpuWindows.publicKey, bigOrderTwo.id, setOf(generic.publicKey), DiAmount.of(110L, DiUnit.CORES))
                .changes(SANDBOX, cpuWindows.publicKey, bigOrderTwo.id, setOf(browser.publicKey), DiAmount.of(120L, DiUnit.CORES))
                .build())
            .performBy(AMOSOV_F)
        val fetchedRequest3 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest3)
        val changes3 = fetchedRequest3.changes
        Assertions.assertEquals(12, changes3.size)
        val owningCostByResourceKey3 = changes3.associateBy { Pair(it.resource.key, it.segmentKeys.firstOrNull()) }
            .mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("0.20939976").multiply(BigDecimal(10L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("s3_hdd_x2", null)])
        Assertions.assertEquals(BigDecimal("350.52").multiply(BigDecimal(20L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("cpu_linux_hdd", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("350.52").multiply(BigDecimal(30L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("cpu_linux_hdd", "sandbox_browser")])
        Assertions.assertEquals(BigDecimal("350.52").multiply(BigDecimal(40L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("cpu_linux_hdd", "sandbox_yabs")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(50L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("cpu_linux_ssd", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(60L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("cpu_linux_ssd", "sandbox_browser")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(70L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("cpu_linux_ssd", "sandbox_yabs")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(80L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("cpu_macos", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(90L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("cpu_macos", "sandbox_browser")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(100L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("cpu_macos", "sandbox_mobile_monorepo")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(110L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("cpu_windows", "sandbox_generic")])
        Assertions.assertEquals(BigDecimal("404.51").multiply(BigDecimal(120L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("cpu_windows", "sandbox_browser")])
    }

    @Test
    fun ydbOwningCostFormulaTest() {
        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val service = serviceDao.create(Service
            .withKey(YDB)
            .withName("YDB")
            .withAbcServiceId(YDB_ABC_SERVICE_ID)
            .withPriority(190)
            .build())
        updateHierarchy()
        val cpu = resourceDao.create(Resource.Builder("total_cpu", service)
            .name("total_cpu")
            .type(DiResourceType.PROCESSOR)
            .build())
        val ram = resourceDao.create(Resource.Builder("total_ram", service)
            .name("total_ram")
            .type(DiResourceType.MEMORY)
            .build())
        val storageGroups = resourceDao.create(Resource.Builder("storage_groups", service)
            .name("storage_groups")
            .type(DiResourceType.ENUMERABLE)
            .build())
        val segmentation = segmentationDao.create(Segmentation.Builder(ydbSegmentationKey)
            .name(ydbSegmentationKey)
            .description(ydbSegmentationKey)
            .priority(1)
            .build())
        updateHierarchy()
        val ydbRu = segmentDao.create(Segment.Builder("ydb-ru", segmentation)
            .priority(1)
            .name("ydb-ru")
            .description("ydb-ru")
            .build())
        val ydbRuPrestable = segmentDao.create(Segment.Builder("ydb-ru-prestable", segmentation)
            .priority(1)
            .name("ydb-ru-prestable")
            .description("ydb-ru-prestable")
            .build())
        val ydbEu = segmentDao.create(Segment.Builder("ydb-eu", segmentation)
            .priority(1)
            .name("ydb-eu")
            .description("ydb-eu")
            .build())
        updateHierarchy()
        resourceSegmentationDao.setSegmentations(cpu, setOf(segmentation))
        resourceSegmentationDao.setSegmentations(ram, setOf(segmentation))
        resourceSegmentationDao.setSegmentations(storageGroups, setOf(segmentation))
        updateHierarchy()
        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)
                .changes(YDB, "total_cpu", bigOrderTwo.id, setOf(ydbRu.publicKey), DiAmount.of(1L, DiUnit.CORES))
                .changes(YDB, "total_cpu", bigOrderTwo.id, setOf(ydbRuPrestable.publicKey), DiAmount.of(2L, DiUnit.CORES))
                .changes(YDB, "total_cpu", bigOrderTwo.id, setOf(ydbEu.publicKey), DiAmount.of(3L, DiUnit.CORES))
                .changes(YDB, "total_ram", bigOrderTwo.id, setOf(ydbRu.publicKey), DiAmount.of(4L, DiUnit.GIBIBYTE))
                .changes(YDB, "total_ram", bigOrderTwo.id, setOf(ydbRuPrestable.publicKey), DiAmount.of(5L, DiUnit.GIBIBYTE))
                .changes(YDB, "total_ram", bigOrderTwo.id, setOf(ydbEu.publicKey), DiAmount.of(6L, DiUnit.GIBIBYTE))
                .changes(YDB, "storage_groups", bigOrderTwo.id, setOf(ydbRu.publicKey), DiAmount.of(7L, DiUnit.COUNT))
                .changes(YDB, "storage_groups", bigOrderTwo.id, setOf(ydbRuPrestable.publicKey), DiAmount.of(8L, DiUnit.COUNT))
                .changes(YDB, "storage_groups", bigOrderTwo.id, setOf(ydbEu.publicKey), DiAmount.of(9L, DiUnit.COUNT))
                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        updateHierarchy()
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(9, changes.size)
        val owningCostByResourceKey = changes.associateBy { Pair(it.resource.key, it.segmentKeys.firstOrNull()) }
            .mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("177.658").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("total_cpu", "ydb-ru")])
        Assertions.assertEquals(BigDecimal("177.658").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("total_cpu", "ydb-ru-prestable")])
        Assertions.assertEquals(BigDecimal("177.658").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("total_cpu", "ydb-eu")])
        Assertions.assertEquals(BigDecimal("29.693").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("total_ram", "ydb-ru")])
        Assertions.assertEquals(BigDecimal("29.693").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("total_ram", "ydb-ru-prestable")])
        Assertions.assertEquals(BigDecimal("29.693").multiply(BigDecimal(6L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("total_ram", "ydb-eu")])
        Assertions.assertEquals(BigDecimal("2161.08").multiply(BigDecimal(7L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("storage_groups", "ydb-ru")])
        Assertions.assertEquals(BigDecimal("2161.08").multiply(BigDecimal(8L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("storage_groups", "ydb-ru-prestable")])
        Assertions.assertEquals(BigDecimal("2161.08").multiply(BigDecimal(9L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey[Pair("storage_groups", "ydb-eu")])
        quotaChangeOwningCostRefreshManager.refresh()
        val fetchedRequest2 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest2)
        val changes2 = fetchedRequest2.changes
        Assertions.assertEquals(9, changes2.size)
        val owningCostByResourceKey2 = changes2.associateBy { Pair(it.resource.key, it.segmentKeys.firstOrNull()) }
            .mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("177.658").multiply(BigDecimal(1L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("total_cpu", "ydb-ru")])
        Assertions.assertEquals(BigDecimal("177.658").multiply(BigDecimal(2L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("total_cpu", "ydb-ru-prestable")])
        Assertions.assertEquals(BigDecimal("177.658").multiply(BigDecimal(3L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("total_cpu", "ydb-eu")])
        Assertions.assertEquals(BigDecimal("29.693").multiply(BigDecimal(4L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("total_ram", "ydb-ru")])
        Assertions.assertEquals(BigDecimal("29.693").multiply(BigDecimal(5L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("total_ram", "ydb-ru-prestable")])
        Assertions.assertEquals(BigDecimal("29.693").multiply(BigDecimal(6L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("total_ram", "ydb-eu")])
        Assertions.assertEquals(BigDecimal("2161.08").multiply(BigDecimal(7L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("storage_groups", "ydb-ru")])
        Assertions.assertEquals(BigDecimal("2161.08").multiply(BigDecimal(8L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("storage_groups", "ydb-ru-prestable")])
        Assertions.assertEquals(BigDecimal("2161.08").multiply(BigDecimal(9L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey2[Pair("storage_groups", "ydb-eu")])
        dispenser().quotaChangeRequests()
            .byId(quotaRequests.first.id)
            .update(BodyUpdateBuilder()
                .changes(YDB, "total_cpu", bigOrderTwo.id, setOf(ydbRu.publicKey), DiAmount.of(10L, DiUnit.CORES))
                .changes(YDB, "total_cpu", bigOrderTwo.id, setOf(ydbRuPrestable.publicKey), DiAmount.of(20L, DiUnit.CORES))
                .changes(YDB, "total_cpu", bigOrderTwo.id, setOf(ydbEu.publicKey), DiAmount.of(30L, DiUnit.CORES))
                .changes(YDB, "total_ram", bigOrderTwo.id, setOf(ydbRu.publicKey), DiAmount.of(40L, DiUnit.GIBIBYTE))
                .changes(YDB, "total_ram", bigOrderTwo.id, setOf(ydbRuPrestable.publicKey), DiAmount.of(50L, DiUnit.GIBIBYTE))
                .changes(YDB, "total_ram", bigOrderTwo.id, setOf(ydbEu.publicKey), DiAmount.of(60L, DiUnit.GIBIBYTE))
                .changes(YDB, "storage_groups", bigOrderTwo.id, setOf(ydbRu.publicKey), DiAmount.of(70L, DiUnit.COUNT))
                .changes(YDB, "storage_groups", bigOrderTwo.id, setOf(ydbRuPrestable.publicKey), DiAmount.of(80L, DiUnit.COUNT))
                .changes(YDB, "storage_groups", bigOrderTwo.id, setOf(ydbEu.publicKey), DiAmount.of(90L, DiUnit.COUNT))
                .build())
            .performBy(AMOSOV_F)
        val fetchedRequest3 = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest3)
        val changes3 = fetchedRequest3.changes
        Assertions.assertEquals(9, changes3.size)
        val owningCostByResourceKey3 = changes3.associateBy { Pair(it.resource.key, it.segmentKeys.firstOrNull()) }
            .mapValues { it.value.owningCost }
        Assertions.assertEquals(BigDecimal("177.658").multiply(BigDecimal(10L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("total_cpu", "ydb-ru")])
        Assertions.assertEquals(BigDecimal("177.658").multiply(BigDecimal(20L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("total_cpu", "ydb-ru-prestable")])
        Assertions.assertEquals(BigDecimal("177.658").multiply(BigDecimal(30L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("total_cpu", "ydb-eu")])
        Assertions.assertEquals(BigDecimal("29.693").multiply(BigDecimal(40L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("total_ram", "ydb-ru")])
        Assertions.assertEquals(BigDecimal("29.693").multiply(BigDecimal(50L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("total_ram", "ydb-ru-prestable")])
        Assertions.assertEquals(BigDecimal("29.693").multiply(BigDecimal(60L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("total_ram", "ydb-eu")])
        Assertions.assertEquals(BigDecimal("2161.08").multiply(BigDecimal(70L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("storage_groups", "ydb-ru")])
        Assertions.assertEquals(BigDecimal("2161.08").multiply(BigDecimal(80L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("storage_groups", "ydb-ru-prestable")])
        Assertions.assertEquals(BigDecimal("2161.08").multiply(BigDecimal(90L), ProviderOwningCostFormula.MATH_CONTEXT)
            .setScale(0, RoundingMode.HALF_UP).toString(), owningCostByResourceKey3[Pair("storage_groups", "ydb-eu")])
    }

    @Test
    fun ytOwningCostFormulaTest() {
        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val yt = serviceDao.create(Service
            .withKey(YT)
            .withName("YT")
            .withAbcServiceId(470)
            .withPriority(42)
            .build())

        updateHierarchy()

        val BURST_GUARANTEE_CPU = "burst_guarantee_cpu"
        val CPU = "cpu"
        val CPU_FLOW = "cpu_flow"
        val HDD = "hdd"
        val MEM_BURST = "mem_burst"
        val MEM_RELAXED = "mem_relaxed"
        val MEM_STRONG = "mem_strong"
        val SSD = "ssd"
        val CPU_DYNT = "cpu_dynt"
        val MEM_DYNT = "mem_dynt"

        val burstCpu = createResource(yt, BURST_GUARANTEE_CPU, DiResourceType.PROCESSOR)
        val cpu = createResource(yt, CPU, DiResourceType.PROCESSOR)
        val cpuFlow = createResource(yt, CPU_FLOW, DiResourceType.PROCESSOR)
        val storage = createResource(yt, HDD, DiResourceType.STORAGE)
        val memBurst = createResource(yt, MEM_BURST, DiResourceType.MEMORY)
        val memRelaxed = createResource(yt, MEM_RELAXED, DiResourceType.MEMORY)
        val memStrong = createResource(yt, MEM_STRONG, DiResourceType.MEMORY)
        val ssd = createResource(yt, SSD, DiResourceType.STORAGE)
        val cpuDynt = createResource(yt, CPU_DYNT, DiResourceType.PROCESSOR)
        val memDynt = createResource(yt, MEM_DYNT, DiResourceType.MEMORY)

        val ytSegmentation = segmentationDao.create(
            Segmentation.Builder(ytSegmentationKey).name(ytSegmentationKey)
                .description(ytSegmentationKey).priority(42).build()
        )
        updateHierarchy()

        val hahn = "hahn"
        val arnold = "arnold"
        val hume = "hume"
        val freud = "freud"
        val bohr = "bohr"
        val landau = "landau"
        val markov = "markov"
        val senecaSas = "seneca-sas"
        val senecaVla = "seneca-vla"
        val senecaKlg = "seneca-klg"
        val zeno = "zeno"
        val nash = "nash"
        val vanga = "vanga"
        val pythia = "pythia"
        val locke = "locke"

        val segments = listOf(hahn, arnold, hume, freud, bohr, landau, markov, senecaSas, senecaVla, senecaKlg, zeno,
            nash, vanga, pythia, locke)
        segments.forEach {
            createSegment(ytSegmentation, it)
        }

        val resources = listOf(burstCpu, cpu, cpuFlow, storage, memBurst, memRelaxed, memStrong, ssd, cpuDynt, memDynt)
        resources.forEach {
            resourceSegmentationDao.setSegmentations(it, setOf(ytSegmentation))
        }
        updateHierarchy()

        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)

                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(hahn), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(arnold), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(hume), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(freud), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(bohr), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(landau), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(markov), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(senecaSas), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(senecaVla), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(senecaKlg), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(zeno), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(nash), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(vanga), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(pythia), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, BURST_GUARANTEE_CPU, bigOrderTwo.id, setOf(locke), DiAmount.of(100L, DiUnit.CORES))


                .changes(YT, CPU, bigOrderTwo.id, setOf(hahn), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(arnold), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(hume), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(freud), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(bohr), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(landau), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(markov), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(senecaSas), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(senecaVla), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(senecaKlg), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(zeno), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(nash), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(vanga), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(pythia), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU, bigOrderTwo.id, setOf(locke), DiAmount.of(100L, DiUnit.CORES))

                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(hahn), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(arnold), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(hume), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(freud), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(bohr), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(landau), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(markov), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(senecaSas), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(senecaVla), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(senecaKlg), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(zeno), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(nash), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(vanga), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(pythia), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_FLOW, bigOrderTwo.id, setOf(locke), DiAmount.of(100L, DiUnit.CORES))

                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(hahn), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(arnold), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(hume), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(freud), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(bohr), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(landau), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(markov), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(senecaSas), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(senecaVla), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(senecaKlg), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(zeno), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(nash), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(vanga), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(pythia), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_STRONG, bigOrderTwo.id, setOf(locke), DiAmount.of(100L, DiUnit.GIBIBYTE))

                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(hahn), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(arnold), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(hume), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(freud), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(bohr), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(landau), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(markov), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(senecaSas), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(senecaVla), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(senecaKlg), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(zeno), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(nash), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(vanga), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(pythia), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_RELAXED, bigOrderTwo.id, setOf(locke), DiAmount.of(100L, DiUnit.GIBIBYTE))

                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(hahn), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(arnold), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(hume), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(freud), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(bohr), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(landau), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(markov), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(senecaSas), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(senecaVla), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(senecaKlg), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(zeno), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(nash), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(vanga), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(pythia), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_BURST, bigOrderTwo.id, setOf(locke), DiAmount.of(100L, DiUnit.GIBIBYTE))

                .changes(YT, HDD, bigOrderTwo.id, setOf(hahn), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(arnold), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(hume), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(freud), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(bohr), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(landau), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(markov), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(senecaSas), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(senecaVla), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(senecaKlg), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(zeno), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(nash), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(vanga), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(pythia), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, HDD, bigOrderTwo.id, setOf(locke), DiAmount.of(100L, DiUnit.GIBIBYTE))

                .changes(YT, SSD, bigOrderTwo.id, setOf(hahn), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(arnold), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(hume), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(freud), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(bohr), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(landau), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(markov), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(senecaSas), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(senecaVla), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(senecaKlg), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(zeno), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(nash), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(vanga), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(pythia), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, SSD, bigOrderTwo.id, setOf(locke), DiAmount.of(100L, DiUnit.GIBIBYTE))

                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(hahn), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(arnold), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(hume), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(freud), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(bohr), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(landau), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(markov), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(senecaSas), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(senecaVla), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(senecaKlg), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(zeno), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(nash), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(vanga), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(pythia), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(YT, MEM_DYNT, bigOrderTwo.id, setOf(locke), DiAmount.of(100L, DiUnit.GIBIBYTE))

                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(hahn), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(arnold), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(hume), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(freud), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(bohr), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(landau), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(markov), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(senecaSas), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(senecaVla), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(senecaKlg), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(zeno), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(nash), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(vanga), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(pythia), DiAmount.of(100L, DiUnit.CORES))
                .changes(YT, CPU_DYNT, bigOrderTwo.id, setOf(locke), DiAmount.of(100L, DiUnit.CORES))

                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        updateHierarchy()
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(resources.size * segments.size, changes.size)
        val changeByResourceSegment = changes.associateBy { Pair(it.resource.key, it.segmentKeys.first()) }

        val math = YtOwningCostFormula.MATH_CONTEXT
        val cpuUsagePrice = BigDecimal("53.297")
        val cpuStrongGuaranteePrice = BigDecimal("124.361")
        val cpuBurstGuaranteePrice = BigDecimal("24.872")
        val cpuRelaxedGuaranteePrice = BigDecimal("99.489")
        val hddPrice = BigDecimal("0.115")
        val ssdPrice = BigDecimal("1.566")
        val ramUsagePrice = BigDecimal("29.693")
        val ramBurstPrice = ramUsagePrice.multiply(YtOwningCostFormula.BURST_MEMORY_TARIFF_COEFFICIENT, math)
        val ramRelaxedPrice = ramUsagePrice.multiply(YtOwningCostFormula.RELAXED_MEMORY_TARIFF_COEFFICIENT, math)
        val dyntMemoryPrice = ramUsagePrice.multiply(YtOwningCostFormula.DYNT_MEMORY_TARIFF_COEFFICIENT, math)
        val dyntCpuPrice = cpuUsagePrice.add(cpuBurstGuaranteePrice, math).add(cpuRelaxedGuaranteePrice, math)

        val cpuStrongPrice = cpuUsagePrice.add(cpuStrongGuaranteePrice, math)
        Assertions.assertEquals(cpuStrongPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU, hahn)]?.owningCost)
        Assertions.assertEquals(cpuStrongPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU, arnold)]?.owningCost)
        Assertions.assertEquals(cpuStrongPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU, hume)]?.owningCost)
        Assertions.assertEquals(cpuStrongPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU, freud)]?.owningCost)
        Assertions.assertEquals(cpuStrongPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU, bohr)]?.owningCost)
        Assertions.assertEquals(cpuStrongPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU, landau)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU, markov)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU, senecaSas)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU, senecaVla)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU, senecaKlg)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU, zeno)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU, nash)]?.owningCost)
        Assertions.assertEquals(cpuStrongPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU, vanga)]?.owningCost)
        Assertions.assertEquals(cpuStrongPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU, pythia)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU, locke)]?.owningCost)

        val cpuBurstPrice = cpuUsagePrice.add(cpuBurstGuaranteePrice, math)
        Assertions.assertEquals(cpuBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, hahn)]?.owningCost)
        Assertions.assertEquals(cpuBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, arnold)]?.owningCost)
        Assertions.assertEquals(cpuBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, hume)]?.owningCost)
        Assertions.assertEquals(cpuBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, freud)]?.owningCost)
        Assertions.assertEquals(cpuBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, bohr)]?.owningCost)
        Assertions.assertEquals(cpuBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, landau)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, markov)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, senecaSas)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, senecaVla)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, senecaKlg)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, zeno)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, nash)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, vanga)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, pythia)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(BURST_GUARANTEE_CPU, locke)]?.owningCost)

        val cpuRelaxedPrice = cpuUsagePrice.add(cpuRelaxedGuaranteePrice, math)
        Assertions.assertEquals(cpuRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_FLOW, hahn)]?.owningCost)
        Assertions.assertEquals(cpuRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_FLOW, arnold)]?.owningCost)
        Assertions.assertEquals(cpuRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_FLOW, hume)]?.owningCost)
        Assertions.assertEquals(cpuRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_FLOW, freud)]?.owningCost)
        Assertions.assertEquals(cpuRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_FLOW, bohr)]?.owningCost)
        Assertions.assertEquals(cpuRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_FLOW, landau)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU_FLOW, markov)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU_FLOW, senecaSas)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU_FLOW, senecaVla)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU_FLOW, senecaKlg)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU_FLOW, zeno)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU_FLOW, nash)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU_FLOW, vanga)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU_FLOW, pythia)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU_FLOW, locke)]?.owningCost)

        Assertions.assertEquals(ramUsagePrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_STRONG, hahn)]?.owningCost)
        Assertions.assertEquals(ramUsagePrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_STRONG, arnold)]?.owningCost)
        Assertions.assertEquals(ramUsagePrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_STRONG, hume)]?.owningCost)
        Assertions.assertEquals(ramUsagePrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_STRONG, freud)]?.owningCost)
        Assertions.assertEquals(ramUsagePrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_STRONG, bohr)]?.owningCost)
        Assertions.assertEquals(ramUsagePrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_STRONG, landau)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_STRONG, markov)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_STRONG, senecaSas)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_STRONG, senecaVla)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_STRONG, senecaKlg)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_STRONG, zeno)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_STRONG, nash)]?.owningCost)
        Assertions.assertEquals(ramUsagePrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_STRONG, vanga)]?.owningCost)
        Assertions.assertEquals(ramUsagePrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_STRONG, pythia)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_STRONG, locke)]?.owningCost)

        Assertions.assertEquals(ramBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_BURST, hahn)]?.owningCost)
        Assertions.assertEquals(ramBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_BURST, arnold)]?.owningCost)
        Assertions.assertEquals(ramBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_BURST, hume)]?.owningCost)
        Assertions.assertEquals(ramBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_BURST, freud)]?.owningCost)
        Assertions.assertEquals(ramBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_BURST, bohr)]?.owningCost)
        Assertions.assertEquals(ramBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_BURST, landau)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_BURST, markov)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_BURST, senecaSas)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_BURST, senecaVla)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_BURST, senecaKlg)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_BURST, zeno)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_BURST, nash)]?.owningCost)
        Assertions.assertEquals(ramBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_BURST, vanga)]?.owningCost)
        Assertions.assertEquals(ramBurstPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_BURST, pythia)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_BURST, locke)]?.owningCost)

        Assertions.assertEquals(ramRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_RELAXED, hahn)]?.owningCost)
        Assertions.assertEquals(ramRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_RELAXED, arnold)]?.owningCost)
        Assertions.assertEquals(ramRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_RELAXED, hume)]?.owningCost)
        Assertions.assertEquals(ramRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_RELAXED, freud)]?.owningCost)
        Assertions.assertEquals(ramRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_RELAXED, bohr)]?.owningCost)
        Assertions.assertEquals(ramRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_RELAXED, landau)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_RELAXED, markov)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_RELAXED, senecaSas)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_RELAXED, senecaVla)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_RELAXED, senecaKlg)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_RELAXED, zeno)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_RELAXED, nash)]?.owningCost)
        Assertions.assertEquals(ramRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_RELAXED, vanga)]?.owningCost)
        Assertions.assertEquals(ramRelaxedPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_RELAXED, pythia)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_RELAXED, locke)]?.owningCost)

        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, hahn)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, arnold)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, hume)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, freud)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, bohr)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, landau)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, markov)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, senecaSas)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, senecaVla)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, senecaKlg)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, zeno)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, nash)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, vanga)]?.owningCost)
        Assertions.assertEquals(hddPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(HDD, pythia)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(HDD, locke)]?.owningCost)

        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, hahn)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, arnold)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, hume)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, freud)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, bohr)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, landau)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, markov)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, senecaSas)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, senecaVla)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, senecaKlg)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, zeno)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, nash)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, vanga)]?.owningCost)
        Assertions.assertEquals(ssdPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(SSD, pythia)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(SSD, locke)]?.owningCost)

        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, hahn)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, arnold)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, hume)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, freud)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, bohr)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, landau)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, markov)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, senecaSas)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, senecaVla)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, senecaKlg)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, zeno)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, nash)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, vanga)]?.owningCost)
        Assertions.assertEquals(dyntCpuPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(CPU_DYNT, pythia)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(CPU_DYNT, locke)]?.owningCost)

        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, hahn)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, arnold)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, hume)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, freud)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, bohr)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, landau)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, markov)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, senecaSas)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, senecaVla)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, senecaKlg)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, zeno)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, nash)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, vanga)]?.owningCost)
        Assertions.assertEquals(dyntMemoryPrice.multiply(BigDecimal(100), math).toView(), changeByResourceSegment[Pair(MEM_DYNT, pythia)]?.owningCost)
        Assertions.assertEquals("0", changeByResourceSegment[Pair(MEM_DYNT, locke)]?.owningCost)
    }

    @Test
    fun testSaas() {
        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val saas = serviceDao.create(Service.withKey(SAAS)
            .withName(SAAS)
            .withAbcServiceId(470)
            .withPriority(42)
            .build())

        updateHierarchy()

        val cpu = createResource(saas, "cpu", DiResourceType.PROCESSOR)
        val ram = createResource(saas, "ram", DiResourceType.MEMORY)
        val ssd = createResource(saas, "ssd", DiResourceType.STORAGE)
        val ioNet = createResource(saas, "io_net", DiResourceType.BINARY_TRAFFIC)

        val locationSegmentation = Hierarchy.get().segmentationReader.read(Segmentation.Key(locationSegmentationKey))
        updateHierarchy()

        createSegment(locationSegmentation, "SAS")
        createSegment(locationSegmentation, "VLA")
        createSegment(locationSegmentation, "KLG")

        val resources = listOf(cpu, ram, ssd, ioNet)
        resources.forEach {
            resourceSegmentationDao.setSegmentations(it, setOf(locationSegmentation))
        }
        updateHierarchy()

        prepareCampaignResources()
        val quotaRequests = dispenser().quotaChangeRequests()
            .create(requestBodyBuilderWithDefaultFields()
                .projectKey(TEST_PROJECT_KEY)

                .changes(SAAS, cpu.publicKey, bigOrderTwo.id, setOf("SAS"), DiAmount.of(100L, DiUnit.CORES))
                .changes(SAAS, ram.publicKey, bigOrderTwo.id, setOf("SAS"), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(SAAS, ssd.publicKey, bigOrderTwo.id, setOf("SAS"), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(SAAS, ioNet.publicKey, bigOrderTwo.id, setOf("SAS"), DiAmount.of(100L, DiUnit.MIBPS))

                .changes(SAAS, cpu.publicKey, bigOrderTwo.id, setOf("VLA"), DiAmount.of(100L, DiUnit.CORES))
                .changes(SAAS, ram.publicKey, bigOrderTwo.id, setOf("VLA"), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(SAAS, ssd.publicKey, bigOrderTwo.id, setOf("VLA"), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(SAAS, ioNet.publicKey, bigOrderTwo.id, setOf("VLA"), DiAmount.of(100L, DiUnit.MIBPS))

                .changes(SAAS, cpu.publicKey, bigOrderTwo.id, setOf("KLG"), DiAmount.of(100L, DiUnit.CORES))
                .changes(SAAS, ram.publicKey, bigOrderTwo.id, setOf("KLG"), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(SAAS, ssd.publicKey, bigOrderTwo.id, setOf("KLG"), DiAmount.of(100L, DiUnit.GIBIBYTE))
                .changes(SAAS, ioNet.publicKey, bigOrderTwo.id, setOf("KLG"), DiAmount.of(100L, DiUnit.MIBPS))

                .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                .build(), campaign.id)
            .performBy(AMOSOV_F)
        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changes = fetchedRequest.changes
        Assertions.assertEquals(12, changes.size)
        val owningCostByResourceKey = changes.associateBy { Pair(it.resource.key, it.segmentKeys.first()) }
            .mapValues { it.value.owningCost }

        val cpuSkuPrice = BigDecimal("162.81")
        val hddSkuPrice = BigDecimal("0.115")
        val ramSkuPrice = BigDecimal("25.12")
        val ssdSkuPrice = BigDecimal("1.566")

        val math = ProviderOwningCostFormula.MATH_CONTEXT
        val perCpuPrice = cpuSkuPrice.add(hddSkuPrice.multiply(BigDecimal(28L), math), math)
        val perRamPrice = ramSkuPrice.add(hddSkuPrice.multiply(BigDecimal(7), math), math)
        val perSsdPrice = ssdSkuPrice.add(hddSkuPrice.multiply(BigDecimal("0.5", math), math), math)

        Assertions.assertEquals(perCpuPrice.multiply(BigDecimal(100), math).toView(), owningCostByResourceKey[Pair(cpu.publicKey, "SAS")])
        Assertions.assertEquals(perRamPrice.multiply(BigDecimal(100), math).toView(), owningCostByResourceKey[Pair(ram.publicKey, "SAS")])
        Assertions.assertEquals(perSsdPrice.multiply(BigDecimal(100), math).toView(), owningCostByResourceKey[Pair(ssd.publicKey, "SAS")])
        Assertions.assertEquals("0", owningCostByResourceKey[Pair(ioNet.publicKey, "SAS")])

        Assertions.assertEquals(perCpuPrice.multiply(BigDecimal(100), math).toView(), owningCostByResourceKey[Pair(cpu.publicKey, "VLA")])
        Assertions.assertEquals(perRamPrice.multiply(BigDecimal(100), math).toView(), owningCostByResourceKey[Pair(ram.publicKey, "VLA")])
        Assertions.assertEquals(perSsdPrice.multiply(BigDecimal(100), math).toView(), owningCostByResourceKey[Pair(ssd.publicKey, "VLA")])
        Assertions.assertEquals("0", owningCostByResourceKey[Pair(ioNet.publicKey, "VLA")])

        Assertions.assertEquals(perCpuPrice.multiply(BigDecimal(100), math).toView(), owningCostByResourceKey[Pair(cpu.publicKey, "KLG")])
        Assertions.assertEquals(perRamPrice.multiply(BigDecimal(100), math).toView(), owningCostByResourceKey[Pair(ram.publicKey, "KLG")])
        Assertions.assertEquals(perSsdPrice.multiply(BigDecimal(100), math).toView(), owningCostByResourceKey[Pair(ssd.publicKey, "KLG")])
        Assertions.assertEquals("0", owningCostByResourceKey[Pair(ioNet.publicKey, "KLG")])
    }

    @Test
    fun testYP() {
        val campaign = prepareCampaign()
        createProject(TEST_PROJECT_KEY, YANDEX, AMOSOV_F.login)
        val yp = Hierarchy.get().serviceReader.read(YP)

        val ssdKey = "ssd_segmented"
        createResource(yp, ssdKey, DiResourceType.STORAGE)
        val hddKey = "hdd_segmented"
        createResource(yp, hddKey, DiResourceType.STORAGE)
        val cpuKey = "cpu_segmented"
        createResource(yp, cpuKey, DiResourceType.PROCESSOR)
        val ramKey = "ram_segmented"
        createResource(yp, ramKey, DiResourceType.MEMORY)
        val ioHddKey = "io_hdd"
        createResource(yp, ioHddKey, DiResourceType.BINARY_TRAFFIC)
        val ioSsdKey = "io_ssd"
        createResource(yp, ioSsdKey, DiResourceType.BINARY_TRAFFIC)
        val ioNetKey = "io_net"
        createResource(yp, ioNetKey, DiResourceType.BINARY_TRAFFIC)

        updateHierarchy()
        prepareCampaignResources()

        val quotaRequests = dispenser().quotaChangeRequests()
            .create(
                requestBodyBuilderWithDefaultFields()
                    .projectKey(TEST_PROJECT_KEY)
                    .changes(YP, cpuKey, bigOrderTwo.id, emptySet(), DiAmount.of(1000L, DiUnit.CORES))
                    .changes(YP, ssdKey, bigOrderTwo.id, emptySet(), DiAmount.of(10000L, DiUnit.GIBIBYTE))
                    .changes(YP, hddKey, bigOrderTwo.id, emptySet(), DiAmount.of(10000L, DiUnit.GIBIBYTE))
                    .changes(YP, ramKey, bigOrderTwo.id, emptySet(), DiAmount.of(1000L, DiUnit.GIBIBYTE))
                    .changes(YP, ioHddKey, bigOrderTwo.id, emptySet(), DiAmount.of(1L, DiUnit.MIBPS))
                    .changes(YP, ioSsdKey, bigOrderTwo.id, emptySet(), DiAmount.of(1L, DiUnit.MIBPS))
                    .changes(YP, ioNetKey, bigOrderTwo.id, emptySet(), DiAmount.of(1L, DiUnit.MIBPS))
                    .type(DiQuotaChangeRequest.Type.RESOURCE_PREORDER)
                    .resourcePreorderReasonType(DiResourcePreorderReasonType.GROWTH)
                    .build(),
                campaign.id
            )
            .performBy(AMOSOV_F)

        updateHierarchy()

        val fetchedRequest = createAuthorizedLocalClient(AMOSOV_F)
            .path("/v1/quota-requests/" + quotaRequests.first.id)
            .get(DiQuotaChangeRequest::class.java)
        Assertions.assertNotNull(fetchedRequest)
        val changeByResourceKey = fetchedRequest.changes.associateBy { it.resource.key }

        Assertions.assertEquals("162810", changeByResourceKey[cpuKey]?.owningCost)
        Assertions.assertEquals("15660", changeByResourceKey[ssdKey]?.owningCost)
        Assertions.assertEquals("1150", changeByResourceKey[hddKey]?.owningCost)
        Assertions.assertEquals("25120", changeByResourceKey[ramKey]?.owningCost)
        Assertions.assertEquals("0", changeByResourceKey[ioHddKey]?.owningCost)
        Assertions.assertEquals("0", changeByResourceKey[ioSsdKey]?.owningCost)
        Assertions.assertEquals("0", changeByResourceKey[ioNetKey]?.owningCost)
    }

    private fun BigDecimal.toView() = this.setScale(0, RoundingMode.HALF_UP).toString()

    private fun createResource(
        service: Service,
        key: String,
        type: DiResourceType
    ): Resource {
        return resourceDao.create(Resource.Builder(key, service).name(key).type(type).build())
    }

    private fun createSegment(
        segmentation: Segmentation,
        key: String
    ): Segment {
        return segmentDao.create(Segment.Builder(key, segmentation)
            .priority(1)
            .name(key)
            .description(key)
            .build())
    }

    private fun prepareMdsService(): Service {
        val service = serviceDao.create(Service.withKey("mds_new")
            .withName("MDS new")
            .withAbcServiceId(895)
            .withSettings(Service.Settings.builder()
                .accountActualValuesInQuotaDistribution(true)
                .requireZeroQuotaUsageForProjectDeletion(true)
                .build())
            .build())
        serviceDao.attachAdmin(service, personDao.readPersonByLogin(SLONNN.login))
        serviceDao.attachAdmin(service, personDao.readPersonByLogin(ZOMB_MOBSEARCH))
        return service
    }

    private fun prepareAvatarsService(): Service {
        val service = serviceDao.create(Service.withKey("avatars")
            .withName("Avatars")
            .withAbcServiceId(1075)
            .withSettings(Service.Settings.builder()
                .accountActualValuesInQuotaDistribution(true)
                .requireZeroQuotaUsageForProjectDeletion(true)
                .build())
            .build())
        serviceDao.attachAdmin(service, personDao.readPersonByLogin(SLONNN.login))
        serviceDao.attachAdmin(service, personDao.readPersonByLogin(ZOMB_MOBSEARCH))
        return service
    }

    private fun prepareS3Service(): Service {
        val service = serviceDao.create(Service.withKey("s3")
            .withName("S3")
            .withAbcServiceId(5535)
            .withSettings(Service.Settings.builder()
                .accountActualValuesInQuotaDistribution(true)
                .requireZeroQuotaUsageForProjectDeletion(true)
                .build())
            .build())
        serviceDao.attachAdmin(service, personDao.readPersonByLogin(SLONNN.login))
        serviceDao.attachAdmin(service, personDao.readPersonByLogin(ZOMB_MOBSEARCH))
        return service
    }

    private fun prepareSolomonService(): Service {
        val service = serviceDao.create(Service.withKey("solomon")
            .withName("Solomon")
            .withAbcServiceId(700)
            .withSettings(Service.Settings.builder()
                .accountActualValuesInQuotaDistribution(true)
                .requireZeroQuotaUsageForProjectDeletion(true)
                .build())
            .build())
        serviceDao.attachAdmin(service, personDao.readPersonByLogin(SLONNN.login))
        serviceDao.attachAdmin(service, personDao.readPersonByLogin(ZOMB_MOBSEARCH))
        return service
    }

    private fun prepareCampaign(): Campaign {
        val date = LocalDate.of(2022, Month.AUGUST, 1)
        val campaign = campaignDao.insertForTest(QuotaChangeRequestValidationTest.defaultCampaignBuilder(bigOrderTwo)
            .setKey("aug2022aggregated")
            .setName("aug2022aggregated")
            .setId(176L)
            .setStartDate(date)
            .setBigOrders(listOf(Campaign.BigOrder(bigOrderTwo.id, date)))
            .setType(Campaign.Type.AGGREGATED)
            .build())
        botCampaignGroupDao.create(
            QuotaChangeRequestValidationTest.defaultCampaignGroupBuilder()
                .addBigOrder(Objects.requireNonNull(bigOrderManager.getById(bigOrderTwo.id)))
                .addCampaign(campaignDao.readForBot(setOf(campaign.id)).values.iterator().next())
                .build()
        )
        return campaign
    }

}
