package ru.yandex.qe.dispenser;

import javax.annotation.Resource;

import org.apache.http.impl.client.CloseableHttpClient;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.solomon.SolomonHolder;
import ru.yandex.qe.http.HttpClientBuilder;
import ru.yandex.qe.http.retries.RetryStrategy;
import ru.yandex.qe.http.telemetry.MetricsInstrumentedHttpClientBuilder;

@Component
public class TvmPassportHttpClientFactoryBean implements FactoryBean<CloseableHttpClient> {

    @NotNull
    private final TvmServiceTicketAspectFactory tvmServiceTicketAspectFactory;

    @Resource(name = "passportRetryStrategy")
    private RetryStrategy retryStrategy;

    @Value("${passport.telemetry.enabled:true}")
    private boolean telemetryEnabled;

    @Value("${plugin.passport.http.log.enable:false}")
    private boolean logEnabled;

    @Value("${plugin.passport.connect.timeout:1000}")
    private int connectTimeout;

    @Value("${plugin.passport.socket.timeout:5000}")
    private int socketTimeout;

    @Autowired
    private InstrumentedQloudTvmService qloudTvmService;
    @Value("${qloud.tvm.blackbox.tvm_service_id:223}")
    private String blackboxTvmClientId;

    @Autowired
    private SolomonHolder solomonHolder;

    public TvmPassportHttpClientFactoryBean() {
        tvmServiceTicketAspectFactory = new TvmServiceTicketAspectFactory(() -> qloudTvmService.getTicket(blackboxTvmClientId));
    }

    @Override
    public CloseableHttpClient getObject() throws Exception {

        if (telemetryEnabled) {
            return new MetricsInstrumentedHttpClientBuilder("tvmPassport",
                    solomonHolder.getRootRegistry())
                    .setRetryStrategy(retryStrategy)
                    .setLogEnable(logEnabled)
                    .setConnectTimeout(connectTimeout)
                    .setSocketTimeout(socketTimeout)
                    .addAspectFactory(tvmServiceTicketAspectFactory)
                    .build();
        } else {
            return new HttpClientBuilder()
                    .setRetryStrategy(retryStrategy)
                    .setLogEnable(logEnabled)
                    .setConnectTimeout(connectTimeout)
                    .setSocketTimeout(socketTimeout)
                    .addAspectFactory(tvmServiceTicketAspectFactory)
                    .build();
        }
    }

    @Override
    public Class<?> getObjectType() {
        return CloseableHttpClient.class;
    }

    @Override
    public boolean isSingleton() {
        return true;
    }
}

