package ru.yandex.qe.dispenser.datasources;

import java.beans.PropertyVetoException;

import javax.inject.Named;
import javax.sql.DataSource;

import com.mchange.v2.c3p0.ComboPooledDataSource;
import org.jetbrains.annotations.NotNull;
import org.postgresql.Driver;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.ImportResource;

import ru.yandex.qe.dispenser.solomon.SolomonHolder;

@Configuration
@ImportResource("classpath:spring/solomon-ctx.xml")
public class DataSourcesConfiguration {

    @Bean(name = "dataSource")
    public DataSource dataSource(@Named("dataSourceSensors") @NotNull final DataSourceSensors dataSourceSensors,
                                 @Named("pooledDataSource") @NotNull final ComboPooledDataSource dataSource) {
        return new ProxyDataSource(dataSource, dataSourceSensors);
    }

    @Bean(name = "readOnlyDataSource")
    public DataSource readOnlyDataSource(@Named("readOnlyDataSourceSensors") @NotNull final DataSourceSensors dataSourceSensors,
                                         @Named("readOnlyPooledDataSource") @NotNull final ComboPooledDataSource dataSource) {
        return new ProxyDataSource(dataSource, dataSourceSensors);
    }

    @Bean(name = "dataSourceSensors")
    public DataSourceSensors dataSourceSensors(@NotNull final SolomonHolder solomonHolder) {
        return new DataSourceSensors(solomonHolder.getRootRegistry(), "rw");
    }

    @Bean(name = "readOnlyDataSourceSensors")
    public DataSourceSensors readOnlyDataSourceSensors(@NotNull final SolomonHolder solomonHolder) {
        return new DataSourceSensors(solomonHolder.getRootRegistry(), "ro");
    }

    @Bean(name = "connectionPoolSensors")
    public ConnectionPoolSensors connectionPoolSensors(@NotNull final SolomonHolder solomonHolder,
                                                       @Named("pooledDataSource") @NotNull final ComboPooledDataSource dataSource) {
        return new ConnectionPoolSensors(solomonHolder.getRootRegistry(), "rw", dataSource);
    }

    @Bean(name = "readOnlyConnectionPoolSensors")
    public ConnectionPoolSensors readOnlyConnectionPoolSensors(@NotNull final SolomonHolder solomonHolder,
                                                       @Named("readOnlyPooledDataSource") @NotNull final ComboPooledDataSource dataSource) {
        return new ConnectionPoolSensors(solomonHolder.getRootRegistry(), "ro", dataSource);
    }

    @Bean(name = "pooledDataSource", destroyMethod = "close")
    public ComboPooledDataSource pooledDataSource(@Value("${database.jdbcUrl}") final String jdbcUrl,
                                 @Value("${database.user}") final String user,
                                 @Value("${database.password}") final String password,
                                 @Value("${dataSource.rw.acquireIncrement}") final int acquireIncrement,
                                 @Value("${dataSource.rw.minPoolSize}") final int minPoolSize,
                                 @Value("${dataSource.rw.maxPoolSize}") final int maxPoolSize,
                                 @Value("${dataSource.rw.initialPoolSize}") final int initialPoolSize,
                                 @Value("${dataSource.rw.numHelperThreads}") final int numHelperThreads,
                                 @Value("${dataSource.rw.checkoutTimeout}") final int checkoutTimeout,
                                 @Value("${dataSource.rw.idleConnectionTestPeriod.seconds}") final int idleConnectionTestPeriod,
                                 @Value("${dataSource.rw.testConnectionOnCheckIn}") final boolean testConnectionOnCheckIn,
                                 @Value("${dataSource.rw.testConnectionOnCheckOut}") final boolean testConnectionOnCheckOut) {
        final ComboPooledDataSource dataSource = new ComboPooledDataSource();
        try {
            dataSource.setDriverClass(Driver.class.getName());
        } catch (PropertyVetoException e) {
            throw new RuntimeException("Invalid db driver class", e);
        }
        dataSource.setJdbcUrl(jdbcUrl);
        dataSource.setUser(user);
        dataSource.setPassword(password);
        dataSource.setAcquireIncrement(acquireIncrement);
        dataSource.setMinPoolSize(minPoolSize);
        dataSource.setMaxPoolSize(maxPoolSize);
        dataSource.setInitialPoolSize(initialPoolSize);
        dataSource.setNumHelperThreads(numHelperThreads);
        dataSource.setCheckoutTimeout(checkoutTimeout);
        dataSource.setTestConnectionOnCheckin(testConnectionOnCheckIn);
        dataSource.setTestConnectionOnCheckout(testConnectionOnCheckOut);
        dataSource.setIdleConnectionTestPeriod(idleConnectionTestPeriod);
        return dataSource;
    }

    @Bean(name = "readOnlyPooledDataSource", destroyMethod = "close")
    public ComboPooledDataSource readOnlyPooledDataSource(@Value("${database.readOnlyJdbcUrl}") final String jdbcUrl,
                                         @Value("${database.user}") final String user,
                                         @Value("${database.password}") final String password,
                                         @Value("${dataSource.ro.acquireIncrement}") final int acquireIncrement,
                                         @Value("${dataSource.ro.minPoolSize}") final int minPoolSize,
                                         @Value("${dataSource.ro.maxPoolSize}") final int maxPoolSize,
                                         @Value("${dataSource.ro.initialPoolSize}") final int initialPoolSize,
                                         @Value("${dataSource.ro.numHelperThreads}") final int numHelperThreads,
                                         @Value("${dataSource.ro.checkoutTimeout}") final int checkoutTimeout,
                                         @Value("${dataSource.ro.idleConnectionTestPeriod.seconds}") final int idleConnectionTestPeriod,
                                         @Value("${dataSource.ro.testConnectionOnCheckIn}") final boolean testConnectionOnCheckIn,
                                         @Value("${dataSource.ro.testConnectionOnCheckOut}") final boolean testConnectionOnCheckOut) {
        final ComboPooledDataSource dataSource = new ComboPooledDataSource();
        try {
            dataSource.setDriverClass(Driver.class.getName());
        } catch (PropertyVetoException e) {
            throw new RuntimeException("Invalid db driver class", e);
        }
        dataSource.setJdbcUrl(jdbcUrl);
        dataSource.setUser(user);
        dataSource.setPassword(password);
        dataSource.setAcquireIncrement(acquireIncrement);
        dataSource.setMinPoolSize(minPoolSize);
        dataSource.setMaxPoolSize(maxPoolSize);
        dataSource.setInitialPoolSize(initialPoolSize);
        dataSource.setNumHelperThreads(numHelperThreads);
        dataSource.setCheckoutTimeout(checkoutTimeout);
        dataSource.setTestConnectionOnCheckin(testConnectionOnCheckIn);
        dataSource.setTestConnectionOnCheckout(testConnectionOnCheckOut);
        dataSource.setIdleConnectionTestPeriod(idleConnectionTestPeriod);
        return dataSource;
    }

}
