package ru.yandex.qe.dispenser.passport.blackbox;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.MoreObjects;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.Cacheable;

import ru.yandex.qe.dispenser.passport.PassportException;
import ru.yandex.qe.dispenser.passport.PassportService;
import ru.yandex.qe.dispenser.passport.Session;
import ru.yandex.qe.dispenser.passport.SslSession;


/**
 * User: alms
 * Date: 27.07.2009
 * Time: 15:04:06
 */
public class BlackBoxPassportService implements PassportService {

    private static final Logger LOG = LoggerFactory.getLogger(BlackBoxPassportService.class);

    private static final String SESSION_ID_METHOD = "sessionid";
    private static final String OAUTH_METHOD = "oauth";
    private static final String LOGIN_FIELD = "accounts.login.uid";

    private final HttpClient httpClient;
    private final String serviceRootUrl;
    private final String blackboxDomain;

    public BlackBoxPassportService(final HttpClient httpClient, final String serviceRootUrl, final String blackboxDomain) {
        this.httpClient = httpClient;
        this.serviceRootUrl = serviceRootUrl;
        this.blackboxDomain = blackboxDomain;
    }

    public BlackBoxPassportService() {
        this.httpClient = null;
        this.serviceRootUrl = null;
        this.blackboxDomain = null;
    }

    @Nonnull
    @Override
    @Cacheable(value = "blackboxSessionCache")
    public Session getSession(final String sessionId, final String userIp) {
        try {
            final URI uri = buildURI(sessionId, userIp);
            return httpClient.execute(new HttpGet(uri), new ResponseHandler<Session>() {
                @Override
                public Session handleResponse(final HttpResponse response) throws IOException {
                    final String blackBoxRawResponse = IOUtils.toString(response.getEntity().getContent(), StandardCharsets.UTF_8);
                    try {
                        final Element blackBoxXmlResponse = parseXml(blackBoxRawResponse);
                        return parseAnswer(blackBoxXmlResponse, sessionId, userIp);
                    } catch (JDOMException | IOException ex) {
                        if (blackBoxRawResponse != null) {
                            throw new PassportException(String.format("passport response parsing failed: %s", blackBoxRawResponse), ex);
                        } else {
                            throw new PassportException("Empty passport response", ex);
                        }
                    }
                }
            });
        } catch (IOException ex) {
            throw new PassportException("Blackbox httpGet failed", ex);
        } catch (URISyntaxException ex) {
            throw new RuntimeException(ex);
        }
    }

    @Nonnull
    @Override
    @Cacheable(value = "blackboxSessionCache")
    public SslSession getSslSession(final String sessionId, final String sslSessionId, final String userIp) {
        try {
            final URI uri = buildSslURI(sessionId, sslSessionId, userIp);
            return httpClient.execute(new HttpGet(uri), new ResponseHandler<SslSession>() {
                @Override
                public SslSession handleResponse(final HttpResponse response) throws IOException {
                    final String blackBoxRawResponse = IOUtils.toString(response.getEntity().getContent(), StandardCharsets.UTF_8);
                    try {
                        final Element blackBoxXmlResponse = parseXml(blackBoxRawResponse);
                        return parseSslAnswer(blackBoxXmlResponse, sessionId, userIp);
                    } catch (JDOMException | IOException ex) {
                        if (blackBoxRawResponse != null) {
                            throw new PassportException(String.format("passport response parsing failed: %s", blackBoxRawResponse), ex);
                        } else {
                            throw new PassportException("Empty passport response", ex);
                        }
                    }
                }
            });
        } catch (IOException ex) {
            throw new PassportException("Blackbox httpGet failed", ex);
        } catch (URISyntaxException ex) {
            throw new RuntimeException(ex);
        }
    }

    @Nonnull
    @Override
    @Cacheable(value = "blackboxOauthSessionCache")
    public Session getOauthSession(final String authorization, final String userIp) {
        try {
            final URI uri = buildOauthURI(userIp);
            final HttpGet request = new HttpGet(uri);
            request.setHeader("Authorization", authorization);
            LOG.info("Executing: {}", request);
            return httpClient.execute(request, new ResponseHandler<Session>() {
                @Override
                public Session handleResponse(final HttpResponse response) throws IOException {
                    final String blackBoxRawResponse = IOUtils.toString(response.getEntity().getContent(), StandardCharsets.UTF_8);
                    try {
                        final Element blackBoxXmlResponse = parseXml(blackBoxRawResponse);
                        return parseAnswer(blackBoxXmlResponse, authorization, userIp);
                    } catch (JDOMException | IOException ex) {
                        if (blackBoxRawResponse != null) {
                            throw new PassportException(String.format("passport response parsing failed: %s", blackBoxRawResponse), ex);
                        } else {
                            throw new PassportException("Empty passport response", ex);
                        }
                    }
                }
            });
        } catch (IOException ex) {
            throw new PassportException("Blackbox httpGet failed", ex);
        } catch (URISyntaxException ex) {
            throw new RuntimeException(ex);
        }
    }


    @Nullable
    public URI buildURI(final String sessionId, final String userIp) throws URISyntaxException {
        final URIBuilder builder = new URIBuilder(serviceRootUrl);
        builder.setPath("/blackbox");
        builder.setParameter("method", SESSION_ID_METHOD);
        builder.setParameter("sessionid", sessionId);
        builder.setParameter("host", blackboxDomain);
        builder.setParameter("userip", userIp);
        builder.setParameter("dbfields", LOGIN_FIELD);
        return builder.build();
    }

    @Nullable
    public URI buildSslURI(final String sessionId, final String sslSessionId, final String userIp) throws URISyntaxException {
        final URIBuilder builder = new URIBuilder(serviceRootUrl);
        builder.setPath("/blackbox");
        builder.setParameter("method", SESSION_ID_METHOD);
        builder.setParameter("sessionid", sessionId);
        if (sslSessionId != null) {
            builder.setParameter("sslsessionid", sslSessionId);
        }
        builder.setParameter("host", blackboxDomain);
        builder.setParameter("userip", userIp);
        builder.setParameter("dbfields", LOGIN_FIELD);
        return builder.build();
    }

    public URI buildOauthURI(final String userIp) throws URISyntaxException {
        final URIBuilder builder = new URIBuilder(serviceRootUrl);
        builder.setPath("/blackbox");
        builder.setParameter("method", OAUTH_METHOD);
        builder.setParameter("userip", userIp);
        builder.setParameter("dbfields", LOGIN_FIELD);
        return builder.build();
    }

    @Nonnull
    public Element parseXml(final String string) throws JDOMException, IOException {
        final SAXBuilder builder = new SAXBuilder();
        final Reader reader = new StringReader(string);
        final Document doc = builder.build(reader);
        return doc.detachRootElement();
    }

    @Nonnull
    SslSession parseSslAnswer(final Element root, final String sessionId, final String userIp) throws PassportException, JDOMException {
        final String status = root.getChildText("status");
        final String error = root.getChildText("error");

        if (status == null) {
            if (error != null && !"OK".equals(error)) {
                throw new PassportException(error);
            } else {
                throw new JDOMException();
            }
        }

        final SslSession result;
        switch (status) {
            case "VALID": {
                final String login = getTextWithAttribute(root, "dbfield", "id", "accounts.login.uid");
                final long uid = Long.parseLong(
                        MoreObjects.firstNonNull(root.getChildText("uid"), root.getChildText("liteuid")));
                final int secure = Integer.parseInt(root.getChild("auth").getChildText("secure"));
                result = new SslSession(uid, login, false, true, secure != 0);
                break;
            }
            case "NEED_RESET": {
                final String login = getTextWithAttribute(root, "dbfield", "id", "accounts.login.uid");
                final long uid = Long.parseLong(
                        MoreObjects.firstNonNull(root.getChildText("uid"), root.getChildText("liteuid")));
                final boolean secure = Boolean.parseBoolean(root.getChildText("secure"));
                result = new SslSession(uid, login, true, true, secure);
                break;
            }
            case "EXPIRED":
            case "DISABLED":
            case "INVALID":
            case "NOAUTH":
                LOG.info("for ip {} and sessionId {} passport return status {} with error: {}", userIp, sessionId, status, error);
                result = new SslSession(null, null, false, false, false);
                break;
            default:
                throw new RuntimeException(String.format(
                        "for ip %s and sessionId %s passport return invalid status %s", userIp, sessionId, status));
        }
        return result;
    }

    @Nonnull
    Session parseAnswer(final Element root, final String sessionId, final String userIp) throws PassportException, JDOMException {
        final String status = root.getChildText("status");
        final String error = root.getChildText("error");

        if (status == null) {
            if (error != null && !"OK".equals(error)) {
                throw new PassportException(error);
            } else {
                throw new JDOMException();
            }
        }

        final Session result;
        switch (status) {
            case "VALID": {
                final String login = getTextWithAttribute(root, "dbfield", "id", "accounts.login.uid");
                final long uid = Long.parseLong(
                        MoreObjects.firstNonNull(root.getChildText("uid"), root.getChildText("liteuid")));
                result = new Session(uid, login);
                break;
            }
            case "NEED_RESET": {
                final String login = getTextWithAttribute(root, "dbfield", "id", "accounts.login.uid");
                final long uid = Long.parseLong(
                        MoreObjects.firstNonNull(root.getChildText("uid"), root.getChildText("liteuid")));
                result = new Session(uid, login, null, true, 0, null, false, true);
                break;
            }
            case "EXPIRED":
            case "DISABLED":
            case "INVALID":
            case "NOAUTH":
                LOG.info("for ip {} and sessionId {} passport return status {} with error: {}", userIp, sessionId, status, error);
                result = new Session(null, null, false);
                break;
            default:
                throw new RuntimeException(String.format(
                        "for ip %s and sessionId %s passport return invalid status %s", userIp, sessionId, status));
        }
        return result;
    }


    @Nonnull
    private String getTextWithAttribute(final Element root, final String tagName, final String attrName, final String attrValue) {
        final List<Element> elements = root.getChildren(tagName);

        for (final Object obj : elements) {
            final Element el = (Element) obj;
            if (attrValue.equals(el.getAttributeValue(attrName))) {
                return el.getText();
            }
        }

        throw new RuntimeException();
    }
}
