package ru.yandex.qe.dispenser.quartz;

import java.util.ArrayList;
import java.util.List;

import javax.inject.Inject;

import org.quartz.Trigger;
import org.springframework.context.ApplicationContext;
import org.springframework.scheduling.quartz.SchedulerFactoryBean;

/**
 * Этот Scheduler умеет игнорировать несуществующие триггеры,
 * например если триггер находится в неактивном профиле и не был
 * проинициализирован Spring'ом.
 */
public class OptionalTriggerSchedulerFactoryBean extends SchedulerFactoryBean {
    @Inject
    private ApplicationContext applicationContext;

    public void setAutowiringJobFactory(AutowiringSpringBeanJobFactory jobFactory) {
        jobFactory.setApplicationContext(applicationContext);
        setJobFactory(jobFactory);
    }

    /**
     * Сюда можно передать полный список потенциальных триггеров. Фактически активные
     * триггеры будут определены активными профилями на этапе запуска приложения.
     * Несуществующие триггеры будут проигнорированы.
     *
     * @param triggerBeanIds список id'шек потенциальных триггеров.
     */
    public void setPotentialTriggers(List<String> triggerBeanIds) {
        final List<Trigger> deFactoExistingTriggers = new ArrayList<>();
        for (String beanId : triggerBeanIds) {
            if (applicationContext.containsBeanDefinition(beanId)) {
                final Trigger trigger = applicationContext.getBean(beanId, Trigger.class);
                deFactoExistingTriggers.add(trigger);
            }
        }
        if (!deFactoExistingTriggers.isEmpty()) {
            setTriggers(deFactoExistingTriggers.toArray(new Trigger[0]));
        }
    }
}
