package ru.yandex.qe.dispenser.quartz.job;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.SchedulerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionStatus;

import ru.yandex.qe.dispenser.domain.tracker.TrackerManager;
import ru.yandex.qe.dispenser.domain.util.FunctionalUtils;
import ru.yandex.startrek.client.model.CommentCreate;

@DisallowConcurrentExecution
public class QuartzTrackerCommentJob implements Job {

    private static final Logger LOG = LoggerFactory.getLogger(QuartzTrackerCommentJob.class);

    @Inject
    private TrackerManager trackerManager;
    @Inject
    private PlatformTransactionManager transactionManager;

    private String issueKey;
    private String comment;
    private String summoned;

    public void setIssueKey(final String issueKey) {
        this.issueKey = issueKey;
    }

    public void setComment(final String comment) {
        this.comment = comment;
    }

    public void setSummoned(final String summoned) {
        this.summoned = summoned;
    }

    @Override
    @SuppressWarnings("ProhibitedExceptionCaught")
    public void execute(final JobExecutionContext context) throws JobExecutionException {
        final TransactionStatus status = transactionManager.getTransaction(null);
        boolean success = false;

        try {
            CommentCreate.Builder commentBuilder = CommentCreate.comment(this.comment);
            if (!StringUtils.isEmpty(summoned)) {
                commentBuilder = commentBuilder.summonees(summoned);
            }
            trackerManager.createComment(issueKey, commentBuilder.build());
            success = true;
        } catch (final Throwable e) {
            transactionManager.rollback(status);
            FunctionalUtils.throwIfUnrecoverable(e);
            LOG.error("QuartzTrackerCommentJob failed", e);
        }

        if (success) {
            try {
                context.getScheduler().unscheduleJob(context.getTrigger().getKey());
            } catch (SchedulerException e) {
                LOG.error("Unchedule QuartzTrackerCommentTrigger failed", e);
                transactionManager.rollback(status);

                final JobExecutionException jobExecutionException = new JobExecutionException();
                jobExecutionException.setUnscheduleFiringTrigger(true);
                throw jobExecutionException;
            }

            transactionManager.commit(status);
        }
    }

}
