package ru.yandex.qe.dispenser.quartz.job;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.SchedulerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionStatus;

import ru.yandex.qe.dispenser.domain.tracker.TrackerManager;
import ru.yandex.qe.dispenser.domain.util.FunctionalUtils;
import ru.yandex.startrek.client.model.IssueUpdate;

@DisallowConcurrentExecution
public class TicketUpdateJob implements Job {

    private static final Logger LOG = LoggerFactory.getLogger(TicketUpdateJob.class);
    public static final String TRIGGER_GROUP = "TrackerUpdateGroupTrigger";

    @Inject
    private TrackerManager trackerManager;
    @Inject
    private PlatformTransactionManager transactionManager;
    @Inject
    private TicketUpdateSerializationUtils objectMapperHolder;

    private String issueKey;
    private String transition;
    private String body;

    public void setIssueKey(final String issueKey) {
        this.issueKey = issueKey;
    }

    public void setTransition(final String transition) {
        this.transition = transition;
    }

    public void setBody(final String body) {
        this.body = body;
    }

    @Override
    public void execute(final JobExecutionContext context) throws JobExecutionException {
        final TransactionStatus status = transactionManager.getTransaction(null);
        boolean success = false;

        try {
            final IssueUpdate issueUpdate = objectMapperHolder.parseIssueUpdate(body);
            if (StringUtils.isEmpty(transition)) {
                trackerManager.updateIssue(issueKey, issueUpdate);
            } else {
                trackerManager.executeTransition(issueKey, transition, issueUpdate);
            }
            success = true;
        } catch (final Throwable e) {
            transactionManager.rollback(status);
            FunctionalUtils.throwIfUnrecoverable(e);
            LOG.error("QuartzTicketUpdateJob failed", e);
        }

        if (success) {
            try {
                context.getScheduler().unscheduleJob(context.getTrigger().getKey());
            } catch (SchedulerException e) {
                LOG.error("Unschedule QuartzTicketUpdateTrigger failed", e);
                transactionManager.rollback(status);

                final JobExecutionException jobExecutionException = new JobExecutionException();
                jobExecutionException.setUnscheduleFiringTrigger(true);
                throw jobExecutionException;
            }

            transactionManager.commit(status);
        }
    }
}
