package ru.yandex.qe.dispenser.quartz.job;

import java.io.IOException;
import java.util.Iterator;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.exc.InvalidFormatException;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.node.NullNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.joda.JodaModule;

import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.kotlin.KotlinModule;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.startrek.client.model.CollectionUpdate;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.IssueUpdate;
import ru.yandex.startrek.client.model.LinkCreate;
import ru.yandex.startrek.client.model.Relationship;
import ru.yandex.startrek.client.model.ScalarUpdate;
import ru.yandex.startrek.client.model.Update;
import ru.yandex.startrek.client.utils.LocalDateModule;
import ru.yandex.startrek.client.utils.StartrekClientModule;

public class TicketUpdateSerializationUtils {

    private final ObjectMapper objectMapper;

    public TicketUpdateSerializationUtils() {
        final SimpleModule simpleModule = new SimpleModule();
        simpleModule.addDeserializer(IssueUpdate.class, new IssueUpdateDeserializer());
        simpleModule.addDeserializer(Update.class, new UpdateDeserializer());
        simpleModule.addDeserializer(CommentCreate.class, new CommentCreateDeserializer());
        simpleModule.addDeserializer(LinkCreate.class, new LinkCreateDeserializer());
        objectMapper = new ObjectMapper()
                .registerModule(new JodaModule())
                .registerModule(new LocalDateModule())
                .registerModule(new StartrekClientModule(Cf.map()))
                .registerModule(simpleModule)
                .registerModule(new KotlinModule.Builder().build())
                .registerModule(new Jdk8Module())
                .registerModule(new JavaTimeModule())

                .enable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS)
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    public IssueUpdate parseIssueUpdate(final String body) throws IOException {
        return objectMapper.readValue(body, IssueUpdate.class);
    }

    public String writeValueAsString(final IssueUpdate issueUpdate) throws JsonProcessingException {
        return objectMapper.writeValueAsString(issueUpdate);
    }

    public class UpdateDeserializer extends StdDeserializer<Update<?>> {

        protected UpdateDeserializer() {
            super(Update.class);
        }

        @Override
        public Update<?> deserialize(final JsonParser p, final DeserializationContext ctxt) throws IOException {
            final TreeNode treeNode = p.readValueAsTree();
            final TreeNode set = treeNode.get("set");
            final TreeNode unset = treeNode.get("unset");
            final TreeNode add = treeNode.get("add");
            final TreeNode remove = treeNode.get("remove");

            if (treeNode.isValueNode()) {
                if (treeNode instanceof NullNode) {
                    return ScalarUpdate.unset();
                } else if (treeNode instanceof TextNode) {
                    final TextNode text = (TextNode) treeNode;
                    return ScalarUpdate.set(text.asText());
                }
            }

            if (add != null) {
                if (remove != null) {
                    return CollectionUpdate.cons(
                            Cf.list(objectMapper.convertValue(add, Object[].class)),
                            Cf.list(objectMapper.convertValue(remove, Object[].class)));
                }
                return CollectionUpdate.add(objectMapper.convertValue(add, Object[].class));
            }
            if (remove != null) {
                return CollectionUpdate.remove(objectMapper.convertValue(remove, Object[].class));
            }
            if (unset != null) {
                return ScalarUpdate.unset();
            }

            if (set != null) {
                return ScalarUpdate.set(objectMapper.convertValue(set, Object.class));
            }

            throw new InvalidFormatException(p, "Unsupported update type ", treeNode, Update.class);
        }
    }

    public class LinkCreateDeserializer extends StdDeserializer<LinkCreate> {
        protected LinkCreateDeserializer() {
            super(LinkCreate.class);
        }

        @Override
        public LinkCreate deserialize(final JsonParser p, final DeserializationContext ctxt) throws IOException {
            final TreeNode treeNode = p.readValueAsTree();

            final TreeNode issue = treeNode.get("issue");
            if (issue != null) {
                final String relationshipValue = ((TextNode) treeNode.get("relationship")).textValue();
                Relationship relationship = null;
                for (final Relationship rel : Relationship.values()) {
                    if (rel.toString().equals(relationshipValue)) {
                        relationship = rel;
                    }
                }

                return LinkCreate.local()
                        .issue(((TextNode) issue).asText())
                        .relationship(relationship)
                        .build();
            }

            throw new InvalidFormatException(p, "Unsupported link type ", treeNode, LinkCreate.class);
        }
    }

    public class CommentCreateDeserializer extends StdDeserializer<CommentCreate> {
        protected CommentCreateDeserializer() {
            super(CommentCreate.class);
        }

        @Override
        public CommentCreate deserialize(final JsonParser p, final DeserializationContext ctxt) throws IOException {
            final CommentCreate.Builder builder = CommentCreate.builder();
            final TreeNode treeNode = p.readValueAsTree();

            final Iterator<String> fieldNames = treeNode.fieldNames();
            while (fieldNames.hasNext()) {
                final String fieldName = fieldNames.next();
                final TreeNode fieldValue = treeNode.get(fieldName);
                if (fieldValue == null) {
                    continue;
                }
                switch (fieldName) {
                    case "text":
                        builder.comment(((TextNode) fieldValue).asText());
                        break;
                    case "summonees":
                        builder.summonees(objectMapper.convertValue(fieldValue, String[].class));
                        break;
                    default:
                        throw new InvalidFormatException(p, "Unsupported field " + fieldName, treeNode, CommentCreate.class);
                }
            }


            return builder.build();
        }
    }

    public class IssueUpdateDeserializer extends StdDeserializer<IssueUpdate> {

        public IssueUpdateDeserializer() {
            super(IssueUpdate.class);
        }

        @Override
        public IssueUpdate deserialize(final JsonParser p, final DeserializationContext ctxt) throws IOException {

            final IssueUpdate.Builder builder = IssueUpdate.builder();
            final TreeNode treeNode = p.readValueAsTree();

            final Iterator<String> fieldNames = treeNode.fieldNames();
            while (fieldNames.hasNext()) {
                final String fieldName = fieldNames.next();
                final TreeNode fieldValue = treeNode.get(fieldName);
                switch (fieldName) {
                    case "links":
                        final LinkCreate[] links = objectMapper.convertValue(fieldValue, LinkCreate[].class);
                        for (final LinkCreate link : links) {
                            builder.link(link);
                        }
                        break;
                    case "comment":
                        builder.comment(objectMapper.convertValue(fieldValue, CommentCreate.class));
                        break;
                    case "attachments":
                    case "worklog":
                        throw new InvalidFormatException(p, "Unsupported field " + fieldName, treeNode, IssueUpdate.class);
                    default:
                        builder.update(fieldName, objectMapper.convertValue(fieldValue, Update.class));
                        break;
                }
            }
            return builder.build();
        }
    }
}
